/* FileChooserUI.java --
   Copyright (C) 2002, 2003 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

package javax.swing.plaf;

import java.io.File;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileView;

/**
 * An abstract base class for delegates that implement the pluggable
 * look and feel for a <code>JFileChooser</code>.
 *
 * @see javax.swing.JFileChooser
 *
 * @author Andrew Selkirk (aselkirk@sympatico.ca)
 * @author Sascha Brawer (brawer@dandelis.ch)
 */
public abstract class FileChooserUI
  extends ComponentUI
{
  /**
   * Constructs a new <code>FileChooserUI</code>.
   */
  public FileChooserUI()
  {
  }

  
  /**
   * Returns a <code>FileFilter</code> that accepts every file.  While
   * the filtering itself is not specific to any look and feel, the
   * text returned by <code>FileFilter.getDescription()</code> need
   * not be the same across all look and feels.
   *
   * @param chooser the <code>JFileChooser</code> for which
   *        a <code>FileFilter</code> is requested.
   *
   * @see javax.swing.JFileChooser#getAcceptAllFileFilter
   * @see javax.swing.filechooser.FileFilter#getDescription
   */
  public abstract FileFilter getAcceptAllFileFilter(JFileChooser chooser);


  /**
   * Returns a view to a file, which is able to retrieve its name,
   * icon, and other properties that are relevant for presenting
   * the file to the user.
   *
   * @param chooser the <code>JFileChooser</code> for which
   *        a <code>FileFilter</code> is requested.
   */
  public abstract FileView getFileView(JFileChooser chooser);


  /**
   * Determines which text is appropriate for the approve button
   * according to the design guidelines of the implemented
   * look and feel.
   *
   * @param chooser the <code>JFileChooser</code> whose
   *        button text is requested.
   *
   * @see javax.swing.JFileChoose#getApproveButtonText
   */
  public abstract String getApproveButtonText(JFileChooser chooser);


  /**
   * Determines which text is appropriate for the title bar of a
   * <code>JFileChooser</code> according to the design guidelines of
   * the implemented look and feel.
   *
   * @param chooser the <code>JFileChooser</code> whose
   *        dialog title is requested.
   *
   * @see javax.swing.JFileChoose#getDialogtitle
   */
  public abstract String getDialogTitle(JFileChooser chooser);


  /**
   * Refreshes the currently displayed directory.
   *
   * @param chooser the <code>JFileChooser</code> whose
   *        dialog title needs re-scanning.
   */
  public abstract void rescanCurrentDirectory(JFileChooser chooser);


  /**
   * Ensures that a specified file is visible in the
   * <code>JFileChooser</code>
   *
   * @param chooser the <code>JFileChooser</code> that
   *        should display the file <code>file</code>.
   *
   * @param file the file that needs to be made visible.
   */
  public abstract void ensureFileIsVisible(JFileChooser chooser, File file);
}
