/* Creation of subprocesses, communicating via pipes.
   Copyright (C) 2001-2003 Free Software Foundation, Inc.
   Written by Bruno Haible <haible@clisp.cons.org>, 2001.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifndef _PIPE_H
#define _PIPE_H

/* Get pid_t.  */
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>

#include <stdbool.h>

/* All these functions create a subprocess and don't wait for its termination.
   They return the process id of the subprocess.  They also return in fd[]
   one or two file descriptors for communication with the subprocess.
   If the subprocess creation fails: if exit_on_error is true, the main
   process exits with an error message; otherwise, -1 is returned and fd[]
   remain uninitialized.

   After finishing communication, the caller should call wait_subprocess()
   to get rid of the subprocess in the process table.

   If exit_on_error is false, a child process id of -1 should be treated the
   same way as a subprocess which accepts no input, produces no output and
   terminates with exit code 127.  Why?  Some errors during posix_spawnp()
   cause the function posix_spawnp() to return an error code; some other
   errors cause the subprocess to exit with return code 127.  It is
   implementation dependent which error is reported which way.  The caller
   must treat both cases as equivalent.  */

/* Open a pipe for output to a child process.
 * The child's stdout goes to a file.
 *
 *           write       system                read
 *    parent  ->   fd[0]   ->   STDIN_FILENO    ->   child
 *
 */
extern pid_t create_pipe_out (const char *progname,
			      const char *prog_path, char **prog_argv,
			      const char *prog_stdout, bool null_stderr,
			      bool exit_on_error,
			      int fd[1]);

/* Open a pipe for input from a child process.
 * The child's stdin comes from a file.
 *
 *           read        system                write
 *    parent  <-   fd[0]   <-   STDOUT_FILENO   <-   child
 *
 */
extern pid_t create_pipe_in (const char *progname,
			     const char *prog_path, char **prog_argv,
			     const char *prog_stdin, bool null_stderr,
			     bool exit_on_error,
			     int fd[1]);

/* Open a bidirectional pipe.
 *
 *           write       system                read
 *    parent  ->   fd[1]   ->   STDIN_FILENO    ->   child
 *    parent  <-   fd[0]   <-   STDOUT_FILENO   <-   child
 *           read        system                write
 *
 */
extern pid_t create_pipe_bidi (const char *progname,
			       const char *prog_path, char **prog_argv,
			       bool null_stderr,
			       bool exit_on_error,
			       int fd[2]);

/* The name of the "always silent" device.  */
#if defined _MSC_VER || defined __MINGW32__
/* Native Woe32 API.  */
# define DEV_NULL "NUL"
#else
/* Unix API.  */
# define DEV_NULL "/dev/null"
#endif

#endif /* _PIPE_H */
