/*
    A system to permit user selection of a block and rotation axis
    of a magic cube.
    Copyright (C) 1998,  2003  John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License,  or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
static const char RCSID[]="$Id: select.c,v 1.2 2008/12/04 23:43:47 jmd Exp $";


/* This library provides  a means of picking a block using the mouse cursor.

Two mutually co-operative mechanisms are used in this library.  There is a
timer callback,  which occurs at regular intervals.  There is also the mouse
motion callback,  which occurs whenever the mouse cursor is moving.  If two
consecutive timer callbacks occur,  without and intervening mouse motion callback,
then the cursor is assumed to be stationary.

If a stationary mouse is detected,  the program goes on to determine which block
in the cube (if any) the cursor is located upon.
*/

#include "select.h"
#include <float.h>
#include <stdio.h>
#include "ui.h"

#include <gtk/gtk.h>


static int idle_threshold;
static gboolean motion= FALSE;
static gboolean Stop_Detected = FALSE;

extern int mouse_x,  mouse_y;
extern double granularity;
extern void (*action)(void);



static gboolean detect_motion (GtkWidget *w,
			       GdkEventMotion *event,  gpointer user_data);

static gboolean UnsetMotion (gpointer data);


static gboolean enableDisableSelection (GtkWidget *w,
		       GdkEventCrossing *event,  gpointer data);

extern GtkWidget *glxarea;
extern GtkWidget *toplevel;


static gboolean timerActive;

static guint timer;

/* is the selection mechanism necessary ? */
static gboolean needSelection;





/* Initialise the selection mechanism.  Holdoff is the time for which
the mouse must stay still,  for anything to happen. Precision is the
minimum distance it must have moved. Do_this is a pointer to a function
to be called when a new block is selected. */
void
initSelection (int holdoff,  double precision,  void (*do_this)(void) )
{


  needSelection = FALSE;
  idle_threshold = holdoff;
  granularity = precision ;

  g_signal_connect (GTK_OBJECT (glxarea), "motion_notify_event",
		      GTK_SIGNAL_FUNC (detect_motion), 0);



  action = do_this;


  timer = g_timeout_add (idle_threshold,  UnsetMotion, 0);

  timerActive=TRUE;

  /* Add a handler to for all those occasions when we don't need the
     selection mechanism going */

  g_signal_connect (GTK_OBJECT (glxarea), "enter-notify-event",
		      GTK_SIGNAL_FUNC (enableDisableSelection), 0);

  g_signal_connect (GTK_OBJECT (glxarea), "leave-notify-event",
		      GTK_SIGNAL_FUNC (enableDisableSelection), 0);
		



  g_signal_connect (GTK_OBJECT (glxarea), "visibility-notify-event",
		      GTK_SIGNAL_FUNC (enableDisableSelection), 0);




  g_signal_connect (GTK_OBJECT (glxarea), "unmap-event",
		      GTK_SIGNAL_FUNC (enableDisableSelection), 0);





}


void
disableSelection (void)
{

    g_source_remove (timer);
    timerActive=FALSE;
}

void
enableSelection (void)
{
    if ( !timerActive) {
      timer = g_timeout_add (idle_threshold,  UnsetMotion, 0);
      timerActive=TRUE;
    }

    needSelection = TRUE ;
}


/* When the window is not mapped,  kill the selection mechanism.  It wastes
processor time */
static gboolean
enableDisableSelection (GtkWidget *w,
		       GdkEventCrossing *event,  gpointer data)
{
  /* This is a kludge to work around a rather horrible bug;  for some
     reason,  some  platforms emit a EnterNotify and LeaveNotify (in
     that order) when animations occur.  This workaround makes sure
     that the window is not `entered twice' */
  static int entered =0;


  switch (event->type) {

  case GDK_ENTER_NOTIFY:
    entered++;
    needSelection = FALSE ;
    if ( !timerActive) {
      timer = g_timeout_add (idle_threshold,  UnsetMotion, 0);
      timerActive=TRUE;
    }

    break;
  case GDK_LEAVE_NOTIFY:
    updateSelection ();
    entered--;
    if ( entered > 0 ) break ;

    needSelection = TRUE ;
    g_source_remove (timer);
    timerActive=FALSE;

    break;

  default:
    break;
  }

  return FALSE;
}




/* This callback occurs whenever the mouse is moving */
static gboolean
detect_motion (GtkWidget *w,  GdkEventMotion *event,  gpointer user_data)
{

  if ( event->type != GDK_MOTION_NOTIFY)
    return FALSE;

  mouse_x = event->x;
  mouse_y = event->y;

  motion = TRUE;
  Stop_Detected = FALSE;

  return FALSE;
}





/* This callback occurs at regular intervals.   The period is determined by
idle_threshold.  It checks to see if the mouse has moved,  since the last
call of this function.
Post-condition:  motion is FALSE.
*/
gboolean
UnsetMotion (gpointer data)
{

  if ( motion == FALSE ) { /* if not moved since last time */

    if ( ! Stop_Detected ) {
      /* in here,  things happen upon the mouse stopping */
      Stop_Detected = TRUE;
      updateSelection ();
    }
  }

  motion = FALSE ;
	
  return TRUE;

}  /* end UnsetMotion */



int
get_widget_height (GtkWidget *w)
{

  return w->allocation.height;

}


