/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <make_GrMC4020Source.h>
#include <GrFFTSink.h>
#include <GrFIRfilterFFF.h>
#include <VrAudioSink.h>
#include <VrConnect.h>
#include <VrMultiTask.h>
#include <VrGUI.h>
#include <VrInterpolatingSigProc.h>
#include <gr_firdes.h>
#include <gr_fir_builder.h>
#include <VrNullSink.h>
#include <VrFileSource.h>
#include <VrFileSink.h>
#include <GrReal.h>
#include <getopt.h>
#include <VrSigSource.h>
#include <atsc_root_raised_cosine_bandpass.h>
#include <atsc_types.h>
#include <atsc_consts.h>
#include <GrMixer.h>
#include <GrAtscSymbolMapper.h>
#include <atsc_vsbtx_lp.h>

double output_rate = 2 * ATSC_SYMBOL_RATE;
static const double IF_freq = 5.75e6;		// output freq of tuner module

// freq of hdtv suppressed carrier pilot tone.  
//
// This assumes that the tuner is actually tuned to the middle of the 6
// MHz channel.  The pilot is 0.31 MHz up from the bottom edge of the channel

static const double pilot_freq = IF_freq - 3e6 + 0.31e6;


//static const double MIXER_GAIN = 128.0;	// linear value
static const double MIXER_GAIN = 1.0;		// linear value


static void 
usage (const char *name)
{
  cerr << "usage: " << name << " [-g] [-r] [-F] [-o <filename>] [-M mu] -f <filename>\n";
  cerr << "  -g : no gui\n";
  cerr << "  -r : continuously repeat the file contents (loop)\n";
  cerr << "  -F : sampling rate is 20 MHz not 2*atsc_symbol_rate\n";
  cerr << "  -f <filename> : take input from filename\n";
  cerr << "  -o <filename> : put output in filename, else VrNullSink\n";
  cerr << "  -M <mu>\n";

  exit (1);
}

int main (int argc, char **argv)
{
  try {
    
    bool use_gui_p = true;
    bool repeat_p = false;		// continuous loop through file
    char *input_filename = 0;
    char *output_filename = 0;
    float mu = 0.0;

    int c;
    while ((c = getopt (argc, argv, "grNFDf:o:M:")) != EOF){
      switch (c){
      case 'g':	use_gui_p = false;    		break;
      case 'r': repeat_p = true;		break;
      case 'F': output_rate = 20e6;		break;
      case 'f': input_filename = optarg;	break;
      case 'o': output_filename = optarg;	break;
      case 'M': mu = strtod (optarg, 0);	break;
      default:
	usage (argv[0]);
      }
    }

    if (input_filename == 0)
      usage (argv[0]);
    
    if (optind != argc)
      usage (argv[0]);

    VrGUI *guimain = 0;
    VrGUILayout *horiz = 0;
    VrGUILayout *vert = 0;
    VrGUILayout *vert2 = 0;

    if (use_gui_p){
      guimain = new VrGUI(argc, argv);
      horiz = guimain->top->horizontal();
      vert = horiz->vertical();
      vert2 = horiz->horizontal()->vertical();
    }

    cerr << "Output Sampling Rate: " << output_rate << endl;
    cerr << "Pilot Freq: " << pilot_freq << endl;

    VrSource<atsc_data_segment>		*source = 0;
    GrAtscSymbolMapper<float>		*mapper = 0;
    VrInterpolatingSigProc<float,float>	*interp = 0;
    GrFIRfilterFFF			*lp_filter = 0;
    GrMixer<float,float>		*mixer = 0;
    GrFIRfilterFFF			*matched_filter = 0;
    VrSink<float>			*final_sink = 0;


    VrSink<float>	*fft_sink0 = 0;
    VrSink<float>	*fft_sink1 = 0;
    VrSink<float>	*fft_sink2 = 0;
    VrSink<float>	*fft_sink3 = 0;

    VrSink<float>	*mapper_log = 0;
    VrSink<float>	*interp_log = 0;
    VrSink<float>	*lowpass_log = 0;
    VrSink<float>	*mixer_log = 0;
    

    bool logging_p = output_filename != 0;
    

    // ================================================================
    //   Get segment data from external file
    // ================================================================

    // --> atsc_data_segment
    source = new VrFileSource<atsc_data_segment>(ATSC_DATA_SEGMENT_RATE,
						 input_filename,
						 repeat_p);

    // ================================================================
    //   Pass it through the symbol mapper
    //   (includes symbol mapping & pilot addition)
    // ================================================================

    mapper = new GrAtscSymbolMapper<float>();

    if (logging_p)
      mapper_log = new VrFileSink<float>("mapper.out");

    
    // ================================================================
    //   2x interpolator (has gain of 0.5)
    // ================================================================

    interp = new VrInterpolatingSigProc<float,float>(1, 2);
    
    if (use_gui_p){
      // sink0 is the output of the interpolator
      fft_sink0 = new GrFFTSink<float>(vert, 0, 50, 512);
    }

    if (logging_p)
      interp_log = new VrFileSink<float>("interp.out");
    


    // ================================================================
    //   Low pass to remove high frequency stuff added by interpolation
    // ================================================================

#if 0 // these windowed designs don't have sufficient attenuation
    
    double lower_edge = 0.25;	// normalized freq (Fs == 1)
    double upper_edge = 0.36;	// normalized freq
    double transition_width = upper_edge - lower_edge;

    vector<float> lp_coeffs =
      gr_firdes::low_pass (2.0,
			   1,	// normalized freq
			   (lower_edge + upper_edge) * 0.5,
			   transition_width,
			   gr_firdes::WIN_BLACKMAN);
    
    cerr << "Number of lp_coeffs: " << lp_coeffs.size () << endl;

    // float --> float
    lp_filter = new GrFIRfilterFFF(1, lp_coeffs);
#else

    lp_filter = new GrFIRfilterFFF(1, new atsc_vsbtx_lp ());

#endif

    if (use_gui_p){
      // sink1 is the output of the lp
      fft_sink1 = new GrFFTSink<float>(vert, 0, 50, 512);
    }

    if (logging_p)
      lowpass_log = new VrFileSink<float>("lowpass.out");


    // ================================================================
    //   Mix with carrier
    // ================================================================

    mixer = new GrMixer<float,float>(pilot_freq, MIXER_GAIN);

    if (use_gui_p){
      // sink2 is the output of the mixer
      fft_sink2 = new GrFFTSink<float>(vert2, 0, 80, 512);
    }

    if (logging_p){
      mixer_log = new VrFileSink<float> ("mixer.out");
    }

    // ================================================================
    //  apply the band pass matched filter (root raised cosine)
    // ================================================================

    matched_filter =
      new GrFIRfilterFFF (1, new atsc_root_raised_cosine_bandpass (IF_freq));

    if (use_gui_p){
      // sink3 is the band passed data
      fft_sink3 = new GrFFTSink<float>(vert2, 0, 80, 512);
    }
    
    // ================================================================
    // final sink is either a file sink or a null sink
    // ================================================================

    if (output_filename)
      final_sink = new VrFileSink<float>(output_filename);

    else
      final_sink = new VrNullSink<float>();


    // connect the modules together

    VrMultiTask *m = new VrMultiTask ();
    m->add (final_sink);

    NWO_CONNECT (source, mapper);
    NWO_CONNECT (mapper, interp);
    NWO_CONNECT (interp, lp_filter);
    NWO_CONNECT (lp_filter, mixer);
    NWO_CONNECT (mixer, matched_filter);
    NWO_CONNECT (matched_filter, final_sink);
    
    if (logging_p){
      NWO_CONNECT (mapper, mapper_log);
      NWO_CONNECT (interp, interp_log);
      NWO_CONNECT (lp_filter, lowpass_log);
      NWO_CONNECT (mixer, mixer_log);
      m->add (mapper_log);
      m->add (interp_log);
      m->add (lowpass_log);
      m->add (mixer_log);
    }

    if (use_gui_p){
      NWO_CONNECT (interp, fft_sink0);
      NWO_CONNECT (lp_filter, fft_sink1);
      NWO_CONNECT (mixer, fft_sink2);
      NWO_CONNECT (matched_filter, fft_sink3);

      m->add (fft_sink0);
      m->add (fft_sink1);
      m->add (fft_sink2);
      m->add (fft_sink3);
    }

    m->start ();
    if (use_gui_p)
      guimain->start ();

    while (1){
      if (use_gui_p)
	guimain->processEvents(10 /*ms*/);
      m->process();
    }	

  } // end try

  catch (std::exception &e){
    cerr << "std library exception: " << e.what () << endl;
    exit (1);
  }
  catch (...) {
    cerr << "unknown exception thrown" << endl;
    exit (1);
  }
}
