/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 *
 * This file is part of GNU Radio
 *
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gr_io_signature.h>
#include <gr_clock_recovery_mm_cc.h>
#include <gri_mmse_fir_interpolator_cc.h>
#include <stdexcept>

// Public constructor

gr_clock_recovery_mm_cc_sptr 
gr_make_clock_recovery_mm_cc(float omega, float gain_omega, float mu, float gain_mu)
{
  return gr_clock_recovery_mm_cc_sptr (new gr_clock_recovery_mm_cc (omega, 
								    gain_omega, 
								    mu,
								    gain_mu));
}

gr_clock_recovery_mm_cc::gr_clock_recovery_mm_cc (float omega, float gain_omega, float mu, float gain_mu)
  : gr_block ("clock_recovery_mm_cc",
	      gr_make_io_signature (1, 1, sizeof (gr_complex)),
	      gr_make_io_signature (1, 1, sizeof (gr_complex))),
    d_mu (mu), d_omega(omega), d_gain_omega(gain_omega), 
    d_gain_mu(gain_mu), d_last_sample(0), d_interp(new gri_mmse_fir_interpolator_cc())
{
  if (omega <= 0.0)
    throw std::out_of_range ("clock rate must be > 0");
  if (gain_mu <  0  || gain_omega < 0)
    throw std::out_of_range ("Gains must be non-negative");

  set_relative_rate (1.0 / omega);
}

gr_clock_recovery_mm_cc::~gr_clock_recovery_mm_cc ()
{
    delete d_interp;
}

void
gr_clock_recovery_mm_cc::forecast(int noutput_items, gr_vector_int &ninput_items_required)
{
  unsigned ninputs = ninput_items_required.size();
  for (unsigned i=0; i < ninputs; i++)
    ninput_items_required[i] =
      (int) ceil((noutput_items * d_omega) + d_interp->ntaps());
}

gr_complex
gr_clock_recovery_mm_cc::slicer_0deg (gr_complex sample)
{
  gr_complex out;
  if( fabs(sample.real()) > fabs(sample.imag()) ) {
    if(sample.real() > 0)
      return gr_complex(1.0,0.0);
    else
      return gr_complex(-1.0,0.0);
  }
  else {
    if(sample.imag() > 0)
      return gr_complex(0.0, 1.0);
    else
      return gr_complex(0.0, -1.0);
  }
}

gr_complex
gr_clock_recovery_mm_cc::slicer_45deg (gr_complex sample)
{
  float real,imag;
  if(sample.real() > 0)
    real=1;
  else
    real=-1;
  if(sample.imag() > 0)
    imag = 1;
  else
    imag = -1;
  return gr_complex(real,imag);
}

int
gr_clock_recovery_mm_cc::general_work (int noutput_items,
				       gr_vector_int &ninput_items,
				       gr_vector_const_void_star &input_items,
				       gr_vector_void_star &output_items)
{
  const gr_complex *in = (const gr_complex *) input_items[0];
  gr_complex *out = (gr_complex *) output_items[0];
  
  int 	ii = 0;				// input index
  int  	oo = 0;				// output index
  float mm_val;
  //gr_complex sliced_data, mm_val;
  
  while((d_mu >= 1.0)&&(ii < ninput_items[0] - d_interp->ntaps() + 1)) {
    ii++;
    d_mu -= 1.0;
  }
  while ((oo < noutput_items) && (ii < ninput_items[0] - d_interp->ntaps() + 1)) {
    out[oo] = d_interp->interpolate (&in[ii], d_mu);
    out[oo] = out[oo] * gr_complex(1,1);
    mm_val = real(d_last_sample * slicer_0deg(out[oo]) -out[oo] * slicer_0deg(d_last_sample) );

    d_last_sample = out[oo];
    
    printf("%f\t%f\t%f\t%f\t%f\t%f\t%f\t%d\t%d\n",d_last_sample.real(),d_last_sample.imag(),slicer_0deg(d_last_sample).real(),slicer_0deg(d_last_sample).imag(),mm_val,d_omega,d_mu,ii,oo);

    d_omega = d_omega + d_gain_omega * mm_val;
    d_mu = d_mu + d_omega + d_gain_mu * mm_val;
    
    while((d_mu >= 1.0)&&(ii < ninput_items[0] - d_interp->ntaps() + 1)) {
      ii++;
      d_mu -= 1.0;
    }
    oo++;
  }
  consume_each (ii);
  return oo;
}

//    ii += (int) floor(d_mu);
//  d_mu = d_mu - floor(d_mu);
