#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

_default_bitrate = 500e3

def _gen_tx_info(converter_rate):
    results = []
    for spb in (2,3,4,5):
        for interp in range(16, 512 + 1, 4):
            bitrate = converter_rate / interp / spb
            results.append((bitrate, spb, interp))
    results.sort()
    return results

def _gen_rx_info(converter_rate):
    results = []
    for spb in (2,3,4,5):
        for decim in range(8, 256 + 1, 2):
            bitrate = converter_rate / decim / spb
            results.append((bitrate, spb, decim))
    results.sort()
    return results
    
def _filter_info(info, samples_per_baud, xrate):
    if samples_per_baud is not None:
        info = [x for x in info if x[1] == samples_per_baud]
    if xrate is not None:
        info = [x for x in info if x[2] == xrate]
    return info

def _pick_best(target_bitrate, info):
    if len(info) == 0:
        raise RuntimeError, "info is zero length!"

    if target_bitrate is None:     # return the fastest one
        return info[-1]
    
    # Find the closest matching bitrate.
    # In the event of a tie, the one with the lowest samples_per_baud wins.
    # (We already sorted them, so the first one is the one we take)

    best = info[0]
    best_delta = abs(target_bitrate - best[0])
    for x in info[1:]:
        delta = abs(target_bitrate - x[0])
        if delta < best_delta:
            best_delta = delta
            best = x
    return best

def _pick_bitrate(bitrate, samples_per_baud, xrate, converter_rate, gen_info):
    if samples_per_baud is not None and xrate is not None:  # completely determined
        return (float(converter_rate) / xrate / samples_per_baud, samples_per_baud, xrate)

    if bitrate is None and samples_per_baud is None and xrate is None:
        bitrate = _default_bitrate

    # now we have a target bitrate and possibly an xrate or
    # samples_per_baud constraint, but not both of them.

    return _pick_best(bitrate,
                      _filter_info(gen_info(converter_rate), samples_per_baud, xrate))
    
# ---------------------------------------------------------------------------------------

def pick_tx_bitrate(bitrate, samples_per_baud, interp_rate, converter_rate=128e6):
    """
    Given the 4 input parameters, return at configuration that matches

    @param bitrate: desired bitrate or None
    @type bitrate: number or None
    @param samples_per_baud: samples/baud (aka samples/symbol)
    @type samples_per_baud: number or None
    @param interp_rate: USRP interpolation factor
    @type interp_rate: integer or None

    @returns tuple (bitrate, samples_per_baud, interp_rate)
    """
    return _pick_bitrate(bitrate, samples_per_baud, interp_rate, converter_rate, _gen_tx_info)


def pick_rx_bitrate(bitrate, samples_per_baud, decim_rate, converter_rate=64e6):
    """
    Given the 4 input parameters, return at configuration that matches

    @param bitrate: desired bitrate or None
    @type bitrate: number or None
    @param samples_per_baud: samples/baud (aka samples/symbol)
    @type samples_per_baud: number or None
    @param decim_rate: USRP decimation factor
    @type decim_rate: integer or None

    @returns tuple (bitrate, samples_per_baud, decim_rate)
    """
    return _pick_bitrate(bitrate, samples_per_baud, decim_rate, converter_rate, _gen_rx_info)
