/*
 * GNUsound - a sound editor for GNOME.
 * Copyright (C) 2006  Pascal Haakmat <pascal@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * A copy of the GNU General Public License can be found in the file
 * LICENSE in the top directory of the source distribution. If not,
 * write to the Free Software * Foundation, Inc., 675 Mass Ave,
 * Cambridge, MA 02139, USA.
 *
 */

#include <assert.h>
#include <config.h>
#include <stdlib.h>
#include <errno.h>

#include "timer.h"

static struct timer tmr_class = { 0, };

int timer_init() {
    tmr_class.nsrc = 0;
    timer_register_source("Internal");
}

int 
timer_register_source(const char *name) {
    int i;
    char *s;

    if(tmr_class.nsrc == MAX_SRC)
        return -ENOMEM;

    for(i = 0; i < tmr_class.nsrc; i++) 
        if(!strcmp(tmr_class.src[i].name, name))
            return -EEXIST;
    
    s = strdup(name); 
    if(!s)
        return -ENOMEM;
    
    tmr_class.src[tmr_class.nsrc].name = s;
    tmr_class.src[tmr_class.nsrc].id = tmr_class.nsrc;
    tmr_class.src[tmr_class.nsrc].flags = TIMER_FLAGS_DISABLED;
    tmr_class.src[tmr_class.nsrc].time = -1;
    tmr_class.src[tmr_class.nsrc].resolution = 1;
    
    tmr_class.nsrc++;

    return tmr_class.nsrc-1;
}

static int
timer_check_id(int id,
               int reject_flags) {

    if(id < 0 || id >= tmr_class.nsrc)
        return -EINVAL;

    if(tmr_class.src[id].flags & reject_flags)
        return -EPERM;

    return 0;
}

int 
timer_unregister_source(int id) {
    int r;

    if((r = timer_check_id(id, 0)))
        return r;

    tmr_class.src[id].flags |= TIMER_FLAGS_UNREGISTERED;

    return 0;
}

const struct time_source *
timer_get_next_source(int id) {
    int i;

    for(i = id + 1; i < tmr_class.nsrc; i++) {
        if(tmr_class.src[id].flags & TIMER_FLAGS_UNREGISTERED)
            continue;
        return &tmr_class.src[i];
    }

    return NULL;
}

int
timer_enable(int id) {
    int r;

    if((r = timer_check_id(id, TIMER_FLAGS_UNREGISTERED)))
        return r;

    tmr_class.src[id].flags &= ~TIMER_FLAGS_DISABLED;

    return 0;
}

int
timer_disable(int id) {
    int r;

    if((r = timer_check_id(id, TIMER_FLAGS_UNREGISTERED)))
        return r;

    tmr_class.src[id].flags |= TIMER_FLAGS_DISABLED;
    
    return 0;
}

int 
timer_set_resolution(int id,
                     long long nsec) {
    int r;

    if((r = timer_check_id(id, TIMER_FLAGS_UNREGISTERED)))
        return r;

    tmr_class.src[id].resolution = nsec;
}

long long 
timer_get_resolution(int id) {
    int r;

    if((r = timer_check_id(id, TIMER_FLAGS_UNREGISTERED)))
        return r;

    return tmr_class.src[id].resolution;
}

struct timer *
timer_obtain() {
    struct timer *tmr = malloc(sizeof(*tmr));

    if(!tmr)
        return NULL;

    memcpy(tmr, &tmr_class, sizeof(*tmr));

    return tmr;
}

void timer_discard(struct timer *tmr) {
    
    assert(tmr != NULL);

    free(tmr);
}

int 
timer_activate(struct timer *tmr, 
               int id) {
    int r;

    assert(tmr != NULL);

    if(id == -1) {
        tmr->active_src = -1;
        return 0;
    }

    if((r = timer_check_id(id, (TIMER_FLAGS_DISABLED | 
                                TIMER_FLAGS_UNREGISTERED))))
        return r;

    tmr->active_src = id;

    return 0;
}

int 
timer_set_time(struct timer *tmr, 
               int id, 
               long long nsec) {

    int r;

    assert(tmr != NULL);

    if((r = timer_check_id(id, (TIMER_FLAGS_DISABLED | 
                                TIMER_FLAGS_UNREGISTERED))))
        return r;

    tmr->src[id].time = nsec;

    return 0;
}

int 
timer_adjust_time(struct timer *tmr, 
                  int id, 
                  long long nsec) {
    int r;

    assert(tmr != NULL);

    if((r = timer_check_id(id, (TIMER_FLAGS_DISABLED | 
                                TIMER_FLAGS_UNREGISTERED))))
        return r;

    if(tmr->src[id].time < 0)
        return -EAGAIN;

    tmr->src[id].time += nsec;

    return 0;
}

long long 
timer_get_time(struct timer *tmr, 
               int id) {
    int r;

    assert(tmr != NULL);

    if((r = timer_check_id(id, (TIMER_FLAGS_DISABLED | 
                                TIMER_FLAGS_UNREGISTERED))))
        return r;

    if(tmr->src[id].time == -1)
        return -EAGAIN;

    return tmr->src[id].time;
}

int 
timer_get_active_source(struct timer *tmr) {
    
    assert(tmr != NULL);

    if(tmr->active_src < 0)
        return -ENOENT;
    
    return tmr->active_src;
}

