/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright( C) 2000, 2001 Edgar Denny( e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 * ( at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <ctype.h>

#include "text_util.h"
#include "food.h"
#include "recipe.h"
#include "support.h"
#include "recipe_win.h"
#include "nutr_goal.h"
#include "nutr_goal_dlg_ui.h"
#include "rdi.h"
#include "wrap_mysql.h"

static GladeXML *xml = NULL;

static void load_xml( void);
static void connect_signals( void);
static void nutr_goal_load_data( void);
static GList * get_changed_nutr_goal( void);

/* callbacks. */
static void on_close_button_released( GtkButton *, gpointer);
static void on_save_button_released( GtkButton *, gpointer);

struct t_nutr_goal {
	char *widget; /* name of Glade widget */
	char *idx; /* index to nutr_no in database. */
};

/* nutrient labeled "no_name" - it is vitamin A, 
 * but measured in IU rather than mcg RE. */
struct t_nutr_goal ng_data[NO_NUTR] = 
{
	/* macro-nutrients */
	{"protein", "203"}, {"fat", "204"}, {"sat_fat", "606"}, 
	{"mono_fat", "645"}, {"poly_fat", "646"}, {"carb", "205"}, 
	{"chol", "601"}, {"fiber", "291"}, {"kj", "268"}, {"kcal", "208"}, 
	/* micro-nutrients. */
	{"vita", "392"}, {"no_name", "318"}, {"vite", "394"}, {"vitc", "401"}, 
	{"thia", "404"}, {"ribo", "405"}, {"nia", "406"}, {"panto", "410"}, 
	{"vitb6", "415"}, {"fol", "417"}, {"vitb12", "418"}, {"ca", "301"}, 
	{"fe", "303"}, {"mg", "304"}, {"p", "305"}, {"k", "306"}, 
	{"na", "307"}, {"zn", "309"}, {"cu", "312"}, {"mn", "315"}, 
	{"se", "317"}, 
	/* proteins */
	{"tryp", "501"}, {"thre", "502"}, {"isol", "503"}, 
	{"leuc", "504"}, {"lysi", "505"}, {"meth", "506"}, {"cyst", "507"}, 
	{"phen", "508"}, {"tyro", "509"}, {"vali", "510"}, {"argi", "511"}, 
	{"hist", "512"}, {"alan", "513"}, {"aspa", "514"}, {"glut", "515"}, 
	{"glyc", "516"}, {"prol", "517"}, {"seri", "518"}, 
	/* fats */
	{"fat_4d0", "607"},  {"fat_6d0", "608"}, {"fat_8d0", "609"}, 
	{"fat_10d0", "610"}, {"fat_12d0", "611"}, {"fat_14d0", "612"}, 
	{"fat_16d0", "613"}, {"fat_18d0", "614"}, {"fat_20d0", "615"}, 
	{"fat_18d1", "617"}, {"fat_18d2", "618"}, {"fat_18d3", "619"},
	{"fat_20d4", "620"}, {"fat_22d6", "621"}, {"fat_22d0", "624"}, 
	{"fat_14d1", "625"}, {"fat_16d1", "626"}, {"fat_18d4", "627"}, 
	{"fat_20d1", "628"}, {"fat_20d5", "629"}, {"fat_22d1", "630"}, 
	{"fat_22d5", "631"}, {"fat_15d0", "652"}, {"fat_17d0", "653"}, 
	{"fat_24d0", "654"},  
	/* miscelaneous */
	{"ash", "207"}, {"alco", "221"}, {"water", " 255"}, {"phyto", "636"}, 
	{"theob", "263"}, {"caff", "262"}, {"sugar", "269" } 
};

/* load the glade xml if not already loaded. */
static void
load_xml()
{
	static gboolean loaded_xml = FALSE;

	/* load the glade interface. */
	if ( !loaded_xml) {
		xml = glade_xml_new( 
			GNUTRITION_GLADEDIR "/nutr_goal_dlg.glade", NULL);
		loaded_xml = TRUE;
		if ( xml) {
			connect_signals();
		} else {
			g_log( "Gnutrition", G_LOG_LEVEL_ERROR,
				"cannot load: nutr_goal_dlg.glade\n");
			return;
		}
	}
}

static void
connect_signals()
{
	gtk_signal_connect( GTK_OBJECT( 
		glade_xml_get_widget( xml, "close_button")),
		"released", GTK_SIGNAL_FUNC( on_close_button_released), NULL);
	
	gtk_signal_connect( GTK_OBJECT( 
		glade_xml_get_widget( xml, "save_button")),
		"released", GTK_SIGNAL_FUNC( on_save_button_released), NULL);
}

/* get the nutrition goal and put values in the text entries. */
static void
nutr_goal_load_data()
{
	GtkWidget *entry;
	char *widget_name;
	GList *list, *ptr;
	char **elm;
	int i;

	/* initialize all the entries to zero. */
	for ( i=0; i<NO_NUTR; i++ ) {
		widget_name = g_strconcat( ng_data[i].widget, "_entry", NULL);
		if ( strcmp( widget_name, "no_name_entry") == 0) {
			g_free( widget_name);
			continue;
		}
		entry = glade_xml_get_widget( xml, widget_name);
		gtk_entry_set_text( GTK_ENTRY( entry), "0.0000");
		g_free( widget_name);
	}

	list = gnutr_get_nutrition_goal();

	for ( ptr = list; ptr; ptr = ptr->next) {
		elm = ( char **)ptr->data;
		/* elm[0] = nutr_no, elm[1] = goal_val. */

		for ( i=0; i<NO_NUTR; i++) {
			if ( strcmp( elm[0], ng_data[i].idx) == 0) {
				widget_name = g_strconcat( ng_data[i].widget, 
					"_entry", NULL);
				entry = glade_xml_get_widget( xml, widget_name);
				gtk_entry_set_text( GTK_ENTRY( entry), elm[1]);
				g_free( widget_name);
			}
		}
	}

	gnutr_free_row_list( list, 2);
}

/* Display nutrient goal dialog. */
void
gnutr_show_nutr_goal_dlg()
{
	GtkWidget *widget;

	if ( !xml) load_xml();

	nutr_goal_load_data();

	/* display the dialog. */
	widget = glade_xml_get_widget( xml, "nutr_goal_dlg");
	gtk_widget_show( widget);
}

/* get the values from the nutrient goal dialog so that they
 * can be saved to the database. */
static GList *
get_changed_nutr_goal() 
{
	GtkWidget *entry;
	char *widget_name;
	int i;
	double num;
	char *endpoint, *text_num;
	char **elm;
	GList *list = NULL;

	for ( i=0; i<NO_NUTR; i++) {
		widget_name = g_strconcat(  ng_data[i].widget, "_entry", NULL);

		if ( strcmp( widget_name, "no_name_entry") == 0) continue;

		entry = glade_xml_get_widget( xml, widget_name);
		g_assert( entry);
		g_free( widget_name);

		/* get the goal_val from the entry. */
		text_num =  gtk_entry_get_text( GTK_ENTRY( entry));
		num = strtod( text_num, &endpoint);

		/* check that the text is a number, if not call up an
		 * error message dialog. */
		if ( text_num == endpoint) {
			GtkWidget *error_box;
			char *error_msg;

			error_msg = g_strdup_printf( "Invalid number: %s", 
				text_num);
			error_box = gnome_error_dialog( error_msg);
			gtk_widget_show( error_box);
			g_free( error_msg);
			return NULL;
		} 

		/* append the nutr_no and goal_val to the list. */
		elm = ( char **)g_malloc( 2 * sizeof( char *));
		elm[0] = g_strdup( ng_data[i].idx);
		elm[1] = g_strdup( text_num);
		list = g_list_append( list, ( gpointer)elm);
	}
	return list;
}

/* When the "Close" button is released in the nutrient goal dialog - hide 
 * the nutrient goal dialog, discard any changes. */
static void
on_close_button_released( GtkButton *button,
                          gpointer   user_data)
{
	GtkWidget *widget = glade_xml_get_widget( xml, "nutr_goal_dlg");
	gtk_widget_hide( widget);
}

/* When the "Save" button released in the nutrient goal dialog - save the 
 * nutrient goals to file. */
static void
on_save_button_released( GtkButton *button,
                         gpointer   user_data)
{
	GList *list;

	list = get_changed_nutr_goal();

	gnutr_save_nutr_goal( list);
}
