/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla XUL Toolkit Testing Code.
 *
 * The Initial Developer of the Original Code is
 * Paolo Amadini <http://www.amadzone.org/>.
 * Portions created by the Initial Developer are Copyright (C) 2009
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

/**
 * Runs a browser-chrome test defined through a generator function.
 *
 * This object is a singleton, initialized automatically when this script is
 * included. Every browser-chrome test file includes a new copy of this script.
 */
var testRunner = {
  _testIterator: null,
  _lastEventResult: undefined,
  _testRunning: false,
  _eventRaised: false,

  // --- Main test runner ---

  /**
   * Runs the test described by the provided generator function asynchronously.
   *
   * Calling yield in the generator will cause it to wait until continueTest is
   * called. The parameter provided to continueTest will be the return value of
   * the yield operator.
   *
   * @param aGenerator
   *        Test generator function. The function will be called with no
   *        arguments to retrieve its iterator.
   */
  runTest: function TR_runTest(aGenerator) {
    waitForExplicitFinish();
    testRunner._testIterator = aGenerator();
    testRunner.continueTest();
  },

  /**
   * Continues the currently running test.
   *
   * @param aEventResult
   *        This will be the return value of the yield operator in the test.
   */
  continueTest: function TR_continueTest(aEventResult) {
    // Store the last event result, or set it to undefined.
    testRunner._lastEventResult = aEventResult;

    // Never reenter the main loop, but notify that the event has been raised.
    if (testRunner._testRunning) {
      testRunner._eventRaised = true;
      return;
    }

    // Enter the main iteration loop.
    testRunner._testRunning = true;
    try {
      do {
        // Call the iterator, but don't leave the loop if the expected event is
        // raised during the execution of the generator.
        testRunner._eventRaised = false;
        testRunner._testIterator.send(testRunner._lastEventResult);
      } while (testRunner._eventRaised);
    }
    catch (e) {
      // This block catches exceptions raised by the generator, including the
      // normal StopIteration exception. Unexpected exceptions are reported as
      // test failures.
      if (!(e instanceof StopIteration))
        ok(false, e);
      // In any case, stop the tests in this file.
      finish();
    }

    // Wait for the next event or finish.
    testRunner._testRunning = false;
  },

  // --- Auxiliary functions ---

  _isFirstEvent: true,

  /**
   * Continues the running test every second time this function is called.
   */
  continueAfterTwoEvents: function TR_continueAfterTwoEvents() {
    if (testRunner._isFirstEvent) {
      testRunner._isFirstEvent = false;
      return;
    }
    testRunner._isFirstEvent = true;
    testRunner.continueTest();
  },

  // --- Support for multiple tests ---

  /**
   * This generator function yields each value obtained by the given generators,
   * in the order they are specified in the array.
   *
   * @param aArrayOfGenerators
   *        Array of generator functions. The functions will be called with no
   *        arguments to retrieve their iterators.
   *
   * @return
   *        The iterator generated by this generator function.
   */
  chainGenerator: function TR_chainGenerator(aArrayOfGenerators) {
    // Obtain each iterator in turn.
    for (let [, curGenerator] in Iterator(aArrayOfGenerators)) {
      var curIterator = curGenerator();
      // Call each iterator until it completes, while ensuring propagation of
      // both the the values provided to and returned by the yield operator.
      try {
        var value = undefined;
        while (true) {
          value = yield curIterator.send(value);
        }
      }
      catch(e if e instanceof StopIteration) {
        // The iterator has finished providing all the values.
      }
    }
  },

  /**
   * Runs multiple tests.
   *
   * This function operates asynchronously. Because of this, it should be called
   * by the last line of a test script.
   *
   * @param aArrayOfTestGenerators
   *        Array containing references to the test generator functions to run.
   *
   * @see #runTest
   */
  runTests: function TR_runTests(aArrayOfTestGenerators) {
    testRunner.runTest(function() {
      return testRunner.chainGenerator(aArrayOfTestGenerators);
    });
  }
};
