/*

      This file is part of the Goptical Core library.
  
      The Goptical library is free software; you can redistribute it
      and/or modify it under the terms of the GNU General Public
      License as published by the Free Software Foundation; either
      version 3 of the License, or (at your option) any later version.
  
      The Goptical library is distributed in the hope that it will be
      useful, but WITHOUT ANY WARRANTY; without even the implied
      warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
      See the GNU General Public License for more details.
  
      You should have received a copy of the GNU General Public
      License along with the Goptical library; if not, write to the
      Free Software Foundation, Inc., 59 Temple Place, Suite 330,
      Boston, MA 02111-1307 USA
  
      Copyright (C) 2010-2011 Free Software Foundation, Inc
      Author: Alexandre Becoulet

*/


#include <deque>

#include <Goptical/Trace/Tracer>
#include <Goptical/Trace/Result>
#include <Goptical/Trace/Ray>
#include <Goptical/Trace/Ray>
#include <Goptical/Sys/System>
#include <Goptical/Sys/Source>
#include <Goptical/Error>
#include <Goptical/Sys/Surface>
#include <Goptical/Math/VectorPair>
#include <Goptical/Trace/Distribution>
#include <Goptical/Trace/Sequence>

namespace _Goptical {

  namespace Trace {

    Tracer::Tracer(const const_ref<Sys::System> &system)
      : _system(system),
        _params(system->get_tracer_params()),
        _result(),
        _result_ptr(&_result)
    {
    }

    Tracer::~Tracer()
    {
    }

    template <IntensityMode m> void Tracer::trace_seq_template()
    {
      Result &result = *_result_ptr;

      result.init(*_system);

      // stack of rays to propagate
      rays_queue_t tmp[2];

      unsigned int swaped = 0;
      rays_queue_t *generated;
      rays_queue_t *source_rays = &tmp[1];
      const std::vector<const_ref<Sys::Element> > &seq = _params._sequence->_list;
      const Sys::Element *entrance = 0;

      // find entry element (first non source)
      for (unsigned int i = 0; i < seq.size(); i++)
        {
          if (!dynamic_cast<const Sys::Source *>(seq[i].ptr()))
            {
              entrance = seq[i].ptr();
              break;
            }
        }

      for (unsigned int i = 0; i < seq.size(); i++)
        {
          const Sys::Element *element = seq[i].ptr();

          if (_system != element->get_system())
            throw Error("Sequence contains element which is not part of the System");

          if (!element->is_enabled())
            continue;

          Result::element_result_s &er = result.get_element_result(*element);

          generated = er._generated ? er._generated : &tmp[swaped];
          result._generated_queue = generated;
          generated->clear();

          if (const Sys::Source *source = dynamic_cast<const Sys::Source *>(element))
            {
              result._sources.push_back(source);
              Sys::Source::targets_t elist;
              if (entrance)
                elist.push_back(entrance);
              source->generate_rays<m>(result, elist);
            }
          else
            {
              element->process_rays<m>(result, source_rays);
              // swap ray buffers
            }

          GOPTICAL_DEBUG(" " << generated->size() << " rays generated by " << *element);
          source_rays = generated;
          swaped ^= 1;
        }

      result._generated_queue = 0;
    }

    template <IntensityMode m> void Tracer::trace_template()
    {
      Result            &result = *_result_ptr;

      result.init(*_system);

      if (_params._propagation_mode != RayPropagation)
        throw Error("Diffractive propagation not supported in non sequential mode");

      // stack of rays to propagate

      rays_queue_t source_rays;

      Sys::Source::targets_t entry;
      entry.push_back(&_system->get_entrance_pupil());

      // FIXME avoid container use here
      std::vector<const Sys::Source *> slist;
      delegate_push<typeof(slist), const Sys::Source &> d(slist);
      _system->get_elements<Sys::Source>(d);

      GOPTICAL_FOREACH(s, slist)
        {
          const Sys::Source &source = **s;

          if (_system != source.get_system())
            throw Error("can not trace with Source which is not part of the System");

          if (!source.is_enabled())
            continue;

          result._sources.push_back(&source);

          // get rays from source
          source_rays.clear();
          result._generated_queue = &source_rays;
          source.generate_rays<m>(result, entry);

          // copy to source generated rays
          {
            Result::element_result_s &source_er = result.get_element_result(source);

            if (source_er._generated)
              *source_er._generated = source_rays;
          }

          GOPTICAL_DEBUG("NSeq Ray Trace: " << source_rays.size() << " Rays");

          // trace each ray generated by source through the system

          rays_queue_t gqueue;
          result._generated_queue = &gqueue;

          GOPTICAL_FOREACH(r, source_rays)
            {
              Ray *ray = *r;
              unsigned int bounce = _params._max_bounce;

              // trace relfected/refracted ray further
              while (1)
                {
                  // check bounce limit
                  if (!bounce--)
                    result._bounce_limit_count++;
                  else
                    {
                      Math::VectorPair3 intersect; // intersection point and normal (intersect surface local)

                      // find ray / surface interction
                      if (Sys::Surface *s = _system->colide_next(_params, intersect, *ray))
                        {
                          result.add_intercepted(*s, *ray);

                          // transform incident ray to surface local
                          const Math::Transform<3> &t = ray->get_creator()->get_transform_to(*s);
                          Math::VectorPair3 local(t.transform_line(*ray));

                          s->trace_ray<m>(result, *ray, local, intersect);
                        }
                    }

                  // pick next ray to trace further through the system
                  if (gqueue.empty())
                    break;
                  
                  ray = gqueue.front();
                  gqueue.pop_front();

                  result.add_generated(*ray->get_creator(), *ray);
                }
            }
        }

      result._generated_queue = 0;
    }

    void Tracer::trace()
    {
      Result    &result = *_result_ptr;

      // clear previous results
      result.prepare();

      result._params = &_params;

      switch (_params._intensity_mode)
        {
        case SimpleTrace:
          if (!_params._sequential_mode)
            trace_template<SimpleTrace>();
          else
            trace_seq_template<SimpleTrace>();
          break;

        case IntensityTrace:
          if (!_params._sequential_mode)
            trace_template<IntensityTrace>();
          else
            trace_seq_template<IntensityTrace>();
          break;

        case PolarizedTrace:
          if (!_params._sequential_mode)
            trace_template<PolarizedTrace>();       
          else
            trace_seq_template<PolarizedTrace>();
          break;
        }
    }

  }

}

