/* modfuncs.c: -*- C -*-  Dynamically loaded Meta-HTML modules. */

/*  Copyright (c) 1996 Brian J. Fox
    Author: Brian J. Fox (bfox@ai.mit.edu) Tue Dec 24 10:06:54 1996.

   This file is part of <Meta-HTML>(tm), a system for the rapid
   deployment of Internet and Intranet applications via the use of the
   Meta-HTML language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

/* This version of Meta-HTML can handle dynamically loaded modules. */
#define PACKAGE_INITIALIZER_EXTRA_CODE \
  pagefunc_set_variable ("mhtml::module-capable", "true");

#include "language.h"
#if defined (USE_SHL_LOAD)
#  include <dl.h>
#else
#  include <dlfcn.h>
#endif

#if !defined (RTLD_NOW)
#  define RTLD_NOW 1
#endif

static void pf_load_module (PFunArgs);

static PFunDesc func_table[] =
{
  { "LOAD-MODULE",	0, 0, pf_load_module },
  { (char *)NULL,	0, 0, (PFunHandler *)NULL }
};

PACKAGE_INITIALIZER (initialize_module_functions)

DOC_SECTION (DYNAMIC-MODULES)

static char *static_dirs[5] =
 { "/opt/metahtml/lib", "/opt/metahtml/modules", "/www/lib",
   "/www/bin", (char *)NULL };

static char *
fully_qualified_module_name (char *name)
{
  int namelen = strlen (name);

  if ((*name != '/') && (namelen < 1024))
    {
      register int i;
      static char buffer[2048];
      char **dirs = symbol_get_values ("mhtml::module-directories");
      struct stat finfo;
      int add_extension = 0;

      {
	char *temp = strrchr (name, '.');

	if (!temp)
	  add_extension++;
      }

      if (dirs == (char **)NULL)
	dirs = &static_dirs[0];

      for (i = 0; dirs[i] != (char *)NULL; i++)
	{
	  sprintf (buffer, "%s/%s", dirs[i], name);
	  if (add_extension)
	    {
#if defined (USE_SHL_LOAD)
	      strcat (buffer, ".O");
#else
	      strcat (buffer, ".so");
#endif
	    }

	  if (stat (buffer, &finfo) == 0)
	    return (strdup (buffer));
	}
    }

  return (strdup (name));
}

#if defined (USE_SHL_LOAD)
extern int errno;
#  define module_recent_error(file) \
	page_syserr ("LOAD-MODULE: (%s) %s", file, strerror (errno))
#else
#  define module_recent_error(file) \
	page_syserr ("LOAD-MODULE: (%s) %s", file, dlerror ())
#endif

#define MODULE_SYM_NAME "mhtml::loaded-modules"

typedef void VFUN (void);

DEFUN (pf_load_module, module-name,
"Loads a module dynamically at runtime, adding the function
definitions found therein to the global package of <Meta-HTML>
functions.

<code>load-module</code> returns a non-empty string if the module is
loaded successfully, or places an error in <funref language-operators
system-error-output> if not.  The string returned is the fully
qualified pathname of the module just loaded.

<var module-name> is searched for by looking in each directory
specified in the array <varref mhtml::module-directories>, or by
loading the module as if the name specified is the full pathname to
that file.

Once a module is loaded, the functions within it can be invoked just
as with any standard <Meta-HTML> command.

See the <code>examples.c</code> file in the <code>modules</code>
directory of your distribution for more information on writing
<Meta-HTML> modules.

Example:

<example>
<set-var loaded? = <load-module /www/lib/example.so>>
</example>")
{
  char *module_name = mhtml_evaluate_string (get_positional_arg (vars, 0));
  char *result = (char *)NULL;
  char *mode_arg = mhtml_evaluate_string (get_value (vars, "mode"));
#if defined (USE_SHL_LOAD)
  int mode_flag = BIND_IMMEDIATE;
#else
  int mode_flag = RTLD_NOW;
#endif

  if (!empty_string_p (mode_arg))
    {
      if (strcasecmp (mode_arg, "lazy") == 0)
	{
#if defined (USE_SHL_LOAD)
	  mode_flag = BIND_DEFERRED;
#else
	  mode_flag = RTLD_LAZY;
#endif
	}
    }
  xfree (mode_arg);

  if (module_name != (char *)NULL)
    {
      register int i;
      char **modules = symbol_get_values (MODULE_SYM_NAME);
      char *fqn = fully_qualified_module_name (module_name);

      if (modules != (char **)NULL)
	{
	  for (i = 0; modules[i] != (char *)NULL; i++)
	    if (strcmp (fqn, modules[i]) == 0)
	      {
		result = strdup (fqn);
		break;
	      }
	}

      if (result == (char *)NULL)
	{
#if defined (USE_SHL_LOAD)
	  void *handle = (void *)shl_load (fqn, mode_flag, 0L);
#else
	  void *handle = (void *)dlopen (fqn, mode_flag);
#endif

	  if (handle != (void *)NULL)
	    {
	      char *no_init = mhtml_evaluate_string
		(get_value (vars, "noinitialize"));

	      /* Some operating systems lie.  For example, FreeBSD
		 says that dlsym works.  But it apparently doesn't.
		 Don't know why, can't say how.  It ALWAYS calls
		 _init() though, so I guess we can just change all
		 of our loadable libraries to have that function.
		 Can you say "UGH?" */
#if defined (__FreeBSD__)
	      if (no_init == (char *)NULL)
		no_init = strdup ("true");
#endif
	      if (no_init == (char *)NULL)
		{
		  VFUN *initfunc = (VFUN *)NULL;

#if defined (USE_SHL_LOAD)
		  shl_findsym (&handle, "module_initialize", TYPE_PROCEDURE,
			       &initfunc);
#else
		  initfunc = (VFUN *) dlsym (handle, "module_initialize");
#endif

		  if (initfunc != (VFUN *)NULL)
		    {
		      (*initfunc) ();
		      result = strdup (fqn);
		    }
		  else
		    {
		      module_recent_error (fqn);
#if defined (USE_SHL_LOAD)
		      shl_unload (handle);
#else
		      dlclose (handle);
#endif
		    }
		}
	      else
		result = strdup (fqn);

	      if (result != (char *)NULL)
		{
		  Symbol *sym = symbol_intern (MODULE_SYM_NAME);
		  symbol_add_value (sym, fqn);
		}

	      xfree (no_init);
	    }
	  else
	    {
	      /* Error opening library. */
	      module_recent_error (module_name);
	    }
	}

      free (module_name);
      free (fqn);
    }

  if (result != (char *)NULL)
    {
      bprintf_insert (page, start, "%s", result);
      *newstart += strlen (result);
      xfree (result);
    }
}
