## Copyright (C) 2008 Soren Hauberg <soren@hauberg.org>
## Copyright (C) 2015 Julien Bect <jbect@users.sourceforge.net>
##
## This program is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or (at
## your option) any later version.
##
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.  If not, see
## <http://www.gnu.org/licenses/>.

## -*- texinfo -*-
## @deftypefn {Function File} html_help_text (@var{name}, @var{outname}, @var{options})
## Writes a function help text to disk formatted as @t{HTML}.
##
## The help text of the function @var{name} is written to the file @var{outname}
## formatted as @t{HTML}. The design of the generated @t{HTML} page is controlled
## through the @var{options} variable. This is a structure with the following
## optional fields.
##
## @table @samp
## @item header
## This field contains the @t{HTML} header of the generated file. Through this
## things such as @t{CSS} style sheets can be set.
## @item footer
## This field contains the @t{HTML} footer of the generated file. This should
## match the @samp{header} field to ensure all opened tags get closed.
## @item title
## This field sets the title of the @t{HTML} page. This is enforced even if the
## @samp{header} field contains a title.
## @end table
##
## @var{options} structures for various projects can be with the @code{get_html_options}
## function. As a convenience, if @var{options} is a string, a structure will
## be generated by calling @code{get_html_options}.
##
## @seealso{get_html_options, generate_package_html}
## @end deftypefn

function [header, text, footer] = texi2html (text, options = struct (), root = "")

  ## Check number of input arguments
  if (nargin < 1)
    print_usage ();
    error ("Not enough input arguments: at least one argument was expected.");
  endif

  ## Process input argument 'options'
  if (ischar (options)) || (isstruct (options))
    options = get_html_options (options);
  else
    error ("Second input argument must be a string or a structure");
  endif

  ## Add easily recognisable text before and after real text
  start = "###### OCTAVE START ######";
  stop  = "###### OCTAVE STOP ######";
  text = sprintf ("%s\n%s\n%s\n", start, text, stop);

  ## Prevent empty <pre> </pre> blocks
  ## (see https://savannah.gnu.org/bugs/?44451)
  text = regexprep (text, '([\r\n|\n])[ \t]*@group', '$1@group');
  text = regexprep (text, '([\r\n|\n])[ \t]*@end', '$1@end');

  ## Remove one leading white space.  Assuming that all non-empty
  ## lines start with "## ", this prevents one extra white space
  ## from showing up in example blocks.
  text = regexprep (text, '([\r\n|\n])[ \t]', '$1');

  ## Run makeinfo
  orig_text = text;
  [text, status] = __makeinfo__ (text, ...
    "html", @(x) options.seealso (root, x{:}));
  if (status != 0)
    error ("__makeinfo__ returned with error code %d\n. Couldn't parse\
      texinfo:\n%s", status, orig_text (1:min (200, length (orig_text))));
  endif

  ## Check encoding
  tmp = regexp (text, "charset\s*=\s*([^\s\"]*)", "tokens");
  if (! isempty (tmp))
    charset = tmp{1}{1};
    if (! strcmp (options.charset, charset))
      warning (["makeinfo's output is encoded in %s, but will be " ...
        "interpreted with options.charset = %s"], charset, options.charset);
    endif
  endif

  ## Extract the body of makeinfo's output
  p_start = sprintf ('\\s*(<p>)?\\s*%s\\s*(</p>)?\\s*', start);
  p_stop = sprintf ('\\s*(<p>)?\\s*%s\\s*(</p>)?\\s*', stop);
  [i1, i2] = regexp (text, p_start);
  i3 = regexp (text, p_stop);
  text = text((i2 + 1):(i3 - 1));
  
  ## Insert class="deftypefn" attribute
  text = insert_deftypefn_class_attribute (text);

  ## Read 'options' input argument
  [header, title, footer] = get_header_title_and_footer ...
    ("function", options, "", root);

endfunction


function text = insert_deftypefn_class_attribute (text)

  ## @deftypefn pattern for TexInfo 4.x
  p1 = '&mdash;\s*(Function.*?)\s*<br>';
  
  if ~ isempty (regexp (text, p1, 'once'))  ## TexInfo 4.x

    ## <div class="defun">
    ## &mdash; Function File: ... <br>
    ## &mdash; Function File: ... <br>
    ## <blockquote> ... </blockquote>
    ## </div>

    p2 = sprintf (['\\s*<div\\s*(class="[a-z]*")?>\\s*' ...
      '((%s\\s*)+)<blockquote>(.*?)\\s*</blockquote>\\s*</div>\\s*'], p1);
    text = regexprep (text, p2, '<dl>\n$2<dd>$5\n</dd></dl>');
    text = regexprep (text, p1, '<dt class="deftypefn">$1</dt>');
    
  else  ## TexInfo 5.x

    ## <dl>
    ## <dt><a name="index-plot"></a>Function File: ... </dt>
    ## <dt><a name="index-plot-1"></a>Function File: ... </dt>
    ## <dd> ... </dd>
    ## </dl>

    ## @deftypefn pattern for TexInfo 5.x
    p1 = '<dt>\s*((<a.*?</a>)?\s*Function.*?)\s*<br>';

    text = regexprep (text, p1, '<dt class="deftypefn">$1</dt>');

  endif
  
endfunction
