"======================================================================
|
|   Abstract NetClient framework
|
|
 ======================================================================"

"======================================================================
|
| NetUser and NetEnvironment are Copyright 2000 Cincom, Inc.
| NetResponse, PluggableReporter and *Error are (c) 1995 Kazuki Yasumatsu
| and in the public domain.
|
| The rest is copyright 2002, 2007, 2008 Free Software Foundation, Inc.
| and written by Paolo Bonzini.
|
| This file is part of the GNU Smalltalk class library.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
|
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
| General Public License for more details.
|
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LIB.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02110-1301, USA.
|
 ======================================================================"



Eval [
    NetClients at: #LineEndCR put: #cr.
    NetClients at: #LineEndLF put: #nl.
    NetClients at: #LineEndCRLF put: #crnl.
    NetClients at: #LineEndTransparent put: #yourself
]



Object subclass: NetUser [
    | username password |
    
    <category: 'NetClients-Framework'>
    <comment: 'Instances of this class hold the username and password used to login to a mail server.

Instance Variables:
	username	<String>		username string
	password	<String>		password string'>

    NetUser class >> username: aUsername password: aPassword [
	"NetUser username: 'foo' password: 'foo'"

	<category: 'instance creation'>
	| user |
	user := self new.
	^user
	    username: aUsername;
	    password: aPassword yourself
    ]

    password [
	<category: 'accessing'>
	^password
    ]

    password: aString [
	<category: 'accessing'>
	password := aString
    ]

    username [
	<category: 'accessing'>
	^username
    ]

    username: aString [
	<category: 'accessing'>
	username := aString
    ]
]



Object subclass: NetEnvironment [
    | debugStream debugCategories debugClasses trace logFileName |
    
    <category: 'NetClients-Framework'>
    <comment: nil>

    NetEnvironment class [
	| uniqueInstance |
	
    ]

    NetEnvironment class >> default [
	<category: 'accessing'>
	^uniqueInstance isNil 
	    ifTrue: [uniqueInstance := self new]
	    ifFalse: [uniqueInstance]
    ]

    debugCategories [
	<category: 'accessing'>
	debugCategories isNil 
	    ifTrue: 
		[debugCategories := Set new.
		debugCategories add: #general].
	^debugCategories
    ]

    debugClasses [
	<category: 'accessing'>
	^debugClasses isNil 
	    ifTrue: [debugClasses := Set new]
	    ifFalse: [debugClasses]
    ]

    debugStream [
	<category: 'accessing'>
	^debugStream
    ]

    debugStream: aStream [
	<category: 'accessing'>
	debugStream := aStream
    ]

    logFileName [
	<category: 'accessing'>
	logFileName isNil ifTrue: [logFileName := 'NetClientLog.txt'].
	^logFileName
    ]

    logFileName: aString [
	<category: 'accessing'>
	logFileName := aString
    ]

    trace [
	<category: 'accessing'>
	trace isNil ifTrue: [trace := false].
	^trace
    ]

    trace: aBoolean [
	<category: 'accessing'>
	trace := aBoolean
    ]

    debug: aBlock level: aLevel [
	<category: 'debugging'>
	(self trace and: [self debugCategories includes: aLevel]) 
	    ifTrue: [aBlock value]
    ]

    log: aStringOrBlock [
	<category: 'debugging'>
	self log: aStringOrBlock level: #general
    ]

    log: aStringOrBlock level: aLevel [
	<category: 'debugging'>
	| stream i briefMsg aMsg |
	self debug: 
		[(stream := self debugStream) == nil ifTrue: [^self].
		(aStringOrBlock isKindOf: BlockClosure) 
		    ifTrue: [aMsg := aStringOrBlock value]
		    ifFalse: [aMsg := aStringOrBlock].
		i := aMsg size.
		[i > 0 and: [(aMsg at: i) isSeparator]] whileTrue: [i := i - 1].
		briefMsg := aMsg copyFrom: 1 to: i.
		stream
		    cr;
		    nextPutAll: briefMsg;
		    flush]
	    level: aLevel
    ]

    printTrace: aString [
	<category: 'debugging'>
	| stream |
	(stream := self debugStream) == nil ifTrue: [^self].
	stream
	    cr;
	    cr;
	    nextPutAll: ' **** ' asString.
	Date today printOn: stream.
	stream nextPutAll: ' '.
	Time now printOn: stream.
	stream
	    nextPutAll: ' ' , aString , ' ****';
	    flush
    ]

    traceOff [
	<category: 'debugging'>
	self printTrace: 'Stop Trace'.
	self trace: false
    ]

    traceOn [
	<category: 'debugging'>
	self trace: true.
	self printTrace: 'Start Trace'
    ]

    addDebugCategory: symbol [
	<category: 'private'>
	self debugCategories add: symbol
    ]

    removeDebugCategory: symbol [
	<category: 'private'>
	self debugCategories remove: symbol
    ]

    reset [
	<category: 'private'>
	self resetDebugClasses.
	self resetDebugCategories
    ]

    resetDebugCategories [
	<category: 'private'>
	debugCategories := nil
    ]

    resetDebugClasses [
	<category: 'private'>
	debugClasses := nil
    ]

    addToDebug: aClass [
	<category: 'registration'>
	self debugClasses add: aClass
    ]
]



Object subclass: NetClient [
    | hostName portNumber user reporter clientPI |
    
    <import: Sockets>
    <category: 'NetClients-Framework'>
    <comment: '
Copyright (c) Kazuki Yasumatsu, 1995. All rights reserved.
'>

    NetClient class >> connectToHost: aHostname [
	<category: 'instance creation'>
	^self new connectToHost: aHostname port: nil
    ]

    NetClient class >> connectToHost: aHostname port: aPort [
	<category: 'instance creation'>
	^self new connectToHost: aHostname port: aPort
    ]

    NetClient class >> loginToHost: aHostName asUser: userString withPassword: passwdString [
	<category: 'instance creation'>
	^self 
	    loginToHost: aHostName
	    port: nil
	    asUser: userString
	    withPassword: passwdString
    ]

    NetClient class >> loginToHost: aHostName port: aNumber asUser: userString withPassword: passwdString [
	<category: 'instance creation'>
	^self new 
	    loginToHost: aHostName
	    port: aNumber
	    asUser: userString
	    withPassword: passwdString
    ]

    NetClient class >> loginUser: userString withPassword: passwdString [
	<category: 'instance creation'>
	^self loginUser: userString withPassword: passwdString
    ]

    user [
	<category: 'accessing'>
	^user
    ]

    user: aNetUser [
	<category: 'accessing'>
	user := aNetUser
    ]

    username [
	<category: 'accessing'>
	^user username
    ]

    password [
	<category: 'accessing'>
	^self user password
    ]

    username: usernameString password: passwdString [
	<category: 'accessing'>
	user := NetUser username: usernameString password: passwdString
    ]

    clientPI [
	<category: 'accessing'>
	^clientPI
    ]

    clientPI: aProtocolInterpreter [
	<category: 'accessing'>
	clientPI := aProtocolInterpreter
    ]

    hostName [
	<category: 'accessing'>
	^hostName
    ]

    hostName: aString [
	<category: 'accessing'>
	hostName := aString
    ]

    portNumber [
	<category: 'accessing'>
	portNumber isNil ifTrue: [^self protocolInterpreter defaultPortNumber].
	portNumber = 0 ifTrue: [^self protocolInterpreter defaultPortNumber].
	^portNumber
    ]

    portNumber: aNumber [
	<category: 'accessing'>
	portNumber := aNumber
    ]

    reporter [
	<category: 'accessing'>
	reporter isNil ifTrue: [reporter := Reporter new].
	^reporter
    ]

    reporter: aReporter [
	<category: 'accessing'>
	reporter := aReporter
    ]

    protocolInterpreter [
	<category: 'abstract'>
	self subclassResponsibility
    ]

    close [
	<category: 'connection'>
	^self logout
    ]

    connect [
	<category: 'connection'>
	self connectToHost: self hostName port: self portNumber
    ]

    connectIfClosed [
	<category: 'connection'>
	self clientPI closed ifTrue: [self connect]
    ]

    connectToHost: aString [
	"Establish a connection to the host <aString>."

	<category: 'connection'>
	^self connectToHost: aString port: nil
    ]

    connectToHost: aString port: aNumber [
	"Establish a connection to the host <aString>."

	<category: 'connection'>
	self hostName: aString.
	self portNumber: aNumber.
	self clientPI: (self protocolInterpreter client: self).
	self clientPI connect
    ]

    login [
	<category: 'connection'>
	
    ]

    logout [
	<category: 'connection'>
	
    ]

    loginToHost: aHostName asUser: userString withPassword: passwdString [
	<category: 'connection'>
	^self 
	    loginToHost: aHostName
	    port: nil
	    asUser: userString
	    withPassword: passwdString
    ]

    loginToHost: aHostName port: aNumber asUser: userString withPassword: passwdString [
	<category: 'connection'>
	| resp |
	hostName := aHostName.
	portNumber := aNumber.
	self username: userString password: passwdString.
	self connect.
	(resp := self login) completedSuccessfully ifFalse: [^nil]
    ]

    reconnect [
	<category: 'connection'>
	self clientPI close.
	self connect
    ]
]



Object subclass: NetProtocolInterpreter [
    | client connectionStream |
    
    <import: TCP>
    <category: 'NetClients-Framework'>
    <comment: nil>

    NetProtocolInterpreter class >> log: aString level: aLevel [
	<category: 'debugging'>
	NetEnvironment default log: aString level: aLevel
    ]

    NetProtocolInterpreter class >> registerToDebug [
	<category: 'debugging'>
	NetEnvironment default addToDebug: self
    ]

    NetProtocolInterpreter class >> client: aNetClient [
	<category: 'instance creation'>
	^self new client: aNetClient
    ]

    NetProtocolInterpreter class >> new [
	<category: 'instance creation'>
	^self basicNew initialize
    ]

    NetProtocolInterpreter class >> defaultPortNumber [
	<category: 'private-attributes'>
	^nil
    ]

    NetProtocolInterpreter class >> defaultResponseClass [
	<category: 'private-attributes'>
	^NetResponse
    ]

    client [
	<category: 'accessing'>
	^client
    ]

    reporter [
	<category: 'accessing'>
	^self client reporter
    ]

    receiveMessageUntilPeriod [
	"Receive and answer a message until period line."

	<category: 'accessing'>
	| write |
	write := WriteStream on: (String new: 4 * 1024).
	self receiveMessageUntilPeriodInto: write.
	^write contents
    ]

    receiveMessageUntilPeriodInto: aStream [
	"Receive a message until period line into aStream."

	<category: 'accessing'>
	self connectIfClosed.
	MIME.MimeEntity new parseSimpleBodyFrom: self onto: aStream
    ]

    sendMessageWithPeriod: aStream [
	"Send aStream as a message with period."

	<category: 'accessing'>
	self connectIfClosed.
	(PrependDotStream to: self)
	    nextPutAll: aStream;
	    flush
    ]

    skipMessageUntilPeriod [
	"Skip a message until period line."

	<category: 'accessing'>
	self connectIfClosed.
	MIME.MimeEntity new skipSimpleBodyFrom: self
    ]

    binary [
	<category: 'connection'>
	connectionStream class == CrLfStream 
	    ifTrue: [connectionStream := connectionStream stream]
    ]

    isBinary [
	<category: 'connection'>
	^connectionStream class ~~ CrLfStream
    ]

    text [
	<category: 'connection'>
	self binary.
	self lineEndConvention = LineEndCRLF 
	    ifTrue: [connectionStream := CrLfStream on: connectionStream]
    ]

    close [
	<category: 'connection'>
	self closed 
	    ifFalse: 
		[connectionStream close.
		connectionStream := nil].
	self liveAcrossSnapshot ifTrue: [ObjectMemory removeDependent: self]
    ]

    closed [
	<category: 'connection'>
	^connectionStream == nil
    ]

    connectionStream [
	<category: 'connection'>
	^connectionStream
    ]

    connectionStream: aSocket [
	<category: 'connection'>
	connectionStream := aSocket.
	self text.
	self liveAcrossSnapshot ifTrue: [ObjectMemory addDependent: self]
    ]

    connect [
	<category: 'connection'>
	| connection messageText |
	connection := [Socket remote: client hostName port: client portNumber] 
		    on: Error
		    do: 
			[:ex | 
			ex.
			messageText := ex messageText.
			ex return: nil].
	connection isNil ifTrue: [^self connectionFailedError: messageText].
	self connectionStream: connection
    ]

    connectIfClosed [
	<category: 'connection'>
	self closed ifTrue: [self connect]
    ]

    reconnect [
	<category: 'connection'>
	self
	    close;
	    connect
    ]

    decode: aString [
	<category: 'encoding'>
	^aString
    ]

    encode: aString [
	<category: 'encoding'>
	^aString
    ]

    client: aNetClient [
	<category: 'initialize-release'>
	client := aNetClient
    ]

    initialize [
	<category: 'initialize-release'>
	
    ]

    release [
	<category: 'initialize-release'>
	self close
    ]

    checkResponse [
	<category: 'private'>
	self checkResponse: self getResponse
    ]

    checkResponse: response [
	<category: 'private'>
	self checkResponse: response
	    ifError: [self protocolError: response statusMessage]
    ]

    checkResponse: reponse ifError: errorBlock [
	<category: 'private'>
	
    ]

    connectionClosedError: messageText [
	<category: 'private'>
	^(ConnectionClosedError new)
	    tag: messageText;
	    signal: 'Connection closed: ' , messageText
    ]

    connectionFailedError: messageText [
	<category: 'private'>
	^(ConnectionFailedError new)
	    tag: messageText;
	    signal: 'Connection failed: ' , messageText
    ]

    getResponse [
	<category: 'private'>
	^self class defaultResponseClass fromClient: self
    ]

    loginIncorrectError: messageText [
	<category: 'private'>
	^(LoginIncorrectError new)
	    tag: messageText;
	    signal: 'Login incorrect: ' , messageText
    ]

    protocolError: messageText [
	<category: 'private'>
	^(ProtocolError new)
	    tag: messageText;
	    signal: 'Protocol error: ' , messageText
    ]

    lineEndConvention [
	<category: 'private-attributes'>
	^LineEndCRLF
    ]

    liveAcrossSnapshot [
	<category: 'private-attributes'>
	^false
    ]

    atEnd [
	<category: 'stream accessing'>
	^connectionStream atEnd
    ]

    contents [
	<category: 'stream accessing'>
	^self decode: connectionStream contents
    ]

    cr [
	<category: 'stream accessing'>
	| conv |
	conv := self lineEndConvention.
	(conv = LineEndCR or: [conv = LineEndTransparent]) 
	    ifTrue: [^connectionStream nextPut: Character cr].
	conv = LineEndLF ifTrue: [^connectionStream nextPut: Character nl].
	conv = LineEndCRLF 
	    ifTrue: 
		[^connectionStream
		    nextPut: Character cr;
		    nextPut: Character nl].
	self error: 'Undefined line-end convention'
    ]

    flush [
	<category: 'stream accessing'>
	connectionStream flush
    ]

    next [
	<category: 'stream accessing'>
	^connectionStream next
    ]

    next: anInteger [
	<category: 'stream accessing'>
	^self decode: (connectionStream next: anInteger)
    ]

    nextAvailable: anInteger [
	<category: 'stream accessing'>
	^self decode: (connectionStream nextAvailable: anInteger)
    ]

    nextLine [
	<category: 'stream accessing'>
	| write byte |
	write := WriteStream on: (String new: 128).
	[connectionStream atEnd] whileFalse: 
		[byte := connectionStream next.
		byte == Character cr 
		    ifTrue: 
			[connectionStream peekFor: Character nl.
			^self decode: write contents].
		byte == Character nl ifTrue: [^self decode: write contents].
		write nextPut: byte].
	^self decode: write contents
    ]

    nextPut: aCharacter [
	<category: 'stream accessing'>
	connectionStream nextPutAll: (self encode: (String with: aCharacter))
    ]

    nextPutAll: aString [
	<category: 'stream accessing'>
	aString isEmpty ifTrue: [^self].
	connectionStream nextPutAll: (self encode: aString)
    ]

    nl [
	<category: 'stream accessing'>
	| conv |
	conv := self lineEndConvention.
	conv = LineEndCR ifTrue: [^connectionStream nextPut: Character cr].
	(conv = LineEndLF or: [conv = LineEndTransparent]) 
	    ifTrue: [^connectionStream nextPut: Character nl].
	conv = LineEndCRLF 
	    ifTrue: 
		[^connectionStream
		    nextPut: Character cr;
		    nextPut: Character nl].
	self error: 'Undefined line-end convention'
    ]

    species [
	<category: 'stream accessing'>
	^connectionStream species
    ]

    upTo: aCharacter [
	<category: 'stream accessing'>
	| byte |
	aCharacter = Character cr ifTrue: [^self nextLine].
	byte := self encode: (String with: aCharacter).
	byte size = 1 
	    ifTrue: [^self decode: (connectionStream upTo: byte)]
	    ifFalse: [^self decode: (connectionStream upToAll: byte)]
    ]

    update: aSymbol [
	"Dependents of ObjectMemory are sent update:
	 #returnFromSnapshot when a snapshot is started."

	<category: 'updating'>
	self liveAcrossSnapshot 
	    ifTrue: 
		[aSymbol == #returnFromSnapshot ifTrue: [self close]
		"(aSymbol == #aboutToSnapshot or: [aSymbol == #aboutToQuit])
		 ifTrue: [self close]."].
	super update: aSymbol
    ]
]



Object subclass: NetResponse [
    | status statusMessage |
    
    <category: 'NetClients-Framework'>
    <comment: '
Copyright (c) Kazuki Yasumatsu, 1995. All rights reserved.
'>

    NetResponse class >> fromClient: aClient [
	<category: 'instance creation'>
	| response |
	response := self new.
	response parseResponse: aClient.
	^response
    ]

    status [
	<category: 'accessing'>
	^status
    ]

    status: anInteger [
	<category: 'accessing'>
	status := anInteger
    ]

    statusArray [
	<category: 'accessing'>
	| n array |
	status == nil ifTrue: [n := 0] ifFalse: [n := status].
	array := Array new: 3.
	array at: 1 put: n // 100.
	n := n - (n // 100 * 100).
	array at: 2 put: n // 10.
	n := n - (n // 10 * 10).
	array at: 3 put: n.
	^array
    ]

    statusMessage [
	<category: 'accessing'>
	^statusMessage
    ]

    statusMessage: aString [
	<category: 'accessing'>
	statusMessage := aString
    ]

    parseResponse: aClient [
	<category: 'parsing'>
	self parseStatusLine: aClient
    ]

    printOn: aStream [
	<category: 'printing'>
	self printStatusOn: aStream
    ]

    printStatusOn: aStream [
	<category: 'printing'>
	status notNil 
	    ifTrue: 
		[aStream
		    print: status;
		    space].
	statusMessage notNil ifTrue: [aStream nextPutAll: statusMessage]
    ]

    parseStatusLine: aClient [
	<category: 'private'>
	| stream |
	statusMessage := nil.
	
	[stream := aClient nextLine readStream.
	status := Integer readFrom: stream.
	stream next = $-] 
		whileTrue: 
		    [statusMessage == nil 
			ifTrue: [statusMessage := stream upToEnd]
			ifFalse: 
			    [statusMessage := statusMessage , (String with: Character cr) 
					, stream upToEnd]].
	stream skipSeparators.
	statusMessage == nil 
	    ifTrue: [statusMessage := stream upToEnd]
	    ifFalse: 
		[statusMessage := statusMessage , (String with: Character cr) 
			    , stream upToEnd]
    ]
]



Object subclass: Reporter [
    | totalByte readByte |
    
    <category: 'NetClients-URIResolver'>
    <comment: nil>

    readByte [
	<category: 'accessing'>
	^readByte
    ]

    readByte: aValue [
	<category: 'accessing'>
	readByte := aValue
    ]

    endTransfer [
	<category: 'api'>
	^self
    ]

    startTransfer [
	<category: 'api'>
	^self
    ]

    statusString: aString [
	<category: 'api'>
	^self
    ]

    totalByte [
	<category: 'api'>
	^totalByte
    ]

    totalByte: aNumber [
	<category: 'api'>
	totalByte := aNumber
    ]
]



Reporter subclass: PluggableReporter [
    | startTime currentTime totalByte readByte statusBlock |
    
    <category: 'NetClients-Framework'>
    <comment: '
Copyright (c) Kazuki Yasumatsu, 1995. All rights reserved.
'>

    PluggableReporter class >> statusBlock: aBlock [
	<category: 'instance creation'>
	^self new statusBlock: aBlock
    ]

    endTransfer [
	<category: 'accessing'>
	self statusString: 'Transferring: Done.'
    ]

    readByte: anInteger [
	<category: 'accessing'>
	readByte := readByte + anInteger.
	currentTime := Time millisecondClockValue.
	self statusString: self progressStatusString
    ]

    startTransfer [
	<category: 'accessing'>
	readByte := 0.
	startTime := currentTime := Time millisecondClockValue.
	self statusString: 'Transferring: Start.'
    ]

    statusString: statusString [
	<category: 'accessing'>
	statusBlock isNil ifTrue: [^self].
	statusBlock value: statusString
    ]

    totalByte: anInteger [
	<category: 'accessing'>
	totalByte := anInteger
    ]

    progressStatusString [
	<category: 'private'>
	| stream |
	stream := WriteStream on: (String new: 128).
	stream print: readByte.
	totalByte == nil 
	    ifFalse: 
		[stream
		    nextPut: $/;
		    print: totalByte].
	stream nextPutAll: ' bytes'.
	currentTime = startTime 
	    ifFalse: 
		[stream nextPutAll: ' ('.
		(readByte / (currentTime - startTime)) asFloat printOn: stream digits: 2.
		stream nextPutAll: ' Kbytes/sec)'].
	^stream contents
    ]

    statusBlock: aBlock [
	<category: 'private'>
	statusBlock := aBlock
    ]
]



Stream subclass: RemoveDotStream [
    | stream ch atStart |
    
    <category: 'NetClients-Framework'>
    <comment: nil>

    RemoveDotStream class >> on: aStream [
	<category: 'instance creation'>
	^self new initialize: aStream
    ]

    atEnd [
	<category: 'input'>
	ch isNil ifFalse: [^false].
	stream isNil ifTrue: [^true].
	stream atEnd 
	    ifTrue: 
		[stream := nil.
		^true].
	ch := stream next.
	(atStart and: [ch == $.]) 
	    ifFalse: 
		[atStart := ch == Character cr or: [ch == Character nl].
		^false].
	atStart := false.

	"Found dot at start of line, discard it"
	stream atEnd 
	    ifTrue: 
		[stream := ch := nil.
		^true].
	ch := stream next.

	"Found lonely dot, we are at end of stream"
	(ch == Character cr or: [ch == Character nl]) 
	    ifTrue: 
		[ch == Character cr ifTrue: [stream next].
		stream := ch := nil.
		^true].
	^false
    ]

    next [
	<category: 'input'>
	| answer |
	self atEnd ifTrue: [self error: 'end of stream reached'].
	answer := ch.
	ch := nil.
	^answer
    ]

    peek [
	<category: 'input'>
	self atEnd ifTrue: [^nil].
	^ch
    ]

    peekFor: aCharacter [
	<category: 'input'>
	self atEnd ifTrue: [^false].
	ch == aCharacter 
	    ifTrue: 
		[self next.
		^true].
	^false
    ]

    initialize: aStream [
	<category: 'private'>
	stream := aStream.
	atStart := true.
	self atEnd
    ]

    species [
	<category: 'private'>
	^stream species
    ]
]



Stream subclass: PrependDotStream [
    | stream atStart |
    
    <category: 'NetClients-Framework'>
    <comment: 'A PrependDotStream removes a dot to each line starting with a dot, and
ends when its input has a lonely dot.'>

    PrependDotStream class >> to: aStream [
	<category: 'instance creation'>
	^self new initialize: aStream
    ]

    flush [
	<category: 'output'>
	atStart ifFalse: [self nl].
	stream
	    nextPut: $.;
	    nl
    ]

    nextPut: aChar [
	<category: 'output'>
	(atStart and: [aChar == $.]) ifTrue: [stream nextPut: aChar].
	stream nextPut: aChar.
	atStart := aChar == Character nl
    ]

    initialize: aStream [
	<category: 'private'>
	stream := aStream.
	atStart := true
    ]

    species [
	<category: 'private'>
	^stream species
    ]
]



Stream subclass: CrLfStream [
    | stream readStatus eatLf |
    
    <category: 'NetClients-Framework'>
    <comment: 'A CrLfStream acts as a pipe which transforms incoming data into LF-separated
lines, and outgoing data into CRLF-separated lines.'>

    Lf := nil.
    Cr := nil.

    CrLfStream class >> on: aStream [
	<category: 'instance creation'>
	Cr := Character cr.
	Lf := Character nl.
	^self new on: aStream
    ]

    on: aStream [
	<category: 'initializing'>
	stream := aStream.
	eatLf := false.
	readStatus := #none
    ]

    atEnd [
	<category: 'stream'>
	^stream atEnd and: [readStatus == #none]
    ]

    close [
	<category: 'stream'>
	stream close
    ]

    flush [
	<category: 'stream'>
	stream flush
    ]

    next [
	<category: 'stream'>
	| result |
	readStatus == #none 
	    ifFalse: 
		[readStatus == Cr ifTrue: [stream peekFor: Lf].
		readStatus := #none.
		^Lf].
	result := stream next.
	^(result == Cr or: [result == Lf]) 
	    ifTrue: 
		[readStatus := result.
		Cr]
	    ifFalse: [result]
    ]

    nextLine [
	<category: 'stream'>
	| line |
	line := self upTo: Cr.
	self next.	"Eat line feed"
	^line
    ]

    nextPut: aCharacter [
	<category: 'stream'>
	eatLf 
	    ifTrue: 
		[eatLf := false.
		aCharacter == Lf ifTrue: [^self]]
	    ifFalse: 
		[aCharacter == Lf 
		    ifTrue: 
			[stream
			    nextPut: Cr;
			    nextPut: Lf;
			    flush.
			^self]].
	stream nextPut: aCharacter.
	aCharacter == Cr 
	    ifTrue: 
		[stream
		    nextPut: Lf;
		    flush.
		eatLf := true]
    ]

    peek [
	<category: 'stream'>
	| result |
	readStatus == #none 
	    ifFalse: 
		[readStatus == Cr ifTrue: [stream peekFor: Lf].
		readStatus := Lf.	"peek for LF just once"
		^Lf].
	result := stream peek.
	^result == Lf ifTrue: [Cr] ifFalse: [result]
    ]

    peekFor: aCharacter [
	<category: 'stream'>
	| result success |
	readStatus == #none 
	    ifFalse: 
		[readStatus == Cr ifTrue: [stream peekFor: Lf].
		success := aCharacter == Lf.
		readStatus := success ifTrue: [#none] ifFalse: [Lf].	"peek for LF just once"
		^success].
	result := stream peek.
	(result == Cr or: [result == Lf]) 
	    ifTrue: 
		[success := aCharacter == Cr.
		success ifTrue: [readStatus := stream next].
		^success].
	success := aCharacter == result.
	success ifTrue: [stream next].
	^success
    ]

    species [
	<category: 'stream'>
	^stream species
    ]

    stream [
	<category: 'stream'>
	^stream
    ]
]



Error subclass: NetClientError [
    
    <category: 'NetClients-Framework'>
    <comment: nil>
]



NetClientError subclass: ConnectionFailedError [
    
    <category: 'NetClients-Framework'>
    <comment: nil>
]



NetClientError subclass: ConnectionClosedError [
    
    <category: 'NetClients-Framework'>
    <comment: nil>
]



NetClientError subclass: ProtocolError [
    
    <category: 'NetClients-Framework'>
    <comment: nil>

    isResumable [
	<category: 'exception handling'>
	^true
    ]
]



NetClientError subclass: LoginIncorrectError [
    
    <category: 'NetClients-Framework'>
    <comment: nil>
]

