# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000-2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import harmonicintervall, melodicintervall, singintervall
import identifyscale, bpm, idbyname, harmonicprogressiondictation
import twelvetone, singchord, idtone, compareintervalls, intervall
import chord, dictation, polyrhythm, rhythm

import gtk, GDK, os, gnome.ui, const, string
from i18n import _
from htmlwidget import HtmlWidget
import utils, history
from configwindow import ConfigWindow
import mpd, gu, soundcard, cfg, configureoutput
import errno

class DocViewer(gtk.GtkVBox):
    def __init__(self, activate_cb):
        gtk.GtkVBox.__init__(self)
        self.m_htmlwidget = HtmlWidget(activate_cb, self.on_anchor_track)
        self.m_htmlwidget.show_all()
        self.pack_start(self.m_htmlwidget)
        self.set_usize(300, 300)
        self.read_docfile = self.m_htmlwidget.read_docfile
        self.on_key_press_event = self.m_htmlwidget.on_key_press_event
        self.m_statusbar = gtk.GtkStatusbar()
        self.m_statusbar.show()
        self.pack_start(self.m_statusbar, gtk.FALSE)
        self.m_link_under_mouse = None
        self.source = self.m_htmlwidget.source
    def on_anchor_track(self, url):
        if url:
            # remove newlines in url because it make the window resize
            s = string.replace(url, "\n", "")
            # we have to use m_link_under_mouse because XmHTML does not
            # always emit the correct signals. Try to go without this
            # when we switch to GtkHTML
            if self.m_link_under_mouse:
                self.m_statusbar.pop(1)
            self.m_statusbar.push(1, s)
            self.m_link_under_mouse = 1
        else:
            self.m_statusbar.pop(1)
            self.m_link_under_mouse = None


class MusicViewerWindow(gtk.GtkDialog):
    def __init__(self, app):
        gtk.GtkDialog.__init__(self)
        self.m_app = app
        self.set_usize(400, 200)
        self.g_music_displayer = mpd.MusicDisplayer()
        self.vbox.pack_start(self.g_music_displayer)
        gu.bButton(self.action_area, _("Close"), self.m_app.close_musicviewer)
        self.connect('destroy', self.m_app.close_musicviewer)
        self.show_all()
    def display_music(self, music):
        fontsize = cfg.get_int('config/feta_font_size=20')
        self.g_music_displayer.display(music, fontsize)

class MainWin(gnome.ui.GnomeApp, cfg.ConfigUtils):
    def __init__(self, app):
        gnome.ui.GnomeApp.__init__(self, 'solfege%i.%i' %
             (configureoutput.MAJOR_VERSION,
              configureoutput.MINOR_VERSION), 'Solfege')
        cfg.ConfigUtils.__dict__['__init__'](self, 'mainwin')
        if self.get_bool('gui/mainwin_user_resizeable'):
            self.set_policy(0, 1, 0)
        else:
            self.set_policy(0, 0, 1)
        self.m_app = app
        self.connect('destroy', lambda w, s=self: s.m_app.quit_program())
        self.connect('key_press_event', self.on_key_press_event)
        self.main_box = gtk.GtkVBox()
        self.main_box.show()
        self.set_contents(self.main_box)
        self.m_exercise = None
        self.m_viewer = None
        self.box_dict = {}
        self.g_config_window = None
        self.m_history = history.History()
        self.m_key_bindings = {'history_back_ak': self.history_back,
                               'history_forward_ak': self.history_forward,
                               'history_reload_ak': self.history_reload}
        self.setup_menubar()
        self.setup_toolbars()
        if not self.get_bool('gui/development_menu_visible'):
            self.get_dock_item_by_name("Menubar").children()[0].children()[1].hide()
    def post_constructor(self):
        if not self.get_bool("gui/navbar_visible"):
            self.get_dock_item_by_name('backandforward').hide()
        if not self.get_bool("gui/toolbar_visible"):
            self.get_dock_item_by_name('main-toolbar').hide()
    def activate_exercise(self, href):
        protocol, action, exname, collection, lessonfile, config = \
           utils.parse_url(href)
        assert protocol == 'solfege'
        if self.m_viewer:
            if self.m_viewer != 'docviewer':
                self.box_dict[self.m_viewer].on_end_practise()
            self.box_dict[self.m_viewer].hide()
        self.box_dict[exname].show()
        self.box_dict[exname].on_start_practise()
        self.m_viewer = exname
        if action in ['practise', 'config', 'statistics']:
            self.box_dict[self.m_viewer].g_notebook.set_page(
               ['practise', 'config', 'statistics'].index(action))
        self.set_title("Solfege - " + self.box_dict[exname].get_pretty_name())
        self.m_history.add(href)
    def _show_docviewer(self):
        if not self.box_dict.has_key('docviewer'):
            self.box_dict['docviewer'] = DocViewer(self.m_app.handle_href)
            self.main_box.pack_start(self.box_dict['docviewer'])
        if self.m_viewer and (self.m_viewer != 'docviewer'):
                self.box_dict[self.m_viewer].hide()
        self.m_viewer = 'docviewer'
        self.box_dict['docviewer'].show()
    def display_docfile(self, fn):
        self._show_docviewer()
        self.box_dict['docviewer'].read_docfile(fn)
        self.set_title("Solfege - %s" % fn)
        self.m_history.add(fn)
    def display_html(self, html):
        self._show_docviewer()
        self.box_dict['docviewer'].source(html)
    def history_back(self, widget=None):
        self.m_history.back()
        self.m_history.lock()
        self.m_app.handle_href(self.m_history.get_current())
        self.m_history.unlock()
    def history_forward(self, widget=None):
        self.m_history.forward()
        self.m_history.lock()
        self.m_app.handle_href(self.m_history.get_current())
        self.m_history.unlock()
    def history_reload(self, widget=None):
        self.m_history.lock()
        self.m_app.handle_href(self.m_history.get_current())
        self.m_history.unlock()
    def initialise_exercise(self, teacher):
        assert not self.box_dict.has_key(teacher.m_exname)
        ed = {const.HARMONIC_INTERVALL: harmonicintervall.Gui,
              const.MELODIC_INTERVALL: melodicintervall.Gui,
              const.SING_INTERVALL: singintervall.Gui,
              const.IDENTIFY_SCALE: identifyscale.Gui,
              const.IDENTIFY_BPM: bpm.Gui,
              const.ID_BY_NAME: idbyname.Gui,
              const.HARMONIC_PROGRESSION_DICTATION: harmonicprogressiondictation.Gui,
              const.TWELVE_TONE: twelvetone.Gui,
              const.SING_CHORD: singchord.Gui,
              const.ID_TONE: idtone.Gui,
              const.COMPARE_INTERVALLS: compareintervalls.Gui,
              const.INTERVALL: intervall.Gui,
              const.CHORD: chord.Gui,
              const.DICTATION: dictation.Gui,
              const.POLYRHYTHM: polyrhythm.Gui,
              const.RHYTHM: rhythm.Gui}
        self.box_dict[teacher.m_exname] = ed[teacher.m_exname](teacher, self)
        self.main_box.pack_start(self.box_dict[teacher.m_exname])
    def on_key_press_event(self, widget, event):
        if event.type == GDK.KEY_PRESS:
            for s in self.m_key_bindings.keys():
                if (event.state, event.keyval) \
                         == utils.parse_key_string(self.get_string(s)):
                    self.m_key_bindings[s]()
                    return 1
        self.box_dict[self.m_viewer].on_key_press_event(widget, event)
    def open_preferences_window(self, widget=None):
        """ugh I am not really sure if this should go into SolfegeApp
        """
        if not self.g_config_window:
            self.g_config_window = ConfigWindow(self.m_app)
            self.g_config_window.set_parent(self)
            self.g_config_window.show()
        else:
            self.g_config_window.show()
    def setup_menubar(self):
        from gnome.ui import *
        file_menu = []
        for e in const.exercises:
            file_menu.append(
             (APP_UI_ITEM, const.exercisedata[e][1], None,
              lambda _o, self=self, e=e: self.m_app.handle_href("solfege:/"+e),
              None, APP_PIXMAP_NONE, None,
              const.exercisedata[e][3], const.exercisedata[e][4]))
        file_menu = file_menu + [
            UIINFO_SEPARATOR,
            UIINFO_MENU_PREFERENCES_ITEM(self.open_preferences_window),
            UIINFO_MENU_EXIT_ITEM(lambda w, s=self: s.m_app.quit_program()),
            ]
        help_menu = [
                    (APP_UI_ITEM, _("_Help on current exercise"), None,
                     lambda w, self=self: self.m_app.please_help_me(), 
                     None, APP_PIXMAP_NONE, None, GDK.F1, 0),
                    (APP_UI_ITEM, _("Welcome"), None,
                    lambda f, self=self: self.m_app.handle_href('welcome.html'),
                    None, APP_PIXMAP_NONE, None, 0, 0),
                    (APP_UI_ITEM, _("All help files"), None,
                    lambda f, self=self: self.m_app.handle_href('index.html'),
                    None, APP_PIXMAP_NONE, None, 0, 0),
                    (APP_UI_ITEM, _("All installed lesson files"), None,
                    lambda w, self=self: self.m_app.handle_href('solfege:all-lessonfiles'),
                    #lambda w, self=self: self.display_html(self.m_app.lessonfile_manager.m_htmldoc),
                    None, APP_PIXMAP_NONE, None, 0, 0),
                    UIINFO_SEPARATOR,
                    (APP_UI_ITEM, _("Copyright notice"), None,
                    lambda f, self=self: self.m_app.handle_href('copyright.html'),
                    None, APP_PIXMAP_NONE, None, 0, 0),
                    (APP_UI_ITEM, _("Credits"), None,
                    lambda f, self=self: self.m_app.handle_href('AUTHORS.html'),
                    None, APP_PIXMAP_NONE, None, 0, 0),
                    (APP_UI_ITEM, _("Mailinglists, web page etc."), None,
                    lambda f, self=self: self.m_app.handle_href('online-resources.html'),
                    None, APP_PIXMAP_NONE, None, 0, 0),
                    (APP_UI_ITEM, _("Reporting bugs"), None,
                    lambda f, self=self: self.m_app.handle_href('bug-reporting.html'),
                    None, APP_PIXMAP_NONE, None, 0, 0),
                    UIINFO_MENU_ABOUT_ITEM(self.m_app.show_about_window, None),
                    ]
        development_menu =  [
            (APP_UI_ITEM, _("This menu contains\nmostly useless stuff"),
             None, None, None, APP_PIXMAP_NONE, None, 0, 0),
             UIINFO_SEPARATOR,
            (APP_UI_ITEM, _("_Tempo"), None,
             lambda _o, s=self: s.m_app.handle_href('solfege:/identify-bpm'),
             None, APP_PIXMAP_NONE, None, 0, 0),
            (APP_UI_ITEM, _("Intervall"), None,
             lambda _o, s=self: s.m_app.handle_href('solfege:/intervalls'),
             None, APP_PIXMAP_NONE, None, 0, 0),
            (APP_UI_ITEM, _("Rhythm"), None,
             lambda _o, s=self: s.m_app.handle_href('solfege:/polyrhythm'),
             None, APP_PIXMAP_NONE, None, 0, 0),
            (APP_UI_ITEM, const.exercisedata[
                            const.HARMONIC_PROGRESSION_DICTATION][1], None,
             lambda _o, s=self: s.m_app.handle_href('solfege:/harmonic-progression-dictation'),
             None, APP_PIXMAP_NONE, None, 0, 0),

            UIINFO_ITEM_STOCK(_("Experiment"), None,
                              self.experiment, STOCK_MENU_EXEC),
                 ]
        self.m_menuinfo = menu = [
            UIINFO_SUBTREE(_("_File"), file_menu),
            UIINFO_SUBTREE(_("_Development"), development_menu),
            UIINFO_SUBTREE(_("_Help"), help_menu),
            ]
        self.create_menus(menu)
    def experiment(self, w=None):
        pass
    def setup_toolbars(self):
        self.g_toolbar = gtk.GtkToolbar(gtk.ORIENTATION_HORIZONTAL,
                                        gtk.TOOLBAR_ICONS)
        self.g_toolbar.set_style({'icons': gtk.TOOLBAR_ICONS,
                                  'text': gtk.TOOLBAR_TEXT,
                                  'both': gtk.TOOLBAR_BOTH}
               [self.get_string('gui/toolbar_style')])
        bg = self.get_style().bg[gtk.STATE_NORMAL]
        for i in const.exercises:
            pix, mask = gtk.create_pixmap_from_xpm(self, None, 
               os.path.join('xpm', const.exercisedata[i][2]))
            self.g_toolbar.append_item(const.exercisedata[i][0],
                    const.exercisedata[i][0], i, gtk.GtkPixmap(pix, mask),
                    lambda id, self=self, i=i: self.m_app.handle_href("solfege:/"+i))
        self.add_toolbar(self.g_toolbar, "main-toolbar",
                         gnome.uiconsts.DOCK_ITEM_BEH_NORMAL, 
                         gnome.uiconsts.DOCK_TOP, 1, 1, 0)
        self.g_navbar = gtk.GtkToolbar(gtk.ORIENTATION_HORIZONTAL,
                                       gtk.TOOLBAR_ICONS)
        self.g_navbar.append_item(_("Back"), None, None, 
            gnome.ui.GnomeStock(gnome.ui.STOCK_PIXMAP_BACK), self.history_back)
        self.g_navbar.append_item(_("Forward"), None, None, 
              gnome.ui.GnomeStock(gnome.ui.STOCK_PIXMAP_FORWARD), 
                                  self.history_forward)
        self.g_navbar.set_style({'icons': gtk.TOOLBAR_ICONS,
                                 'text': gtk.TOOLBAR_TEXT,
                                 'both': gtk.TOOLBAR_BOTH}
               [self.get_string('gui/navbar_style')])
        self.add_toolbar(self.g_navbar, 'backandforward',
                         gnome.uiconsts.DOCK_ITEM_BEH_NORMAL, 
                         gnome.uiconsts.DOCK_TOP, 1, 0, 0)
    def display_sound_config_error_message(self):
        #ugh we should do this properly in the 1.1.x releases
        if not soundcard.synth:
            cfg.clean_key("sound/type")
        import __main__
        if __main__.sound_init_exception == None:
            gnome.ui.GnomeErrorDialog(
"""You should configure sound from the 'Sound' page
of the preferences window.
""")
        elif __main__.sound_init_exception.errno == errno.EACCES:
            gnome.ui.GnomeErrorDialog(
"""The sound init failed: %s
The errno EACCES indicates that you don't have write
permission to the device.""" % __main__.sound_init_exception)
        elif __main__.sound_init_exception.errno == errno.EBUSY:
            gnome.ui.GnomeErrorDialog(
"""The sound init failed: %s
It seems like some other program is using the device. You
should try to quit that other program and restart Solfege."""
 % __main__.sound_init_exception)
        else:
            gnome.ui.GnomeErrorDialog(
"""The sound init failed: %s
It seems like your kernel don't have the correct
sound modules installed, or that they are wrongly
configured. You can check the 'Sound' page in the
preferences window, but I don't think anything there
can help you. (This program need better sound init
code to handle situations like this.)""" 
 % __main__.sound_init_exception)


