# GNU Solfege - ear training for GNOME
# Copyright (C) 2000, 2001, 2002, 2003, 2004, 2005  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, gnome.ui, gobject
import abstract, const, gu, mpd, soundcard
import random, os
import cfg
import lessonfile

RHYTHMS = ("c4", "c8 c8", "c16 c16 c16 c16", "c8 c16 c16",
           "c16 c16 c8", "c16 c8 c16", "c8. c16", "c16 c8.",
           "r4", "r8c8", "r8 c16 c16", "r16 c16 c8", "r16c8c16",
           "r16 c16 c16 c16", "r8 r16 c16", "r16 c8.",
           "c12 c12 c12", "r12 c12 c12",
           "c12 r12 c12", "c12 c12 r12", "r12 r12 c12", "r12 c12 r12",
           )


class Teacher(abstract.LessonbasedTeacher):
    OK = 0
    ERR_PICKY = 1
    ERR_NO_ELEMS = 2
    def __init__(self, exname, app):
        abstract.Teacher.__init__(self, exname, app)
        self.lessonfileclass = lessonfile.IdByNameLessonfile
    def new_question(self):
        """returns:
               Teacher.ERR_PICKY : if the question is not yet solved and the
                                   teacher is picky (== you have to solve the
                                   question before a new is asked).
               Teacher.OK : if a new question was created.
               Teacher.ERR_NO_ELEMS : if no elements are set to be practised.
        """
        if self.m_timeout_handle:
            gobject.source_remove(self.m_timeout_handle)
            self.m_timeout_handle = None

        if self.get_bool('config/picky_on_new_question') \
                 and self.q_status in [const.QSTATUS_NEW, const.QSTATUS_WRONG]:
            return Teacher.ERR_PICKY

        self.q_status = const.QSTATUS_NO

        norest_v = []
        v = []
        for x in range(len(RHYTHMS)):
            if self.get_bool("rhythm_element_%i" % x):
                if not (RHYTHMS[x][0] == "r"
                        and self.get_bool("not_start_with_rest")):
                    norest_v.append(x)
                v.append(x)
        if not v:
            return Teacher.ERR_NO_ELEMS
        if not norest_v:
            return Teacher.ERR_NO_ELEMS
        self.m_question = [random.choice(norest_v)]
        for x in range(1, self.get_int("num_beats")):
            self.m_question.append(random.choice(v))
        self.q_status = const.QSTATUS_NEW
        return Teacher.OK
    def get_music_string(self):
        s = ""
        for x in range(self.get_int("count_in")):
            s = s + "d4 "
        for k in self.m_question:
            s = s + RHYTHMS[k] + " "
        return r"\staff{%s}" % s
    def play_question(self):
        if self.q_status == const.QSTATUS_NO:
            return
        score = mpd.parser.parse_to_score_object(self.get_music_string())
        track = score.get_midi_events_with_channel(9, cfg.get_int('config/preferred_instrument_velocity'))[0]
        track.prepend_bpm(self.get_int("bpm"))
        track.replace_note(mpd.notename_to_int("c"),
                           self.get_int("rhythm_perc"), 9)
        track.replace_note(mpd.notename_to_int("d"),
                           self.get_int("countin_perc"), 9)
        soundcard.synth.play_track(track)
    def guess_answer(self, a):
        assert self.q_status in [const.QSTATUS_NEW, const.QSTATUS_WRONG]
        v = []
        for idx in range(len(self.m_question)):
            v.append(self.m_question[idx] == a[idx])
        if filter(lambda a: a == 0, v) == []:
            self.q_status = const.QSTATUS_SOLVED
            self.maybe_auto_new_question()
            self.m_app.play_happy_sound()
            return 1
        else:
            self.q_status = const.QSTATUS_WRONG
            self.m_app.play_sad_sound()

def create_png_image(fn):
    """
    Create an image by loading a png file from graphics dir
    """
    im = gtk.Image()
    im.set_from_file(os.path.join('graphics', fn)+'.png')
    im.show()
    return im

def create_rhythm_image(rhythm):
    """
    rhythm : a string like 'c8 c8' or 'c8 c16 c16'
    The image returned is shown.
    """
    im = gtk.Image()
    im.set_from_stock('solfege-rhythm-%s' % ''.join(rhythm.split()),
                      gtk.ICON_SIZE_LARGE_TOOLBAR)
    im.show()
    return im

class RhythmViewer(gtk.Frame):
    def __init__(self, parent):
        gtk.Frame.__init__(self)
        self.set_shadow_type(gtk.SHADOW_IN)
        self.g_parent = parent
        self.g_box = gtk.HBox()
        self.g_box.show()
        self.g_box.set_spacing(gnome.ui.PAD_SMALL)
        self.g_box.set_border_width(gnome.ui.PAD)
        self.add(self.g_box)
        self.m_data = []
        # the number of rhythm elements the viewer is supposed to show
        self.m_num_beats = 0
        self.g_face = None
        self.__timeout = None
    def set_num_beats(self, i):
        self.m_num_beats = i
    def clear(self):
        for child in self.g_box.get_children():
            child.destroy()
        self.m_data = []
    def create_holders(self):
        """
        create those |__| that represents one beat
        """
        if self.__timeout:
            gobject.source_remove(self.__timeout)
        self.clear()
        for x in range(self.m_num_beats):
            self.g_box.pack_start(create_png_image('holder'), False)
        self.m_data = []
    def clear_wrong_part(self):
        """When the user have answered the question, this method is used
        to clear all but the first correct elements."""
        # this assert is always true because if there is no rhythm element,
        # then there is a rhythm holder ( |__| )
        assert self.m_num_beats == len(self.g_parent.m_t.m_question)
        self.g_face.destroy()
        self.g_face = None
        for n in range(self.m_num_beats):
            if self.m_data[n] != self.g_parent.m_t.m_question[n]:
                break
        for x in range(n, len(self.g_box.get_children())):
            self.g_box.get_children()[n].destroy()
        self.m_data = self.m_data[:n]
        for x in range(n, self.m_num_beats):
            self.g_box.pack_start(create_png_image('holder'), False)
    def add_rhythm_element(self, i):
        assert len(self.m_data) <= self.m_num_beats
        if len(self.g_box.get_children()) >= self.m_num_beats:
            self.g_box.get_children()[self.m_num_beats-1].destroy()
        vbox = gtk.VBox()
        vbox.show()
        im = create_rhythm_image(RHYTHMS[i])
        vbox.pack_start(im)
        vbox.pack_start(create_png_image('rhythm-wrong'), False, False)
        vbox.get_children()[-1].hide()
        self.g_box.pack_start(vbox, False)
        self.g_box.reorder_child(vbox, len(self.m_data))
        self.m_data.append(i)
    def backspace(self):
        if len(self.m_data) > 0:
            if self.g_face:
                self.g_box.get_children()[-2].destroy()
                self.g_face.destroy()
                self.g_face = None
            self.g_box.get_children()[len(self.m_data)-1].destroy()
            self.g_box.pack_start(create_png_image('holder'), False)
            del self.m_data[-1]
    def mark_wrong(self, idx):
        """
        Mark the rhythm elements that was wrong by putting the content of
        graphics/rhythm-wrong.png (normally a red line) under the element.
        """
        self.g_box.get_children()[idx].get_children()[1].show()
    def len(self):
        "return the number of rhythm elements currently viewed"
        return len(self.m_data)
    def sad_face(self):
        l = gu.HarmonicProgressionLabel(_("Wrong"))
        l.show()
        self.g_box.pack_start(l, False)
        self.g_face = gtk.EventBox()
        self.g_face.connect('button_press_event', self.on_sadface_event)
        self.g_face.show()
        im = gtk.Image()
        im.set_from_stock('solfege-sadface', gtk.ICON_SIZE_LARGE_TOOLBAR)
        im.show()
        self.g_face.add(im)
        self.g_box.pack_start(self.g_face, False)
    def happy_face(self):
        l = gu.HarmonicProgressionLabel(_("Correct"))
        l.show()
        self.g_box.pack_start(l, False)
        self.g_face = gtk.EventBox()
        self.g_face.connect('button_press_event', self.on_happyface_event)
        self.g_face.show()
        im = gtk.Image()
        im.set_from_stock('solfege-happyface', gtk.ICON_SIZE_LARGE_TOOLBAR)
        im.show()
        self.g_face.add(im)
        self.g_box.pack_start(self.g_face, False)
    def on_happyface_event(self, obj, event):
        if event.type == gtk.gdk.BUTTON_PRESS and event.button == 1:
            self.g_parent.new_question()
    def on_sadface_event(self, obj, event):
        if event.type == gtk.gdk.BUTTON_PRESS and event.button == 1:
            self.clear_wrong_part()
    def flash(self, s):
        self.clear()
        l = gtk.Label(s)
        l.set_name("Feedback")
        l.set_alignment(0.0, 0.5)
        l.show()
        self.g_box.pack_start(l, True, True)
        self.__timeout = gobject.timeout_add(const.NORMAL_WAIT, self.unflash)
    def unflash(self, *v):
        self.__timeout = None
        self.clear()


class Gui(abstract.Gui):
    def __init__(self, teacher, window):
        abstract.Gui.__init__(self, teacher, window)
        self.m_key_bindings = {'backspace_ak': self.on_backspace}
        hbox = gu.bHBox(self.practise_box, False)
        for i in range(0, 8):
            b = self.pngbutton(i)
            if i == 0:
                self.g_first_rhythm_button = b
            hbox.pack_start(b, False)
        hbox = gu.bHBox(self.practise_box, False)
        for i in range(8, 16):
            hbox.pack_start(self.pngbutton(i), False)
        hbox = gu.bHBox(self.practise_box, False)
        for i in range(16, 22):
            hbox.pack_start(self.pngbutton(i), False)
        #-------
        self.practise_box.pack_start(gtk.HBox(), False,
                                     padding=gnome.ui.PAD_SMALL)
        self.g_rhythm_viewer = RhythmViewer(self)
        #FIXME the value 52 is dependant on the theme used
        self.g_rhythm_viewer.set_size_request(-1, 52)
        self.g_rhythm_viewer.create_holders()
        self.practise_box.pack_start(self.g_rhythm_viewer, False)

        # action area
        self.g_new = gu.bButton(self.action_area, _("_New"), self.new_question)
        self.g_repeat = gu.bButton(self.action_area, _("_Repeat"), self.repeat_question)
        self.g_repeat.set_sensitive(False)
        self.g_show = gu.bButton(self.action_area, _("_Give up"), self.give_up)
        self.g_show.set_sensitive(False)

        self.g_backspace = gu.bButton(self.action_area, _("_Backspace"),
                     self.on_backspace)
        self.g_backspace.set_sensitive(False)
        self.practise_box.show_all()
        ##############
        # config_box #
        ##############
        self.g_element_frame = frame = gtk.Frame(_("Rhythms to use in question"))
        self.config_box.pack_start(frame, False)
        vbox = gtk.VBox()
        vbox.set_border_width(gnome.ui.PAD_SMALL)
        frame.add(vbox)
        hbox = gu.bHBox(vbox, False)
        for i in range(0, 8):
            hbox.pack_start(self.pngcheckbutton(i), False)
        hbox = gu.bHBox(vbox, False)
        for i in range(8, 16):
            hbox.pack_start(self.pngcheckbutton(i), False)
        hbox = gu.bHBox(vbox, False)
        for i in range(16, 22):
            hbox.pack_start(self.pngcheckbutton(i), False)
        #--------    
        self.config_box.pack_start(gtk.HBox(), False,
                                   padding=gnome.ui.PAD_SMALL)

        table = gtk.Table(2, 4, False)
        self.config_box.pack_start(table, False)
        ###
        label = gtk.Label(_("Number of beats in question:"))
        label.set_alignment(1.0, 0.5)
        table.attach(label, 0, 1, 0, 1, xpadding=gnome.ui.PAD_SMALL,
                      xoptions=gtk.FILL)
        table.attach(gu.nSpinButton(self.m_exname, "num_beats",
                     gtk.Adjustment(4, 1, 100, 1, 10)),
                     1, 2, 0, 1, xoptions=gtk.FILL)
        menu = gu.PercussionNameComboBoxEntry(self.m_exname, "rhythm_perc", "Side Stick")
        table.attach(menu, 2, 3, 0, 1, xoptions=gtk.FILL)
        label = gtk.Label(_("Count in before question:"))
        label.set_alignment(1.0, 0.5)
        table.attach(label, 0, 1, 1, 2, xpadding=gnome.ui.PAD_SMALL,
                     xoptions=gtk.FILL)
        table.attach(gu.nSpinButton(self.m_exname, "count_in",
                     gtk.Adjustment(2, 0, 10, 1, 10)),
                     1, 2, 1, 2, xoptions=gtk.FILL)
        menu = gu.PercussionNameComboBoxEntry(self.m_exname, "countin_perc", "Claves")
        table.attach(menu, 2, 3, 1, 2, xoptions=gtk.FILL)
        #-----
        self.config_box.pack_start(gtk.HBox(), False,
                                   padding=gnome.ui.PAD_SMALL)
        #------
        hbox = gu.bHBox(self.config_box, False)
        hbox.set_spacing(gnome.ui.PAD_SMALL)
        hbox.pack_start(gu.nCheckButton(self.m_exname,
                 "not_start_with_rest",
                 _("Don't start the question with a rest")), False)
        sep = gtk.VSeparator()
        hbox.pack_start(sep, False)
        hbox.pack_start(gtk.Label(_("Beats per minute:")), False)
        spin = gu.nSpinButton(self.m_exname, 'bpm',
                 gtk.Adjustment(60, 20, 240, 1, 10))
        hbox.pack_start(spin, False)
        self._add_auto_new_question_gui(self.config_box)
        self.config_box.show_all()
    def pngbutton(self, i):
        "used by the constructor"
        btn = gtk.Button()
        btn.add(create_rhythm_image(RHYTHMS[i]))
        btn.show()
        btn.connect('clicked', self.guess_element, i)
        return btn
    def pngcheckbutton(self, i):
        btn = gu.nCheckButton(self.m_exname, "rhythm_element_%i" % i, default_value=1)
        btn.add(create_rhythm_image(RHYTHMS[i]))
        btn.show()
        return btn
    def on_backspace(self, widget=None):
        if self.m_t.q_status == const.QSTATUS_SOLVED:
            return
        self.g_rhythm_viewer.backspace()
        if not self.g_rhythm_viewer.m_data:
            self.g_backspace.set_sensitive(False)
    def guess_element(self, sender, i):
        if self.m_t.q_status == const.QSTATUS_NO:
            self.g_rhythm_viewer.flash(_("Click 'New' to begin."))
            return
        if self.m_t.q_status == const.QSTATUS_SOLVED:
            return
        if self.g_rhythm_viewer.len() == len(self.m_t.m_question):
            self.g_rhythm_viewer.clear_wrong_part()
        self.g_rhythm_viewer.add_rhythm_element(i)
        if self.g_rhythm_viewer.len() == len(self.m_t.m_question):
            if self.m_t.guess_answer(self.g_rhythm_viewer.m_data):
                self.g_rhythm_viewer.happy_face()
                self.g_new.set_sensitive(True)
                self.g_new.grab_focus()
                self.g_backspace.set_sensitive(False)
                self.g_show.set_sensitive(False)
            else:
                v = []
                for idx in range(len(self.m_t.m_question)):
                    v.append(self.m_t.m_question[idx] == self.g_rhythm_viewer.m_data[idx])
                for x in range(len(v)):
                    if not v[x]:
                        self.g_rhythm_viewer.mark_wrong(x)
                self.g_rhythm_viewer.sad_face()
        else:
            self.g_backspace.set_sensitive(True)
    def new_question(self, widget=None):
        g = self.m_t.new_question()
        if g == Teacher.OK:
            self.g_first_rhythm_button.grab_focus()
            self.g_rhythm_viewer.set_num_beats(self.get_int('num_beats'))
            self.g_rhythm_viewer.create_holders()
            self.g_show.set_sensitive(True)
            self.g_repeat.set_sensitive(True)
            self.g_new.set_sensitive(
               not self.get_bool('config/picky_on_new_question'))
            self.m_t.play_question()
        elif g == Teacher.ERR_PICKY:
            self.g_rhythm_viewer.flash(_("You have to solve this question first."))
        else:
            assert g == Teacher.ERR_NO_ELEMS
            self.g_repeat.set_sensitive(False)
            self.g_rhythm_viewer.flash(_("You have to configure this exercise properly"))
    def repeat_question(self, *w):
        self.m_t.play_question()
        self.g_first_rhythm_button.grab_focus()
    def on_start_practise(self):
        if self.m_t.m_P.header.rhythm_elements:
            for n in range(22):
                self.m_t.set_bool('rhythm_element_%i' % n, 
                              n in self.m_t.m_P.header.rhythm_elements)
        self.m_t.m_custom_mode = bool(
            not self.m_t.m_P.header.rhythm_elements)
        if self.m_t.m_custom_mode:
            self.g_element_frame.show()
        else:
            self.g_element_frame.hide()
        if 'lesson_heading' in self.m_t.m_P.header:
            self.set_lesson_heading(self.m_t.m_P.header.lesson_heading)
        else:
            self.set_lesson_heading(_("Identify the rhythm"))
        for n in ('bpm',
                  'countin_perc', 'count_in',
                  'rhythm_perc', 'num_beats'):
            if n in self.m_t.m_P.header:
                self.m_t.set_int(n, self.m_t.m_P.header[n])
        if 'not_start_with_rest' in self.m_t.m_P.header:
            self.m_t.set_bool('not_start_with_rest', self.m_t.m_P.header.not_start_with_rest)
        self.g_rhythm_viewer.flash(_("Click 'New' to begin."))
        self.g_new.grab_focus()
    def on_end_practise(self):
        self.m_t.end_practise()
        self.g_new.set_sensitive(True)
        self.g_repeat.set_sensitive(False)
        self.g_show.set_sensitive(False)
        self.g_rhythm_viewer.create_holders()
    def give_up(self, widget=None):
        if self.m_t.q_status == const.QSTATUS_NO:
            return
        self.g_rhythm_viewer.clear()
        for i in self.m_t.m_question:
            self.g_rhythm_viewer.add_rhythm_element(i)
        self.m_t.q_status = const.QSTATUS_SOLVED
        self.g_new.set_sensitive(True)
        self.g_new.grab_focus()
        self.g_show.set_sensitive(False)
        self.g_backspace.set_sensitive(False)

