/*
** Copyright (C) 1999, 2000, Lorenzo Bettini <lorenzo.bettini@penteres.it>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/

// generators.cc

// generators and decorators

#include "textgen.h"
#include "decorators.h"
#include "tags.h"
#include "keys.h"
#include "linenogen.h"

TextGenerator *GlobalGenerator ;
TextGenerator *NormalGenerator ;
TextGenerator *KeywordGenerator ;
TextGenerator *CommentGenerator ;
TextGenerator *StringGenerator ;
TextGenerator *TypeGenerator ;
TextGenerator *NumberGenerator ;
TextGenerator *PreProcGenerator ;
TextGenerator *SymbolGenerator ;
TextGenerator *FunctionGenerator ;
TextGenerator *CBracketGenerator ;


static TextGenerator *createGenerator( TextGenerator *final_dec,
                                       char *key, CharTranslator *tr ) ;

template <class Decorator>
static void decorate_generator(TextGenerator *&decorated, 
                               CharTranslator *trans);

template <class Decorator>
static void decorate_generators(CharTranslator *trans);

void
createGenerators(CharTranslator *trans, int line_number_option)
{
  GlobalGenerator = new TextGenerator (trans);

  TextGenerator *final_decorator;

  if (line_number_option)
    {
      GlobalGenerator = new LineNumberDecorator (GlobalGenerator, trans);
      final_decorator = new TextGenerator (trans);
    }
  else
    {
      final_decorator = GlobalGenerator;
    }

  NormalGenerator = createGenerator (final_decorator, NORMAL, trans);
  KeywordGenerator = createGenerator (final_decorator, KEYWORD, trans ) ;
  CommentGenerator = createGenerator (final_decorator, COMMENT, trans ) ;
  StringGenerator = createGenerator (final_decorator, STRING, trans ) ;
  TypeGenerator = createGenerator (final_decorator, TYPE, trans ) ;
  NumberGenerator = createGenerator (final_decorator, NUMBER, trans ) ;
  PreProcGenerator = createGenerator (final_decorator, PREPROC, trans ) ;
  SymbolGenerator = createGenerator (final_decorator, SYMBOL, trans ) ;
  FunctionGenerator = createGenerator (final_decorator, FUNCTION, trans ) ;
  CBracketGenerator = createGenerator (final_decorator, CBRACKET, trans ) ;

  if (line_number_option)
    decorate_generators <LineNumberDecorator> (trans);
}

void
createGeneratorsForCSS(CharTranslator *trans, int line_number_option)
{
  GlobalGenerator = new TextGenerator (trans);

  TextGenerator *final_decorator;

  if (line_number_option)
    {
      GlobalGenerator = new LineNumberDecorator (GlobalGenerator, trans);
      final_decorator = new TextGenerator (trans);
    }
  else
    {
      final_decorator = GlobalGenerator;
    }

  NormalGenerator = new SpanDecorator (final_decorator, trans, NORMAL);
  KeywordGenerator = new SpanDecorator (final_decorator, trans, KEYWORD) ;
  CommentGenerator = new SpanDecorator (final_decorator, trans, COMMENT) ;
  StringGenerator = new SpanDecorator(final_decorator, trans, STRING) ;
  TypeGenerator = new SpanDecorator(final_decorator, trans, TYPE) ;
  NumberGenerator = new SpanDecorator(final_decorator, trans, NUMBER) ;
  PreProcGenerator = new SpanDecorator(final_decorator, trans, PREPROC) ;
  SymbolGenerator = new SpanDecorator(final_decorator, trans, SYMBOL) ;
  FunctionGenerator = new SpanDecorator(final_decorator, trans, FUNCTION) ;
  CBracketGenerator = new SpanDecorator(final_decorator, trans, CBRACKET) ;

  if (line_number_option)
    decorate_generators <LineNumberDecorator> (trans);
}

template <class Decorator>
void
decorate_generator(TextGenerator *&decorated, CharTranslator *trans)
{
  decorated = new Decorator (decorated, trans);
}

template <class Decorator>
void
decorate_generators(CharTranslator *trans)
{
  decorate_generator<Decorator> (NormalGenerator, trans);
  decorate_generator<Decorator> (KeywordGenerator, trans);
  decorate_generator<Decorator> (CommentGenerator, trans);
  decorate_generator<Decorator> (StringGenerator, trans);
  decorate_generator<Decorator> (TypeGenerator, trans);
  decorate_generator<Decorator> (NumberGenerator, trans);
  decorate_generator<Decorator> (PreProcGenerator, trans);
  decorate_generator<Decorator> (SymbolGenerator, trans);
  decorate_generator<Decorator> (FunctionGenerator, trans);
  decorate_generator<Decorator> (CBracketGenerator, trans);
}

#define GET_DECORATED( d ) ( d ? d : final_dec )

TextGenerator *
createGenerator( TextGenerator *final_dec, char *key, CharTranslator *tr )
{
  Tag *tag = getTag( key ) ;
  TextDecorator *dec = NULL ;

  if ( ! tag ) // no options
    return final_dec ;

  char * color = tag->GetColor() ;
  if ( color )
    dec = new ColorDecorator( final_dec, tr, color ) ;

  if ( tag->IsBold() )
    dec = new TagDecorator( GET_DECORATED( dec ), tr, BOLD_TAG ) ;
  if ( tag->IsItalic() )
    dec = new TagDecorator( GET_DECORATED( dec ), tr, ITALIC_TAG ) ;
  if ( tag->IsUnderline() )
    dec = new TagDecorator( GET_DECORATED( dec ), tr, UNDERLINE_TAG ) ;

  return GET_DECORATED( dec ) ;
  // There should be some options, but it's not ncessary ...
  // so this is just to be safe
}
