--
-- This file is part of TALER
-- Copyright (C) 2014--2023 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_coin_deposits(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(coin_deposit_serial_id INT8 GENERATED BY DEFAULT AS IDENTITY'
      ',batch_deposit_serial_id INT8 NOT NULL'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',coin_sig BYTEA NOT NULL CHECK (LENGTH(coin_sig)=64)'
      ',amount_with_fee taler_amount NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Coins which have been deposited with the respective per-coin signatures.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Link to information about the batch deposit this coin was used for'
    ,'batch_deposit_serial_id'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_coin_deposits(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_coin_deposit_serial_id_pkey'
    ' PRIMARY KEY (coin_deposit_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_unique_coin_sig'
    ' UNIQUE (coin_pub, coin_sig)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_batch '
    'ON ' || table_name || ' '
    '(batch_deposit_serial_id);'
  );
END
$$;


CREATE FUNCTION foreign_table_coin_deposits()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_deposits';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_batch_deposits_id'
    ' FOREIGN KEY (batch_deposit_serial_id) '
    ' REFERENCES batch_deposits (batch_deposit_serial_id) ON DELETE CASCADE'
  );
END
$$;


CREATE OR REPLACE FUNCTION coin_deposits_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
 VALUES
     (NEW.coin_pub
    ,'coin_deposits'
    ,NEW.coin_deposit_serial_id);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION coin_deposits_insert_trigger()
  IS 'Automatically generate coin history entry.';


CREATE FUNCTION master_table_coin_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER coin_deposits_on_insert
    AFTER INSERT
     ON coin_deposits
     FOR EACH ROW EXECUTE FUNCTION coin_deposits_insert_trigger();
END $$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('coin_deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('coin_deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('coin_deposits'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('coin_deposits'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE)
    ;
