/*
 This file is part of GNU Taler
 (C) 2021 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import fs from "fs";
import mustache from "mustache";
import { createDateToStringFunction, createDurationToStringFunction, createNonEmptyFunction } from "./utils.js"
import { exampleData as ShowOrderDetailsExamples } from "./pages/ShowOrderDetails.examples.js";
/**
 * This script will emulate what the merchant backend will do when being requested
 *
 */

const templateDirectory = process.argv[2];
const destDirectory = process.argv[3];

if (!templateDirectory || !destDirectory) {
  console.log("usage: render-mustache <source-directory> <dest-directory>");
  process.exit(1);
}

if (!fs.existsSync(destDirectory)) {
  fs.mkdirSync(destDirectory);
}

function fromCamelCaseName(name: string) {
  const result = name
    .replace(/^[a-z]/, (letter) => `${letter.toUpperCase()}`) //first letter lowercase
    .replace(/_[a-z]/g, (letter) => `${letter[1].toUpperCase()}`); //snake case
  return result;
}
/**
 * Load all the html files
 */
const templateFiles = fs.readdirSync(templateDirectory).filter((f) => /.html/.test(f));
const exampleByTemplate: Record<string, any> = {
  "show_order_details.en.html": ShowOrderDetailsExamples
}

templateFiles.forEach((templateFile) => {
  const html = fs.readFileSync(`${templateDirectory}/${templateFile}`, "utf8");

  const templateFileWithoutExt = templateFile.replace(".en.html", "");
  // const exampleFileName = `src/pages/${fromCamelCaseName(testName)}.examples`;
  // if (!fs.existsSync(`./${exampleFileName}.ts`)) {
  //   console.log(`- skipping ${testName}: no examples found`);
  //   return;
  // }
  // eslint-disable-next-line @typescript-eslint/no-var-requires
  // const pepe = `./${exampleFileName}.ts`
  // const { exampleData } = require(pepe);

  const exampleData = exampleByTemplate[templateFile]
  if (!exampleData) {
    console.log(`- skipping ${templateFile}: no examples found`);
    return;
  }
  const exampleNames = Object.keys(exampleData)
  console.log(`+ rendering ${templateFile}: ${exampleNames.length} examples`);
  exampleNames.forEach((exampleName) => {
    const example = exampleData[exampleName];

    //enhance the example with more information
    example.contract_terms_json = () => JSON.stringify(example.contract_terms);

    example.contract_terms.timestamp_str = createDateToStringFunction(example.contract_terms.timestamp)

    example.contract_terms.hasProducts = createNonEmptyFunction(example.contract_terms.products)
    example.contract_terms.hasAuditors = createNonEmptyFunction(example.contract_terms.auditors)
    example.contract_terms.hasExchanges = createNonEmptyFunction(example.contract_terms.exchanges)

    example.contract_terms.products.forEach((p: any) => {
      p.delivery_date_str = createDateToStringFunction(p.delivery_date)
      p.hasTaxes = createNonEmptyFunction(p.taxes)
    });

    example.contract_terms.has_delivery_info = () =>
      example.contract_terms.delivery_date ||
      example.contract_terms.delivery_location;

    example.contract_terms.delivery_date_str = createDateToStringFunction(example.contract_terms.delivery_date)
    example.contract_terms.pay_deadline_str = createDateToStringFunction(example.contract_terms.pay_deadline)
    example.contract_terms.wire_transfer_deadline_str = createDateToStringFunction(example.contract_terms.wire_transfer_deadline)

    example.contract_terms.refund_deadline_str = createDateToStringFunction(example.contract_terms.refund_deadline)
    example.contract_terms.auto_refund_str = createDurationToStringFunction(example.contract_terms.auto_refund)

    const output = mustache.render(html, example);

    fs.writeFileSync(
      `${destDirectory}/${templateFileWithoutExt}.${exampleName}.html`,
      output,
    );
  });
});
