/*
 Copyright 2019 Florian Dold

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 or implied. See the License for the specific language governing
 permissions and limitations under the License.
*/

import test, { ExecutionContext } from "ava";
import {
  structuredClone,
  structuredEncapsulate,
  structuredRevive,
} from "./structuredClone.js";

function checkClone(t: ExecutionContext, x: any): void {
  t.deepEqual(structuredClone(x), x);
}

test("structured clone", (t) => {
  checkClone(t, "foo");
  checkClone(t, [1, 2]);
  checkClone(t, true);
  checkClone(t, false);
  checkClone(t, { x1: "foo" });
  checkClone(t, { x1: true, x2: false });
  checkClone(t, new Date());
  checkClone(t, [new Date()]);
  checkClone(t, undefined);
  checkClone(t, [undefined]);

  t.throws(() => {
    structuredClone({ foo: () => {} });
  });

  t.throws(() => {
    structuredClone(Promise);
  });

  t.throws(() => {
    structuredClone(Promise.resolve());
  });
});

test("structured clone (array cycles)", (t) => {
  const obj1: any[] = [1, 2];
  obj1.push(obj1);
  const obj1Clone = structuredClone(obj1);
  t.is(obj1Clone, obj1Clone[2]);
});

test("structured clone (object cycles)", (t) => {
  const obj1: any = { a: 1, b: 2 };
  obj1.c = obj1;
  const obj1Clone = structuredClone(obj1);
  t.is(obj1Clone, obj1Clone.c);
});

test("encapsulate", (t) => {
  t.deepEqual(structuredEncapsulate(42), 42);
  t.deepEqual(structuredEncapsulate(true), true);
  t.deepEqual(structuredEncapsulate(false), false);
  t.deepEqual(structuredEncapsulate(null), null);

  t.deepEqual(structuredEncapsulate(undefined), { $: "undef" });
  t.deepEqual(structuredEncapsulate(42n), { $: "bigint", val: "42" });

  t.deepEqual(structuredEncapsulate(new Date(42)), { $: "date", val: 42 });

  t.deepEqual(structuredEncapsulate({ x: 42 }), { x: 42 });

  t.deepEqual(structuredEncapsulate({ $: "bla", x: 42 }), {
    $: "obj",
    val: { $: "bla", x: 42 },
  });

  const x = { foo: 42, bar: {} } as any;
  x.bar.baz = x;

  t.deepEqual(structuredEncapsulate(x), {
    foo: 42,
    bar: {
      baz: { $: "ref", d: 2, p: [] },
    },
  });
});

test("revive", (t) => {
  t.deepEqual(structuredRevive(42), 42);
  t.deepEqual(structuredRevive([1, 2, 3]), [1, 2, 3]);
  t.deepEqual(structuredRevive(true), true);
  t.deepEqual(structuredRevive(false), false);
  t.deepEqual(structuredRevive(null), null);
  t.deepEqual(structuredRevive({ $: "undef" }), undefined);
  t.deepEqual(structuredRevive({ x: { $: "undef" } }), { x: undefined });

  t.deepEqual(structuredRevive({ $: "date", val: 42}), new Date(42));

  {
    const x = { foo: 42, bar: {} } as any;
    x.bar.baz = x;

    const r = {
      foo: 42,
      bar: {
        baz: { $: "ref", d: 2, p: [] },
      },
    };

    t.deepEqual(structuredRevive(r), x);
  }
});
