/*
 * Electric(tm) VLSI Design System
 *
 * File: usrcom.c
 * User interface aid: parameter parsing routines
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "efunction.h"
#include "usr.h"
#include "usredtec.h"

/****************************** PARSING CODE ******************************/

#define	COMMANDS            01		/* user commands */
#define	MACROS              02		/* macro names */
#define	PORTS               04		/* ports names on a facet */
#define	EXPORTS            010		/* exported ports on the node */
#define	WINDOWS            020		/* window names */
#define	HIGHLIGHTS         040		/* highlight names */
#define	ARCS              0100		/* arc prototype names */
#define	CELLS             0200		/* cell names */
#define	FACETS            0400		/* facet names */
#define	PRIMS            01000		/* primitive node prototype names */
#define	PURES            02000		/* pure-layer node prototype names */
#define	LIBS             04000		/* library names */
#define	LAYERS          010000		/* layer names */
#define	POPUPS          020000		/* popup menu names */
#define	CONSTRAINTS     040000		/* constraint solver names */
#define	MBUTTONS       0100000		/* mouse button names */
#define	VIEWS          0200000		/* view names (including "unknown") */
#define	NODEFUN        0400000		/* node functions */
#define	ARCFUN        01000000		/* arc functions */
#define	EDTECPORT     02000000		/* port connections in technology edit */
#define	EDTECLAY      04000000		/* layer names from technology edit */
#define	EDTECARC     010000000		/* arc names from technology edit */
#define	EDTECNODE    020000000		/* node names from technology edit */
#define	EDTECTECH    040000000		/* editable technologies for tech edit */
#define	NETWORKS    0100000000		/* network names */
#define	WORDSET     0200000000		/* the word "SET-MINIMUM-SIZE" */
#define	WORDCLEAR   0400000000		/* the word "CLEAR-MINIMUM-SIZE" */
#define	WORDNODE   01000000000		/* the word "NODE" */
#define	WORDARC    02000000000		/* the word "ARC" */
#define	WORDNEXT   04000000000		/* the word "NEXT-PROTO" */
#define	WORDPREV  010000000000		/* the word "PREV-PROTO" */
#define	WORDTHIS  020000000000		/* the word "THIS-PROTO" */

static INTBIG us_posbits, us_posswitch, us_possearch;
static INTSML us_poslcommands;		/* COMMANDS */
static INTSML us_posmacrovar;		/* MACROS */
static PORTPROTO *us_posports;		/* PORTS */
static PORTEXPINST *us_posexports;	/* EXPORTS */
static WINDOWPART *us_poswindow;	/* WINDOWPARTS */
static INTSML us_poshigh;			/* HIGHLIGHTS */
static ARCPROTO *us_posarcs;		/* ARCS */
static CELL *us_poscells;			/* CELLS */
static NODEPROTO *us_posnodeprotos;	/* FACETS */
static NODEPROTO *us_posprims;		/* PRIMS */
static LIBRARY *us_poslibs;			/* LIBS */
static INTSML us_poslayers;			/* LAYERS */
static POPUPMENU *us_pospopupmenu;	/* POPUPS */
static INTSML us_posconstraint;		/* CONSTRAINTS */
static INTSML us_posmbuttons;		/* MBUTTONS */
static VIEW *us_posviews;			/* VIEWS */
static INTSML us_posfunction;		/* NODEFUN/ARCFUN */
static INTSML us_posspecial;		/* EDTECPORT */
static TECHNOLOGY *us_postechs;		/* EDTECTECH */

/* prototypes for local routines */
INTSML us_paramcommands(char*, COMCOMP*[], char);
INTSML us_topofgetproto(char**);
INTSML us_topofpures(char**);
INTSML us_topofnodefun(char**);
INTSML us_topofarcfun(char**);
INTSML us_topofedtecport(char**);
INTSML us_topofedtecslay(char**);
INTSML us_topofedtecclay(char**);
INTSML us_topofedtectech(char**);
INTSML us_topofeditor(char**);
char *us_nexteditor(void);
INTSML us_parambind(char*, COMCOMP*[], char);
INTSML us_parambindpop(char*, COMCOMP*[], char);
INTSML us_paramconstraintb(char*, COMCOMP*[], char);
INTSML us_paramconstrainta(char*, COMCOMP*[], char);
INTSML us_nextecho(char*, COMCOMP*[], char);
INTSML us_paramhelp(char*, COMCOMP*[], char);
INTSML us_parammenusbp(char*, COMCOMP*[], char);
INTSML us_paramportep(char*, COMCOMP*[], char);
INTSML paramsize(char*, COMCOMP*[], char);
INTSML us_nextsystem(char*, COMCOMP*[], char);
INTSML us_paramsedtecport(char*, COMCOMP*[], char);
INTSML us_nexttechnologyedlp(char*, COMCOMP*[], char);
INTSML paramtechnology(char*, COMCOMP*[], char);
INTSML us_paramtechnologyb(char*, COMCOMP*[], char);
INTSML us_paramtechnologya(char*, COMCOMP*[], char);
INTSML us_paramtellaidb(char*, COMCOMP*[], char);
INTSML us_paramtellaida(char*, COMCOMP*[], char);
INTSML us_paramvisiblelayers(char*, COMCOMP*[], char);

/* parsing routines for a command or macro */
INTSML us_topofcommands(char **c)
{
	us_poslcommands = 0;
	us_posmacrovar = 0;
	us_pospopupmenu = us_firstpopupmenu;
	us_posbits = COMMANDS | MACROS | POPUPS;
	return(1);
}

INTSML us_paramcommands(char *word, COMCOMP *arr[], char breakc)
{
	REGISTER INTSML i, count, ind;
	char *build[MAXPARS];
	REGISTER VARIABLE *var;

	if (*word == 0) return(0);

	/* see if it is a normal command */
	for(ind=0; us_lcommand[ind].name != 0; ind++)
		if (namesame(word, us_lcommand[ind].name) == 0)
	{
		count = us_lcommand[ind].params;
		for(i=0; i<count; i++) arr[i] = us_lcommand[ind].par[i];
		return(count);
	}

	/* see if it is a macro */
	var = us_getmacro(word);
	if (var != NOVARIABLE && getlength(var) >= 3)
	{
		count = us_parsecommand(((char **)var->addr)[2], build);
		for(i=0; i<count; i++) arr[i] = (COMCOMP *)myatoi(build[i]);
		return(count);
	}
	return(0);
}

/* parsing routines for a macro */
INTSML us_topofmacros(char **c)
{
	us_posmacrovar = 0;
	us_posbits = MACROS;
	return(1);
}

/* parsing routines for a popup menu */
INTSML us_topofpopupmenu(char **c)
{
	us_pospopupmenu = us_firstpopupmenu;
	us_posbits = POPUPS;
	return(1);
}

/* parsing routines for a port on the current node */
INTSML us_topofports(char **c)
{
	HIGHLIGHT high;
	REGISTER VARIABLE *var;

	us_posbits = PORTS;
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(0);
	(void)us_makehighlight(((char **)var->addr)[0], &high);

	if ((high.status&HIGHFROM) != 0 && high.fromgeom != NOGEOM &&
		high.fromgeom->entrytype == OBJNODEINST)
			us_posports = high.fromgeom->entryaddr.ni->proto->firstportproto; else
				us_posports = NOPORTPROTO;
	if (us_posports == NOPORTPROTO) return(0);
	return(1);
}

/* parsing routines for an exported port on the current facet */
INTSML us_topofcports(char **c)
{
	REGISTER NODEPROTO *np;

	np = getcurfacet();
	us_posbits = PORTS;
	if (np == NONODEPROTO) us_posports = NOPORTPROTO; else
		us_posports = np->firstportproto;
	if (us_posports == NOPORTPROTO) return(0);
	return(1);
}

/* parsing routines for an exported port on the current node */
INTSML us_topofexpports(char **c)
{
	HIGHLIGHT high;
	REGISTER VARIABLE *var;

	us_posbits = EXPORTS;
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(0);
	(void)us_makehighlight(((char **)var->addr)[0], &high);
	us_posexports = NOPORTEXPINST;
	if ((high.status&HIGHFROM) != 0 && high.fromgeom != NOGEOM &&
		high.fromgeom->entrytype == OBJNODEINST)
			us_posexports = high.fromgeom->entryaddr.ni->firstportexpinst;
	if (us_posexports == NOPORTEXPINST) return(0);
	return(1);
}

/* parsing routines for a window */
INTSML us_topofwindows(char **c)
{
	us_posbits = WINDOWS;
	us_poswindow = el_topwindowpart;
	return(1);
}

/* parsing routines for a layer name */
INTSML us_topoflayers(char **c)
{
	us_posbits = LAYERS;
	us_poslayers = el_curtech->layercount - 1;
	return(1);
}

/* parsing routines for a highlight */
INTSML us_topofhighlight(char **c)
{
	us_posbits = HIGHLIGHTS;
	us_poshigh = 0;
	return(1);
}

/* parsing routines for a node or arc prototype */
INTSML us_topofarcnodes(char **c)
{
	REGISTER char *pt;
	REGISTER TECHNOLOGY *t;

	us_posbits = ARCS | FACETS | PRIMS | CELLS;
	us_posnodeprotos = el_curlib->firstnodeproto;
	us_posprims = el_curtech->firstnodeproto;
	us_posarcs = el_curtech->firstarcproto;
	us_poscells = el_curlib->firstcell;

	/* see if a technology specification was given */
	for(pt = *c; *pt != 0; pt++) if (*pt == ':') break;
	if (*pt == ':')
	{
		*pt = 0;
		t = gettechnology(*c);
		*pt++ = ':';
		if (t != NOTECHNOLOGY)
		{
			*c = pt;
			us_posbits = ARCS | PRIMS;
			us_posprims = t->firstnodeproto;
			us_posarcs = t->firstarcproto;
		}
	}
	return(1);
}

/* parsing routines for a "getproto" command */
INTSML us_topofgetproto(char **c)
{
	(void)us_topofarcnodes(c);
	us_posbits |= WORDNODE | WORDARC | WORDNEXT | WORDPREV | WORDTHIS;
	us_posbits &= ~CELLS;
	return(1);
}

/* parsing routines for a node prototype */
INTSML us_topofnodes(char **c)
{
	REGISTER INTSML i;

	i = us_topofarcnodes(c);
	us_posbits &= ~(ARCS | CELLS);
	return(i);
}

/* parsing routines for a facet */
INTSML us_topoffacets(char **c)
{
	REGISTER char *pt;
	REGISTER LIBRARY *lib;

	/* by default, assume the current library */
	us_posbits = FACETS;
	us_posnodeprotos = el_curlib->firstnodeproto;

	/* see if a library specification was given */
	for(pt = *c; *pt != 0; pt++) if (*pt == ':') break;
	if (*pt == ':')
	{
		*pt = 0;
		lib = getlibrary(*c);
		*pt++ = ':';
		if (lib != NOLIBRARY)
		{
			*c = pt;
			us_posnodeprotos = lib->firstnodeproto;
		}
	}
	return(1);
}

/* parsing routines for a facet or cell name */
INTSML us_topofcellfacets(char **c)
{
	REGISTER char *pt;
	REGISTER LIBRARY *lib;

	/* by default, assume the current library */
	us_posbits = FACETS | CELLS;
	us_posnodeprotos = el_curlib->firstnodeproto;
	us_poscells = el_curlib->firstcell;

	/* see if a library specification was given */
	for(pt = *c; *pt != 0; pt++) if (*pt == ':') break;
	if (*pt == ':')
	{
		*pt = 0;
		lib = getlibrary(*c);
		*pt++ = ':';
		if (lib != NOLIBRARY)
		{
			*c = pt;
			us_posnodeprotos = lib->firstnodeproto;
			us_poscells = lib->firstcell;
		}
	}
	return(1);
}

/* parsing routines for a primitives nodeproto */
INTSML us_topofprims(char **c)
{
	REGISTER char *pt;
	REGISTER TECHNOLOGY *t;

	/* by default, assume the current technology */
	us_posbits = PRIMS;
	us_posprims = el_curtech->firstnodeproto;

	/* see if a technology specification was given */
	for(pt = *c; *pt != 0; pt++) if (*pt == ':') break;
	if (*pt == ':')
	{
		*pt = 0;
		t = gettechnology(*c);
		*pt++ = ':';
		if (t != NOTECHNOLOGY)
		{
			*c = pt;
			us_posprims = t->firstnodeproto;
		}
	}
	return(1);
}

/* parsing routines for a pure-layer nodeproto */
INTSML us_topofpures(char **c)
{
	us_posbits = PURES;
	us_posprims = el_curtech->firstnodeproto;
	return(1);
}

/* parsing routines for an arc prototype */
INTSML us_topofarcs(char **c)
{
	REGISTER char *pt;
	REGISTER TECHNOLOGY *t;

	/* by default, assume the current technology */
	us_posbits = ARCS;
	us_posarcs = el_curtech->firstarcproto;

	/* see if a technology specification was given */
	for(pt = *c; *pt != 0; pt++) if (*pt == ':') break;
	if (*pt == ':')
	{
		*pt = 0;
		t = gettechnology(*c);
		*pt++ = ':';
		if (t != NOTECHNOLOGY)
		{
			*c = pt;
			us_posarcs = t->firstarcproto;
		}
	}
	return(1);
}

/* parsing routines for a constraint solver */
INTSML us_topofconstraints(char **c)
{
	us_posbits = CONSTRAINTS;
	us_posconstraint = 0;
	return(1);
}

/* parsing routines for a mouse button */
INTSML us_topofmbuttons(char **c)
{
	us_posbits = MBUTTONS;
	us_posmbuttons = 0;
	return(1);
}

/* parsing routines for a node function name */
INTSML us_topofnodefun(char **c)
{
	us_posbits = NODEFUN;
	us_posfunction = 0;
	return(1);
}

/* parsing routines for an arc function name */
INTSML us_topofarcfun(char **c)
{
	us_posbits = ARCFUN;
	us_posfunction = 0;
	return(1);
}

/* parsing routines for port information when technology editing */
INTSML us_topofedtecport(char **c)
{
	us_posbits = EDTECPORT;
	us_poscells = el_curlib->firstcell;
	us_posspecial = 0;
	return(1);
}

/* parsing routines for a layer name when technology editing */
INTSML us_topofedteclay(char **c)
{
	us_posbits = EDTECLAY;
	us_poscells = el_curlib->firstcell;
	return(1);
}

/*
 * parsing routines for a layer name or "SET-MINIMUM-SIZE",
 * when technology editing
 */
INTSML us_topofedtecslay(char **c)
{
	us_posbits = EDTECLAY|WORDSET;
	us_poscells = el_curlib->firstcell;
	return(1);
}

/*
 * parsing routines for a layer name or "CLEAR-MINIMUM-SIZE",
 * when technology editing
 */
INTSML us_topofedtecclay(char **c)
{
	us_posbits = EDTECLAY|WORDCLEAR;
	us_poscells = el_curlib->firstcell;
	return(1);
}

/* parsing routines for an arc name when technology editing */
INTSML us_topofedtecarc(char **c)
{
	us_posbits = EDTECARC;
	us_poscells = el_curlib->firstcell;
	return(1);
}

/* parsing routines for a node name when technology editing */
INTSML us_topofedtecnode(char **c)
{
	us_posbits = EDTECNODE;
	us_poscells = el_curlib->firstcell;
	return(1);
}

/* parsing routines for editable technology name when tech editing */
INTSML us_topofedtectech(char **c)
{
	us_posbits = EDTECTECH;
	us_postechs = el_technologies;
	return(1);
}

/* parsing routines for a nodeproto/arcproto/library/macro/portproto */
INTSML us_topofallthings(char **c)
{
	REGISTER NODEPROTO *np;

	us_posbits = PRIMS | CELLS | ARCS | LIBS | MACROS | PORTS | POPUPS | CONSTRAINTS | NETWORKS;
	us_posprims = el_curtech->firstnodeproto;
	us_poscells = el_curlib->firstcell;
	us_posarcs = el_curtech->firstarcproto;
	us_poslibs = el_curlib;
	us_posmacrovar = 0;
	us_pospopupmenu = us_firstpopupmenu;
	us_posconstraint = 0;
	np = getcurfacet();
	if (np != NONODEPROTO) us_posports = np->firstportproto; else
	{
		us_posports = NOPORTPROTO;
		us_posbits &= ~PORTS;
	}
	(void)topofnets(c);
	return(1);
}

char *us_nextparse(void)
{
	char *nextname;
	INTSML unimportant, funct;
	INTBIG bits;
	REGISTER TECHNOLOGY *tech;

	if ((us_posbits&COMMANDS) != 0)
	{
		if (us_poslcommands < us_longcount) return(us_lcommand[us_poslcommands++].name);
		us_posbits &= ~COMMANDS;
	}
	if ((us_posbits&MACROS) != 0)
	{
		while (us_posmacrovar < us_aid->numvar)
		{
			nextname = makename(us_aid->firstvar[us_posmacrovar].key);
			us_posmacrovar++;
			if (namesamen(nextname, "USER_macro_", 11) == 0) return(&nextname[11]);
		}
		us_posbits &= ~MACROS;
	}
	if ((us_posbits&PORTS) != 0)
	{
		if (us_posports != NOPORTPROTO)
		{
			nextname = us_posports->protoname;
			us_posports = us_posports->nextportproto;
			return(nextname);
		}
		us_posbits &= ~PORTS;
	}
	if ((us_posbits&EXPORTS) != 0)
	{
		if (us_posexports != NOPORTEXPINST)
		{
			nextname = us_posexports->exportproto->protoname;
			us_posexports = us_posexports->nextportexpinst;
			return(nextname);
		}
		us_posbits &= ~EXPORTS;
	}
	if ((us_posbits&WINDOWS) != 0)
	{
		if (us_poswindow != NOWINDOWPART)
		{
			(void)initinfstr();
			(void)addstringtoinfstr(describenodeproto(us_poswindow->curnodeproto));
			(void)addtoinfstr('(');
			(void)addstringtoinfstr(us_poswindow->location);
			(void)addtoinfstr(')');
			us_poswindow = us_poswindow->nextwindowpart;
			return(returninfstr());
		}
		us_posbits &= ~WINDOWS;
	}
	if ((us_posbits&HIGHLIGHTS) != 0)
	{
		while (us_poshigh < us_aid->numvar)
		{
			nextname = (char *)us_aid->firstvar[us_poshigh].key;
			us_poshigh++;
			if (namesamen(nextname, "USER_highlight_", 15) == 0) return(&nextname[15]);
		}
		us_posbits &= ~HIGHLIGHTS;
	}
	if ((us_posbits&ARCS) != 0)
	{
		if (us_posarcs != NOARCPROTO)
		{
			nextname = describearcproto(us_posarcs);
			us_posarcs = us_posarcs->nextarcproto;
			return(nextname);
		}
		us_posbits &= ~ARCS;
	}
	if ((us_posbits&CELLS) != 0)
	{
		if (us_poscells != NOCELL)
		{
			nextname = us_poscells->cellname;
			us_poscells = us_poscells->nextcell;
			return(nextname);
		}
		us_posbits &= ~CELLS;
	}
	if ((us_posbits&FACETS) != 0)
	{
		if (us_posnodeprotos != NONODEPROTO)
		{
			nextname = describenodeproto(us_posnodeprotos);
			us_posnodeprotos = us_posnodeprotos->nextnodeproto;
			return(nextname);
		}
		us_posbits &= ~FACETS;
	}
	if ((us_posbits&PRIMS) != 0)
	{
		if (us_posprims != NONODEPROTO)
		{
			nextname = us_posprims->primname;
			us_posprims = us_posprims->nextnodeproto;
			return(nextname);
		}
		us_posbits &= ~PRIMS;
	}
	if ((us_posbits&PURES) != 0)
	{
		while (us_posprims != NONODEPROTO)
		{
			nextname = us_posprims->primname;
			funct = (INTSML)((us_posprims->userbits&NFUNCTION)>>NFUNCTIONSH);
			bits = us_posprims->userbits & NNOTUSED;
			us_posprims = us_posprims->nextnodeproto;
			if (bits != 0) continue;
			if (funct == NPNODE) return(nextname);
		}
		us_posbits &= ~PURES;
	}
	if ((us_posbits&LIBS) != 0)
	{
		if (us_poslibs != NOLIBRARY)
		{
			nextname = us_poslibs->libname;
			us_poslibs = us_poslibs->nextlibrary;
			return(nextname);
		}
		us_posbits &= ~LIBS;
	}
	if ((us_posbits&LAYERS) != 0)
	{
		if (us_poslayers >= 0)
		{
			(void)initinfstr();
			(void)addstringtoinfstr(us_layerletters(el_curtech, us_poslayers));
			(void)addtoinfstr('(');
			(void)addstringtoinfstr(layername(el_curtech, us_poslayers));
			(void)addtoinfstr(')');
			us_poslayers--;
			return(returninfstr());
		}
		us_posbits &= ~LAYERS;
	}
	if ((us_posbits&POPUPS) != 0)
	{
		if (us_pospopupmenu != NOPOPUPMENU)
		{
			nextname = us_pospopupmenu->name;
			us_pospopupmenu = us_pospopupmenu->nextpopupmenu;
			return(nextname);
		}
		us_posbits &= ~POPUPS;
	}
	if ((us_posbits&CONSTRAINTS) != 0)
	{
		if (el_constraints[us_posconstraint].conname != 0)
		{
			nextname = el_constraints[us_posconstraint].conname;
			us_posconstraint++;
			return(nextname);
		}
		us_posbits &= ~CONSTRAINTS;
	}
	if ((us_posbits&MBUTTONS) != 0)
	{
		if (us_posmbuttons < buttoncount())
		{
			nextname = buttonname(us_posmbuttons, &unimportant);
			us_posmbuttons++;
			return(nextname);
		}
		us_posbits &= ~MBUTTONS;
	}
	if ((us_posbits&VIEWS) != 0)
	{
		if (us_posviews == (VIEW *)0)
		{
			us_posviews = el_views;
			return("unknown");
		}
		if (us_posviews != NOVIEW)
		{
			nextname = us_posviews->viewname;
			us_posviews = us_posviews->nextview;
			return(nextname);
		}
		us_posbits &= ~VIEWS;
	}
	if ((us_posbits&NODEFUN) != 0)
	{
		nextname = nodefunctionname(us_posfunction, NONODEINST);
		us_posfunction++;
		if (*nextname != 0) return(nextname);
		us_posbits &= ~NODEFUN;
	}
	if ((us_posbits&ARCFUN) != 0)
	{
		nextname = arcfunctionname(us_posfunction);
		us_posfunction++;
		if (*nextname != 0) return(nextname);
		us_posbits &= ~ARCFUN;
	}
	if ((us_posbits&EDTECPORT) != 0)
	{
		while (us_poscells != NOCELL)
		{
			nextname = us_poscells->cellname;
			us_poscells = us_poscells->nextcell;
			if (namesamen(nextname, "arc-", 4) == 0) return(&nextname[4]);
		}
		switch (us_posspecial++)
		{
			case 0: return("PORT-ANGLE");
			case 1: return("PORT-ANGLE-RANGE");
		}
		us_posbits &= ~EDTECPORT;
	}
	if ((us_posbits&EDTECLAY) != 0)
	{
		while (us_poscells != NOCELL)
		{
			nextname = us_poscells->cellname;
			us_poscells = us_poscells->nextcell;
			if (namesamen(nextname, "layer-", 6) == 0) return(&nextname[6]);
		}
		us_posbits &= ~EDTECLAY;
	}
	if ((us_posbits&EDTECARC) != 0)
	{
		while (us_poscells != NOCELL)
		{
			nextname = us_poscells->cellname;
			us_poscells = us_poscells->nextcell;
			if (namesamen(nextname, "arc-", 4) == 0) return(&nextname[4]);
		}
		us_posbits &= ~EDTECARC;
	}
	if ((us_posbits&EDTECNODE) != 0)
	{
		while (us_poscells != NOCELL)
		{
			nextname = us_poscells->cellname;
			us_poscells = us_poscells->nextcell;
			if (namesamen(nextname, "node-", 5) == 0) return(&nextname[5]);
		}
		us_posbits &= ~EDTECNODE;
	}
	if ((us_posbits&EDTECTECH) != 0)
	{
		while (us_postechs != NOTECHNOLOGY)
		{
			tech = us_postechs;
			us_postechs = us_postechs->nexttechnology;
			if ((tech->userbits&NONSTANDARD) == 0) return(tech->techname);
		}
		us_posbits &= ~EDTECTECH;
	}
	if ((us_posbits&NETWORKS) != 0)
	{
		nextname = nextnets();
		if (nextname != 0) return(nextname);
		us_posbits &= ~NETWORKS;
	}
	if ((us_posbits&WORDSET) != 0)
	{
		us_posbits &= ~WORDSET;
		return("SET-MINIMUM-SIZE");
	}
	if ((us_posbits&WORDCLEAR) != 0)
	{
		us_posbits &= ~WORDCLEAR;
		return("CLEAR-MINIMUM-SIZE");
	}
	if ((us_posbits&WORDNODE) != 0)
	{
		us_posbits &= ~WORDNODE;
		return("NODE");
	}
	if ((us_posbits&WORDARC) != 0)
	{
		us_posbits &= ~WORDARC;
		return("ARC");
	}
	if ((us_posbits&WORDNEXT) != 0)
	{
		us_posbits &= ~WORDNEXT;
		return("NEXT-PROTO");
	}
	if ((us_posbits&WORDPREV) != 0)
	{
		us_posbits &= ~WORDPREV;
		return("PREV-PROTO");
	}
	if ((us_posbits&WORDTHIS) != 0)
	{
		us_posbits &= ~WORDTHIS;
		return("THIS-PROTO");
	}
	return(0);
}

/* parsing routines for a var */
INTSML us_topofvars(char **c)
{
	REGISTER char *pp, save, *zap;
	REGISTER INTSML i, j, res;
	char *qual;
	INTBIG objaddr, objtype;
	REGISTER VARIABLE *var;

	/* ignore any "$" at the start of the variable */
	pp = *c;
	for(i = strlen(pp)-1; i >= 0; i--) if (pp[i] == '$') break;
	if (pp[i] == '$')
	{
		/* see if the variable is in parenthesis (and is complete) */
		if (pp[++i] == '(')
		{
			i++;
			for(j=i; pp[j] != 0; j++) if (pp[j] == ')') return(0);
		}
		pp = &pp[i];
	}

	/* look for point where ambiguity may exist */
	zap = pp;
	for(i = strlen(pp)-1; i >= 0; i--) if (pp[i] == '.' || pp[i] == ':')
	{
		*c = &pp[i+1];
		zap = &pp[i];

		/* if this is a ":" construct, special case it */
		if (*zap == ':')
		{
			if (namesamen(pp, "technology", i) == 0 && i >= 1)
			{
				us_posswitch = -2;   return(topoftechs(c));
			}
			if (namesamen(pp, "library", i) == 0 && i >= 1)
			{
				us_posswitch = -3;   return(topoflibs(c));
			}
			if (namesamen(pp, "aid", i) == 0 && i >= 2)
			{
				us_posswitch = -4;   return(topofaids(c));
			}
			if (namesamen(pp, "arc", i) == 0 && i >= 2)
			{
				us_posswitch = -5;   return(us_topofarcs(c));
			}
			if (namesamen(pp, "node", i) == 0 && i >= 1)
			{
				us_posswitch = -6;   return(us_topofnodes(c));
			}
			if (namesamen(pp, "primitive", i) == 0 && i >= 1)
			{
				us_posswitch = -7;   return(us_topofprims(c));
			}
			if (namesamen(pp, "facet", i) == 0 && i >= 1)
			{
				us_posswitch = -8;   return(us_topoffacets(c));
			}
			if (namesamen(pp, "cell", i) == 0 && i >= 1)
			{
				us_posswitch = -9;   return(topofcells(c));
			}
			if (namesamen(pp, "view", i) == 0 && i >= 1)
			{
				us_posswitch = -10;   return(topofviews(c));
			}
			if (namesamen(pp, "net", i) == 0 && i >= 2)
			{
				us_posswitch = -11;   return(topofnets(c));
			}
			if (namesamen(pp, "window", i) == 0 && i >= 1)
			{
				us_posswitch = -12;   return(us_topofwindows(c));
			}
		}
		break;
	}

	/* now determine the context of this point */
	save = *zap;   *zap = 0;
	res = us_evaluatevariable(pp, &objaddr, &objtype, &qual);
	if (res == 0 && *qual != 0)
	{
		var = getval(objaddr, objtype, -1, qual);
		if (var != NOVARIABLE)
		{
			objaddr = var->addr;   objtype = var->type;
		} else res = 1;
	}
	*zap = save;

	/* initialize lists and return the description */
	if (res != 0) us_posswitch = VINTEGER; else
	{
		us_posswitch = objtype & VTYPE;
		if (us_posswitch != VUNKNOWN && us_posswitch != VINTEGER &&
			us_posswitch != VADDRESS && us_posswitch != VCHAR &&
			us_posswitch != VSTRING && us_posswitch != VFLOAT &&
			us_posswitch != VDOUBLE && us_posswitch != VFRACT &&
			us_posswitch != VSHORT && us_posswitch != VGENERAL)
				us_possearch = initobjlist(objaddr, objtype, 0);
	}
	return(1);
}
char *us_nextvars(void)
{
	VARIABLE *var;

	switch (us_posswitch)
	{
		case -2:          return(nexttechs());
		case -3:          return(nextlibs());
		case -4:          return(nextaids());
		case -9:          return(nextcells());
		case -10:         return(nextviews());
		case -11:         return(nextnets());
		case -5:
		case -6:
		case -7:
		case -8:
		case -12:         return(us_nextparse());
		case VNODEINST:
		case VNODEPROTO:
		case VPORTARCINST:
		case VPORTEXPINST:
		case VPORTPROTO:
		case VARCINST:
		case VARCPROTO:
		case VGEOM:
		case VLIBRARY:
		case VTECHNOLOGY:
		case VAID:
		case VRTNODE:
		case VNETWORK:
		case VCELL:
		case VVIEW:
		case VWINDOWPART:
		case VGRAPHICS:
		case VCONSTRAINT:
		case VWINDOWFRAME: return(nextobjectlist(&var, us_possearch));
	}
	return(0);
}

INTSML us_topofeditor(char **c)
{
	us_possearch = 0;
	return(1);
}

char *us_nexteditor(void)
{
	REGISTER char *nextname;

	nextname = us_editortable[us_possearch].editorname;
	us_possearch++;
	return(nextname);
}

/****************************** PARSING TABLES ******************************/

/* common parsing tables */
KEYWORD us_onoffopt[] =
{
	{"on",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"off",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static KEYWORD yesnoopt[] =
{
	{"yes",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"no",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_yesnop = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "yes or no", 0};
COMCOMP us_noyesp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "yes or no", 0};
COMCOMP us_artlookp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "artwork look", 0};
COMCOMP us_purelayerp = {NOKEYWORD, us_topofpures, us_nextparse, NOPARAMS, NOBACKUP,
	0, " \t", "primitive name", 0};

/* for "arc" command */
COMCOMP us_arcnamep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "local arc name", 0};
static COMCOMP arcpropp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "arc constraints", 0};
static KEYWORD arcpropaopt[] =
{
	{"add",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP arcpropap = {arcpropaopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "should this constraint replace or augment others", 0};
static KEYWORD arcnotopt[] =
{
	{"manhattan",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"fixed-angle",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rigid",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"slide",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"temp-rigid",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"directional",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"negated",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ends-extend",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"center",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"curve",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"constraint",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"skip-tail",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"skip-head",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"name",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_arcnotp = {arcnotopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "arc style parameter to negate", 0};
static KEYWORD arctogopt[] =
{
	{"fixed-angle",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rigid",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"slide",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"directional",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"negated",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ends-extend",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"skip-tail",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"skip-head",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_arctogp = {arctogopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "arc style parameter to toggle", 0};
static KEYWORD arcopt[] =
{
	{"manhattan",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"fixed-angle",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rigid",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"slide",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"temp-rigid",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"directional",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"negated",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ends-extend",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"skip-head",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"skip-tail",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"center",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"curve",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"constraint",          2,{&arcpropp,&arcpropap,NOKEY,NOKEY,NOKEY}},
	{"name",                1,{&us_arcnamep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",                 1,{&us_arcnotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"toggle",              1,{&us_arctogp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"reverse",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_arcp = {arcopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "arc style argument", 0};

/* for "array" command */
COMCOMP us_arrayxp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "X array extent (or 'file' for array file)", 0};
static COMCOMP arrayyp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "Y array extent", 0};
static COMCOMP arrayxop = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "X overlap distance", 0};
static COMCOMP arrayyop = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "Y overlap distance", 0};
static KEYWORD arrayswitchopt[] =
{
	{"no-names",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"diagonal",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP arrayswitchp = {arrayswitchopt,NOTOPLIST,NONEXTLIST,NOPARAMS,
	NOBACKUP, 0, " \t", "array option switch", "use names"};

/* for "bind" command */
extern COMCOMP us_userp;
INTSML us_parambind(char *i, COMCOMP *j[], char c)
{ j[0] = &us_userp; return(1);}
COMCOMP us_bindkeyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, us_parambind, NOBACKUP,
	INPUTOPT, " \t", "key to bind", 0};
static COMCOMP bindmenugp = {NOKEYWORD, us_topofnodes, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "glyph to use for menu display", 0};
static COMCOMP bindmenump = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "message to place in menu display", 0};
COMCOMP us_bindmenuxp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,us_parambind,NOBACKUP,
	0, " \t", "column of menu to bind", 0};
COMCOMP us_bindmenuryp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "row of menu to bind", 0};
static KEYWORD bindmenuxpopt[] =
{
	{"glyph",      2,{&bindmenugp,&us_bindmenuryp,NOKEY,NOKEY,NOKEY}},
	{"message",    2,{&bindmenump,&us_bindmenuryp,NOKEY,NOKEY,NOKEY}},
	{"background", 2,{&bindmenump,&us_bindmenuryp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP bindmenuyp = {bindmenuxpopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "row of menu to bind or 'glyph/message' specification", 0};
COMCOMP us_bindbuttonp = {NOKEYWORD,us_topofmbuttons,us_nextparse,us_parambind,
	NOBACKUP, INPUTOPT, " \t", "mouse button to bind", 0};
COMCOMP us_bindpoprep = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "entry in popup menu to bind", 0};
static KEYWORD bindpopepopt[] =
{
	{"message",    2,{&bindmenump,&us_bindpoprep,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP bindpopep = {bindpopepopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "entry in popup menu to bind or 'message' specification", 0};
INTSML us_parambindpop(char *i, COMCOMP *j[], char c)
{ j[0] = &bindpopep; j[1] = &us_userp; return(2);}
COMCOMP us_bindpopnp = {NOKEYWORD, us_topofpopupmenu, us_nextparse,
	us_parambindpop, NOBACKUP, INPUTOPT, " \t", "popup menu to bind", 0};
static KEYWORD bindsetopt[] =
{
	{"button",         1,{&us_bindbuttonp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"key",            1,{&us_bindkeyp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"menu",           2,{&bindmenuyp,&us_bindmenuxp,NOKEY,NOKEY,NOKEY}},
	{"popup",          1,{&us_bindpopnp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"input-popup",    1,{&us_bindpopnp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_bindsetp = {bindsetopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "what would you like to bind", 0};
static KEYWORD bindnotopt[] =
{
	{"verbose",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_bindnotp = {bindnotopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "what binding condition would you like to negate", 0};
static COMCOMP bindgbuttonp = {NOKEYWORD, us_topofmbuttons, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "mouse button to get", 0};
COMCOMP us_bindgkeyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "key to get", 0};
static COMCOMP bindgmenuxp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "column of menu to get", 0};
static COMCOMP bindgmenuyp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "row of menu to get", 0};
static COMCOMP bindgpopnp = {NOKEYWORD, us_topofpopupmenu, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "popup menu to get", 0};
static COMCOMP bindgpopip = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "index of popup menu to get", 0};
static KEYWORD bindgetopt[] =
{
	{"button",         1,{&bindgbuttonp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"key",            1,{&us_bindgkeyp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"menu",           2,{&bindgmenuyp,&bindgmenuxp,NOKEY,NOKEY,NOKEY}},
	{"popup",          2,{&bindgpopnp,&bindgpopip,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP bindgetp = {bindgetopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "what binding would you like to obtain", 0};
static COMCOMP bindgetlp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "command-interpreter letter to hold the binding", 0};
static KEYWORD bindopt[] =
{
	{"verbose",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",            1,{&us_bindnotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set",            1,{&us_bindsetp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"get",            2,{&bindgetlp,&bindgetp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_bindp = {bindopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "binding option", 0};

/* the "color" command */
COMCOMP us_colorentryp = {NOKEYWORD, us_topoflayers, us_nextparse, NOPARAMS,
	NOBACKUP, NOFILL|INPUTOPT, " \t", "table index or layer letter(s)", 0};
static COMCOMP colorrvaluep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "new value for red colormap entry", 0};
static COMCOMP colorgvaluep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "new value for green colormap entry", 0};
static COMCOMP colorbvaluep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "new value for blue colormap entry", 0};
COMCOMP us_colorpvaluep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "new value for raster pattern", 0};
static COMCOMP colorhighap = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "amount to highlight (0 to 1)", "0.5"};
COMCOMP us_colorhighp = {NOKEYWORD,us_topoflayers,us_nextparse,NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "overlappable layer letter(s) to highlight", 0};
COMCOMP us_colorreadp = {NOKEYWORD, topoffile, nextfile, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT, " \t", "colormap file name", 0};
COMCOMP us_colorwritep = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "colormap output file name", 0};
static KEYWORD coloropt[] =
{
	{"default",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"black-background", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"white-background", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"entry",            4,{&us_colorentryp,&colorrvaluep,&colorgvaluep,&colorbvaluep,NOKEY}},
	{"pattern",          2,{&us_colorentryp,&us_colorpvaluep,NOKEY,NOKEY,NOKEY}},
	{"mix",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"highlight",        2,{&us_colorhighp,&colorhighap,NOKEY,NOKEY,NOKEY}},
	{"read",             1,{&us_colorreadp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"write",            1,{&us_colorwritep,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_colorp = {coloropt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "colormap option", 0};

/* the "commandfile" command */
static KEYWORD commandfileswitchopt[] =
{
	{"verbose",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP commandfileswitchp = {commandfileswitchopt, NOTOPLIST,
	NONEXTLIST, NOPARAMS, NOBACKUP, 0, " \t", "command file switch", 0};
COMCOMP us_commandfilep = {NOKEYWORD, topoffile, nextfile, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT, " \t", "command file name", 0};

/* the "constraint" command */
static COMCOMP constraintup = {NOKEYWORD, us_topofconstraints, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "constraint solver to use", 0};
static COMCOMP constraintap = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	us_paramconstraintb, NOBACKUP, 0, " \t", "message for constraint solver", 0};
INTSML us_paramconstraintb(char *i, COMCOMP *j[], char c)
	{ j[0]= &constraintap; return(1);}
INTSML us_paramconstrainta(char *pt, COMCOMP *j[], char c)
{
	REGISTER INTSML i;

	i = parse(pt, &constraintup, 0);
	if (i >= 0)
	{
		j[0] = el_constraints[i].parse;
		if (j[0] != NOCOMCOMP) return(1);
	}
	return(us_paramconstraintb(pt, j, c));
}
static COMCOMP constrainttp = {NOKEYWORD, us_topofconstraints, us_nextparse,
	us_paramconstrainta, NOBACKUP, INPUTOPT, " \t", "constraint to direct", 0};
static KEYWORD constraintopt[] =
{
	{"tell",     1,{&constrainttp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use",      1,{&constraintup,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP constraintp = {constraintopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "constraint solver action", 0};

/* the "copyfacet" command */
COMCOMP us_copyfacetp = {NOKEYWORD, us_topofcellfacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "source facet name (or library:facet)", 0};
static COMCOMP copyfacetdp = {NOKEYWORD,us_topofcellfacets,us_nextparse,NOPARAMS,
	NOBACKUP, NOFILL|INCLUDENOISE, " \t", "destination facet name", "same facet name"};
static KEYWORD copyfacetqopt[] =
{
	{"quiet",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP copyfacetqp = {copyfacetqopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "verbosity", "report success"};

/* the "create" command */
static COMCOMP createap = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "angle increment from current object", 0};
static COMCOMP createjp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "angle increment from current object to other object", 0};
COMCOMP us_createxp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "X coordinate of object", 0};
COMCOMP us_createyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Y coordinate of object", 0};
static KEYWORD createopt[] =
{
	{"remain-highlighted", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"wait-for-down",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"contents",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"insert",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"breakpoint",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"angle",              1,{&createap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"join-angle",         1,{&createjp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"to",                 2,{&us_createxp,&us_createyp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP createp = {createopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "creation option", 0};

/* the "debug" command */
static KEYWORD debugcopt[] =
{
	{"verbose",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP debugcp = {debugcopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "database checking option", 0};
static COMCOMP debugnp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "numerator of scaling factor", 0};
static COMCOMP debugdp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "denominator of scaling factor", 0};
static COMCOMP debugap = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "arena name to map", 0};
static KEYWORD debugopt[] =
{
	/* standard */
	{"arena",                   1,{&debugap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"check-database",          1,{&debugcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"examine-options",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"freeze-user-interface",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"internal-errors",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"namespace",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"options-changed",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rtree",                   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"undo",                    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	/* special case */
	{"erase-bits",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"prepare-tsmc-io",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"highlight-nonmanhattan",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"label-facet",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"out-of-bounds",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"convert-dashes",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"scale-facet",             2,{&debugnp,&debugdp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP debugp = {debugopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "debugging option", 0};

/* the "defarc" command */
COMCOMP us_defarcwidp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "default arc width", 0};
COMCOMP us_defarcangp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "default arc angle increment", 0};
COMCOMP us_defarcpinp = {NOKEYWORD, us_topofprims, us_nextparse, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "default pin to use for this arc", 0};
static KEYWORD defarcnotopt[] =
{
	{"manhattan",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"fixed-angle",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"slide",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rigid",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"directional",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"negated",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ends-extend",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_defarcnotp = {defarcnotopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "default arc setting", 0};
static KEYWORD defarcopt[] =
{
	{"not",              1,{&us_defarcnotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"default",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"angle",            1,{&us_defarcangp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pin",              1,{&us_defarcpinp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"width",            1,{&us_defarcwidp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"manhattan",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"fixed-angle",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"slide",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rigid",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"directional",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"negated",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ends-extend",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP defarcp = {defarcopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "default arc setting", "describe current"};
COMCOMP us_defarcsp = {NOKEYWORD,us_topofarcs,us_nextparse,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "arc prototype to set defaults (* for all)", 0};

/* the "defnode" command */
COMCOMP us_defnodexsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "default node X size", 0};
COMCOMP us_defnodeysp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "default node Y size", 0};
static COMCOMP defnodepp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "default node placement", "90 degrees more"};
static KEYWORD defnodenotopt[] =
{
	{"alterable",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"centered-primitives", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"check-dates",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"copy-ports",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"expanded",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"from-cell-library",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"instances-locked",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"locked-primitives",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP defnodenotp = {defnodenotopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "default node NOT setting", 0};
static KEYWORD defnodeopt[] =
{
	{"alterable",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"centered-primitives", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"check-dates",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"copy-ports",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"expanded",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"from-cell-library",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"instances-locked",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"locked-primitives",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",                 1,{&defnodenotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"placement",           1,{&defnodepp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"size",                2,{&us_defnodexsp,&us_defnodeysp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_defnodep = {defnodeopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "default node setting", "describe current"};
COMCOMP us_defnodesp = {NOKEYWORD, us_topofnodes, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "node prototype to set defaults (* for all)", 0};

/* the "echo" command */
static COMCOMP echop = {NOKEYWORD,NOTOPLIST,NONEXTLIST,us_nextecho,NOBACKUP,
	0, " \t", "argument to be echoed", 0};
INTSML us_nextecho(char *i, COMCOMP *j[], char c)
{ j[0] = &echop; return(1); }

/* the "editfacet" command */
extern COMCOMP editfacetwindowp;
static KEYWORD editfacetwindowopt[] =
{
	{"non-redundant",    1,{&editfacetwindowp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"new-window",       1,{&editfacetwindowp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP editfacetwindowp = {editfacetwindowopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "option to display in a new window", 0};
COMCOMP us_editfacetp = {NOKEYWORD,us_topofcellfacets,us_nextparse,NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT|INCLUDENOISE, " \t", "cell facet to be edited ('-' to follow ports)", 0};

/* the "erase" command */
static KEYWORD eraseopt[] =
{
	{"geometry",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clean-up",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pass-through",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP erasep = {eraseopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "erase option", 0};

/* the "find" command */
static COMCOMP findnamep = {NOKEYWORD, us_topofhighlight, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "highlight name", 0};
COMCOMP us_findnnamep = {NOKEYWORD, us_topofhighlight, us_nextparse,
	NOPARAMS, NOBACKUP, NOFILL|INPUTOPT|INCLUDENOISE, " \t", "highlight name", 0};
static COMCOMP findnodep = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "name of node to highlight", 0};
COMCOMP us_findexportp = {NOKEYWORD, us_topofcports, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of exported port to highlight", 0};
static COMCOMP findarcp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "name of arc/network to highlight", 0};
static COMCOMP findintp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "angle increment from current position", 0};
COMCOMP us_findobjap = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "address of object being selected", ""};
static KEYWORD findsnapopt[] =
{
	{"none",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"center",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"midpoint",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"endpoint",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tangent",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"perpendicular",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"quadrant",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"intersection",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP findsnapp = {findsnapopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "vertex snapping mode", ""};
static KEYWORD findcondefopt[] =
{
	{"on",                 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"off",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP findcondefp = {findcondefopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "default constraint mode", ""};
static KEYWORD findallopt[] =
{
	{"easy",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"hard",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP findallp = {findallopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "select-all option", ""};
extern COMCOMP us_findp, us_varbs2typ;
static KEYWORD findopt[] =
{
	{"all",                1,{&findallp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"another",            1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"arc",                1,{&findarcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"area-move",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"area-define",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"area-size",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clear",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"comp-interactive",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"constrained",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"constraint-angle",   1,{&findintp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"constraint-default", 1,{&findcondefp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"deselect-arcs",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down-stack",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"exclusively",        1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"export",             1,{&us_findexportp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"extra-info",         1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"interactive",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"more",               1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"name",               1,{&findnamep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"no-box",             1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"node",               1,{&findnodep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"object",             2,{&us_varbs2typ,&us_findobjap,NOKEY,NOKEY,NOKEY}},
	{"port",               1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"save",               1,{&us_findnnamep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set-easy-selection", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set-hard-selection", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"snap-mode",          1,{&findsnapp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"special",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"still",              1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up-stack",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vertex",             1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"within",             1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_findp = {findopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "selection option", 0};

/* the "getproto" command */
COMCOMP us_getproto1p = {NOKEYWORD, us_topofgetproto, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "arc or node prototype", 0};
static COMCOMP getproto2p = {NOKEYWORD, us_topofarcnodes, us_nextparse,
	NOPARAMS, NOBACKUP, 0, " \t", "arc or node prototype", 0};

/* the "grid" command */
COMCOMP us_gridalip = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "grid cursor alignment value", 0};
static COMCOMP gridedgesp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "grid edge alignment value", 0};
static COMCOMP gridxp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "grid spacing in X", 0};
static COMCOMP gridyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "grid spacing in Y", "same as X"};
static KEYWORD gridopt[] =
{
	{"alignment",        1,{&us_gridalip,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edges",            1,{&gridedgesp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"size",             2,{&gridxp,&gridyp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_gridp = {gridopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "grid option", "toggle grid"};

/* the "help" command */
static KEYWORD helpopt[] =
{
	{"manual",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"news",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"illustrate",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_helpp = {helpopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "news, manual, or illustrate", "command-line help"};

/* the "if" command */
static KEYWORD ifropt[] =
{
	{"==",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"!=",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"<",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"<=",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{">",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{">=",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP ifrp = {ifropt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "conditional relationship", 0};
static COMCOMP iftp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "value to be compared", 0};

/* the "interpret" command */
static COMCOMP interpretfip = {NOKEYWORD,topoffile, nextfile,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "file with code to execute", ""};
static KEYWORD interpretfopt[] =
{
	{"lisp",        1,{&interpretfip,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tcl",         1,{&interpretfip,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"java",        1,{&interpretfip,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP interpretfp = {interpretfopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "language to execute", ""};
static COMCOMP interpretcp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "code to execute", "invoke interpreter"};
static KEYWORD interpretopt[] =
{
	{"lisp",        1,{&interpretcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tcl",         1,{&interpretcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"java",        1,{&interpretcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"file",        1,{&interpretfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP interpretp = {interpretopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "language or code to execute", "invoke interpreter"};

/* the "iterate" command */
static COMCOMP iteratemp = {NOKEYWORD, us_topofmacros, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "macro to execute", 0};
static COMCOMP iteratep = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT|INCLUDENOISE, " \t", "number of times to repeat last command, or array variable to iterate", "1"};

/* the "killfacet" command */
COMCOMP us_killfacetp = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of cell facet to delete", 0};

/* the "lambda" command */
COMCOMP us_lambdachp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "new value for lambda", 0};
static KEYWORD lambdadunopt[] =
{
	{"lambda",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"inch",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"centimeter",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"millimeter",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"mil",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"micron",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"centimicron",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"millimicron",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP lambdadunp = {lambdadunopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "units to use when displaying distance", 0};
static KEYWORD lambdaiunopt[] =
{
	{"half-decimicron",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"half-millimicron", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP lambdaiunp = {lambdaiunopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "smallest internal unit", 0};
static KEYWORD lambdaopt[] =
{
	{"change",          1,{&us_lambdachp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"no-scale-change", 1,{&us_lambdachp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"display-units",   1,{&lambdadunp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"internal-units",  1,{&lambdaiunp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_lambdap = {lambdaopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "lambda changing option", "print current values"};

/* the "library" command */
COMCOMP us_libraryup = {NOKEYWORD, topoflibs, nextlibs, NOPARAMS,
	NOBACKUP, NOFILL|INPUTOPT|INCLUDENOISE, " \t", "name of library to use", 0};
COMCOMP us_librarynp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of new library to create", 0};
COMCOMP us_libraryrp = {NOKEYWORD,topoflibfile,nextfile,NOPARAMS,NOBACKUP,
	NOFILL|INPUTOPT, " \t", "name of library file to read", 0};
COMCOMP us_librarywp = {NOKEYWORD, topoflibs, nextlibs, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of library to write", 0};
static KEYWORD librarymcopt[] =
{
	{"make-current", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"merge",        1,{&us_libraryup,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP libraryreadmcp = {librarymcopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "option to library input", 0};
static KEYWORD libraryreadformatopt[] =
{
	{"binary",       1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cif",          1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"def",          1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dxf",          1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edif",         1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gds",          1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lef",          1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"sue",          1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"text",         1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vhdl",         1,{&libraryreadmcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"make-current", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"merge",        1,{&us_libraryup,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP libraryreadformatp = {libraryreadformatopt, NOTOPLIST,
	NONEXTLIST, NOPARAMS, NOBACKUP, 0, " \t", "library read option", "binary"};
static KEYWORD librarywriteformatopt[] =
{
	{"binary",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cif",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dxf",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edif",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gds",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"hpgl",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"l",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lef",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"postscript", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"quickdraw",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"skill",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"text",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_librarywriteformatp = {librarywriteformatopt,NOTOPLIST,
	NONEXTLIST,NOPARAMS, NOBACKUP, 0, " \t", "library write option", "binary"};
static KEYWORD libraryksopt[] =
{
	{"safe",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP libraryksp = {libraryksopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "option to kill without saving", "check first"};
COMCOMP us_librarykp = {NOKEYWORD, topoflibs, nextlibs, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of library to erase", "current"};
COMCOMP us_librarydp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "default path for system libraries", 0};
static KEYWORD libraryopt[] =
{
	{"use",               1,{&us_libraryup,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"new",               1,{&us_librarynp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"read",              2,{&us_libraryrp,&libraryreadformatp,NOKEY,NOKEY,NOKEY}},
	{"write",             2,{&us_librarywp,&us_librarywriteformatp,NOKEY,NOKEY,NOKEY}},
	{"save",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"purge",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"kill",              2,{&us_librarykp,&libraryksp,NOKEY,NOKEY,NOKEY}},
	{"default-path",      1,{&us_librarydp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_libraryp = {libraryopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "library control option", 0};

/* the "macbegin" command */
COMCOMP us_macparamp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "parameter to macro", 0};
COMCOMP us_macbeginnp = {NOKEYWORD, us_topofmacros, us_nextparse, NOPARAMS,
	NOBACKUP, NOFILL|INPUTOPT|INCLUDENOISE, " \t", "macro to be defined", 0};
extern COMCOMP us_macbeginop;
static KEYWORD macbeginopt[] =
{
	{"verbose",     1,{&us_macbeginop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"no-execute",  1,{&us_macbeginop,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_macbeginop = {macbeginopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "macro definition option", 0};

/* the "menu" command */
static KEYWORD menuxopt[] =
{
	{"auto",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP menuxp = {menuxopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "number of columns in menu (or 'auto')", 0};
static COMCOMP menuyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "number of rows in menu", 0};
static KEYWORD menusizeopt[] =
{
	{"size",    2,{&menuxp,&menuyp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP menusizep = {menusizeopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "menu size option", 0};
static COMCOMP menupopsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "number of entries in popup menu", 0};
static COMCOMP menupophp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "header for popup menu", 0};
static KEYWORD menupopopt[] =
{
	{"size",    1,{&menupopsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"header",  1,{&menupophp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP menupopp = {menupopopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "pop-up menu option", 0};
static COMCOMP menupopnp = {NOKEYWORD, us_topofpopupmenu, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT|INCLUDENOISE, " \t", "name of popup menu", 0};
extern COMCOMP us_userp, menusbp;
INTSML us_parammenusbp(char *i, COMCOMP *j[], char c)
{ j[0] = &menusbp; return(1); }
COMCOMP menusbp = {NOKEYWORD, us_topofpopupmenu, us_nextparse, us_parammenusbp,
	NOBACKUP, INPUTOPT, " \t", "popup menu name for menu bar", 0};
static KEYWORD menuopt[] =
{
	{"on",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"off",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"popup",      2,{&menupopnp,&menupopp,NOKEY,NOKEY,NOKEY}},
	{"dopopup",    1,{&us_userp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"setmenubar", 1,{&menusbp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"top",        1,{&menusizep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"bottom",     1,{&menusizep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"left",       1,{&menusizep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"right",      1,{&menusizep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"size",       2,{&menuxp,&menuyp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_menup = {menuopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "menu control option", 0};

/* the "mirror" command */
static KEYWORD mirrorgopt[] =
{
	{"about-grab-point",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"about-trace-point", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"sensibly",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP mirrorgp = {mirrorgopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "option to mirror about grab-point", 0};
static KEYWORD mirroropt[] =
{
	{"horizontal",    1,{&mirrorgp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vertical",      1,{&mirrorgp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_mirrorp = {mirroropt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "direction of mirroring", 0};

/* the "move" command */
static COMCOMP moveap = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "amount to move", 0};
static COMCOMP movexp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "X position to move object (or 'dialog')", 0};
static COMCOMP moveyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Y position to move object", 0};
static COMCOMP moverap = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "angle increment from current position", 0};
static KEYWORD moveopt[] =
{
	{"left",       1,{&moveap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"right",      1,{&moveap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up",         1,{&moveap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down",       1,{&moveap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"to",         2,{&movexp,&moveyp,NOKEY,NOKEY,NOKEY}},
	{"angle",      1,{&moverap,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP movep = {moveopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "object motion options", 0};

/* the "node" command */
static COMCOMP nodeexpp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "number of levels of depth to expand", "infinite"};
static COMCOMP nodeuexpp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "number of levels of depth to un-expand", "infinite"};
static COMCOMP nodenp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "local name to give node", 0};
static COMCOMP nodetaxp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "X coordinate of trace point", 0};
static COMCOMP nodetayp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Y coordinate of trace point", 0};
static COMCOMP nodetcadp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Degrees of circle (default is 360)", 0};
static COMCOMP nodetcarp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Resolution of annulus (number of segments to use)", 0};
static COMCOMP nodetcaop = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Outer radius of annulus", 0};
COMCOMP us_nodetcaip = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Inner radius of annulus", 0};
static COMCOMP nodetptmp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Message to place", 0};
static COMCOMP nodetptsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Size of text", 0};
COMCOMP us_nodetptlp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Layer of text", 0};
extern COMCOMP us_nodetp;
static KEYWORD nodetopt[] =
{
	{"store-trace",       1,{&us_nodetp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"add-point",         2,{&nodetaxp,&nodetayp,NOKEY,NOKEY,NOKEY}},
	{"move-point",        2,{&nodetaxp,&nodetayp,NOKEY,NOKEY,NOKEY}},
	{"delete-point",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"wait-for-down",     1,{&us_nodetp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"construct-annulus", 4,{&us_nodetcaip,&nodetcaop,&nodetcarp,&nodetcadp,NOKEY}},
	{"place-text",        3,{&us_nodetptlp,&nodetptsp,&nodetptmp,NOKEY,NOKEY}},
	{"fillet",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"next-point",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"prev-point",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_nodetp = {nodetopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "cursor trace option", 0};
static KEYWORD nodenotopt[] =
{
	{"expand",       1,{&nodeuexpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"name",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP nodenotp = {nodenotopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "node state NOT options", 0};
static KEYWORD nodeopt[] =
{
	{"cover-implant",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"regrid-facet",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"expand",         1,{&nodeexpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"name",           1,{&nodenp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"trace",          1,{&us_nodetp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",            1,{&nodenotp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_nodep = {nodeopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "node state options", 0};

/* the "offaid" command */
static KEYWORD offaideopt[] =
{
	{"permanently",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP offaidep = {offaideopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "option to make this change nonundoable", 0};
static COMCOMP offaidp = {NOKEYWORD, topofaids, nextaids, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "aid to turn off", 0};

/* the "onaid" command */
static KEYWORD onaideopt[] =
{
	{"no-catch-up",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP onaidep = {onaideopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "aid catch-up options", 0};
static COMCOMP onaidp = {NOKEYWORD, topofaids, nextaids, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "aid to turn on", 0};
COMCOMP us_onoffaidp = {NOKEYWORD, topofaids, nextaids, NOPARAMS, NOBACKUP,
	0, " \t", "aid to control", 0};

/* the "outhier" command */
static COMCOMP outhierp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "number of levels to pop out", "1"};

/* the "package" command */
COMCOMP us_packagep = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "name of new facet in which to package this circuitry", 0};

/* the "port" command */
COMCOMP us_portsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "which port number do you want", 0};
static COMCOMP portunp = {NOKEYWORD, us_topofexpports, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of existing exported port", 0};
extern COMCOMP portcp;
static KEYWORD portmodopt[] =
{
	{"input",         1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"output",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"bidirectional", 1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"power",         1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ground",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock",         1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock1",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock2",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock3",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock4",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock5",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock6",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"refout",        1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"refin",         1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"none",          1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"always-drawn",  1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"body-only",     1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"specify",       1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use",           2,{&portunp,&portcp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP portcp = {portmodopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "port specification and characteristics", 0};
static COMCOMP portecup = {NOKEYWORD, us_topofports, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of port to export", 0};
extern COMCOMP portecp;
static KEYWORD portmodeopt[] =
{
	{"input",         1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"output",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"bidirectional", 1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"power",         1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ground",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock",         1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock1",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock2",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock3",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock4",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock5",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"clock6",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"refout",        1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"refin",         1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"always-drawn",  1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"body-only",     1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"specify",       1,{&portecp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use",           2,{&portecup,&portecp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP portecp = {portmodeopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "port specification and characteristics", 0};
INTSML us_paramportep(char *i, COMCOMP *j[], char c)
{ j[0] = &portecp; return(1); }
COMCOMP us_portep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, us_paramportep, NOBACKUP,
	INPUTOPT, " \t", "exported name of port", 0};
static KEYWORD portuopt[] =
{
	{"all",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"specify",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use",           1,{&portunp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP portup = {portuopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "port specification", 0};
static KEYWORD portnopt[] =
{
	{"export",        1,{&portup,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP portnp = {portnopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "port un-request", 0};
static KEYWORD portlopt[] =
{
	{"none",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"short",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"long",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"crosses",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_portlp = {portlopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "port label display option", 0};
static KEYWORD portmopt[] =
{
	{"remain-highlighted", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP portmp = {portmopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "port move highlighting option", 0};
static COMCOMP portsp = {NOKEYWORD,topoflibs,nextlibs,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "library to synchronize with this", 0};
static KEYWORD portopt[] =
{
	{"export",                1,{&us_portep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",                   1,{&portnp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"re-export-all",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"highlighted-re-export", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"synchronize-library",   1,{&portsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"move",                  1,{&portmp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"change",                1,{&portcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"labels",                1,{&us_portlp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"identify",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_portp = {portopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "control of ports", 0};

/* the "quit" command */
COMCOMP us_quitp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "exit the program", 0};

/* the "rename" command */
COMCOMP us_renameop = {NOKEYWORD, us_topofallthings, us_nextparse, NOPARAMS,
	NOBACKUP, NOFILL|INPUTOPT, " \t", "name of object to be renamed", 0};
COMCOMP us_renamenp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "new name for object", 0};
COMCOMP us_renamecp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "which type of object(s) do you want renamed", 0};

/* the "replace" command */
static KEYWORD replacesopt[] =
{
	{"universally",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"this-facet",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"connected",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP replacesp = {replacesopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "replacement option", 0};
COMCOMP us_replacep = {NOKEYWORD, us_topofarcnodes, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "new prototype to use instead of this", 0};

/* the "rotate" command */
static KEYWORD rotatemopt[] =
{
	{"sensibly",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"about-grab-point",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"about-trace-point", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP rotatemp = {rotatemopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "rotation option", ""};
static KEYWORD rotateopt[] =
{
	{"sensibly",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"about-grab-point",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"about-trace-point", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"more",              1,{&rotatemp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP rotatep = {rotateopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "rotation option", ""};
static KEYWORD rotateaopt[] =
{
	{"interactively",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP rotateap = {rotateaopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "new angle or angle increment", ""};

/* the "show" command */
static COMCOMP showbpp = {NOKEYWORD, us_topofpopupmenu, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Popup menu to display", 0};
static KEYWORD showbopt[] =
{
	{"short",                 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"all",                   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"key",                   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"menu",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"button",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"popup",                 1,{&showbpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP showbp = {showbopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "binding information to show", 0};
static KEYWORD showeopt[] =
{
	{"authors",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP showep = {showeopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "option to list authors", 0};
static COMCOMP showclp = {NOKEYWORD, topoflibs, nextlibs, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "library to show", 0};
static COMCOMP showcmp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "cell names to show (with wildcards)", 0};
extern COMCOMP showcp;
static KEYWORD showcopt[] =
{
	{"library",           2,{&showclp,&showcp,NOKEY,NOKEY,NOKEY}},
	{"matching",          2,{&showcmp,&showcp,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP showcp = {showcopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "cell listing option", 0};
COMCOMP us_showdp = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of facet to show dates", 0};
static COMCOMP showfmp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet names to show (with wildcards)", 0};
static COMCOMP showfdp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "file to create with facet information", 0};
extern COMCOMP showfep;
static KEYWORD showfeopt[] =
{
	{"dates",             1,{&showfep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"library",           2,{&showclp,&showfep,NOKEY,NOKEY,NOKEY}},
	{"edit",              1,{&showfep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"matching",          2,{&showfmp,&showfep,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP showfep = {showfeopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "facet listing", 0};
static KEYWORD showfopt[] =
{
	{"graphically",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"contained-in-this", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dates",             1,{&showfep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"file",              2,{&showfdp,&showfep,NOKEY,NOKEY,NOKEY}},
	{"library",           2,{&showclp,&showfep,NOKEY,NOKEY,NOKEY}},
	{"edit",              1,{&showfep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"matching",          2,{&showfmp,&showfep,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP showfp = {showfopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "facet listing option", 0};
static COMCOMP showmp = {NOKEYWORD, us_topofmacros, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "macro to show", 0};
static KEYWORD showoopt[] =
{
	{"short",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"long",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_showop = {showoopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "amount of information to describe about found object(s)", 0};
extern COMCOMP showpp;
static KEYWORD showpopt[] =
{
	{"clock",        1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"input",        1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"output",       1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"bidirectional",1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"power",        1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ground",       1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"generic",      1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"reference",    1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"all",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP showpp = {showpopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "port type to be shown", 0};
static COMCOMP showhp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "which batch do you wish to examine", "all"};
COMCOMP us_showup = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "name of facet to show usage", 0};
static COMCOMP showvp = {NOKEYWORD, topofcells, nextcells, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "cell whose views are to be shown", 0};
static KEYWORD showopt[] =
{
	{"aids",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"bindings",              1,{&showbp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cells",                 1,{&showcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dates",                 1,{&us_showdp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"environment",           1,{&showep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"facets",                1,{&showfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"history",               1,{&showhp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"libraries",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"macros",                1,{&showmp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"networks",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"object",                1,{&us_showop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"ports",                 1,{&showpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"primitives",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"solvers",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"technologies",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"usage",                 1,{&us_showup,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"views",                 1,{&showvp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_showp = {showopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "information to show", 0};

/* the "size" command */
COMCOMP us_sizeyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "new Y size of node", 0};
COMCOMP us_sizewp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "new width of arc", 0};
INTSML paramsize(char *word, COMCOMP *arr[], char breakc)
{
	REGISTER GEOM **list;
	REGISTER INTSML i;

	if (*word == 0) return(0);

	/* if known keywords are used, no more parsing */
	if (namesame(word, "corner-fixed") == 0 || namesame(word, "center-fixed") == 0 ||
		namesame(word, "grab-point-fixed") == 0 || namesame(word, "nodes") == 0 ||
		namesame(word, "arcs") == 0 || namesame(word, "use-transformation") == 0) return(0);

	/* get list of highlighted objects */
	list = us_gethighlighted(OBJARCINST|OBJNODEINST);
	if (list[0] == NOGEOM) return(0);
	for(i=0; list[i] != NOGEOM; i++) if (list[i]->entrytype == OBJNODEINST)
	{
		/* node found: get Y size too */
		arr[0] = &us_sizeyp;
		return(1);
	}
	return(0);
}
static KEYWORD sizexopt[] =
{
	{"corner-fixed",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"center-fixed",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"grab-point-fixed",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"nodes",                   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"arcs",                    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-transformation",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_sizep = {sizexopt, NOTOPLIST, NONEXTLIST, paramsize, NOBACKUP,
	INPUTOPT, " \t", "new width of arc, new X size of node, or option", 0};

/* the "spread" command */
static KEYWORD spreaddopt[] =
{
	{"left",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"right",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_spreaddp = {spreaddopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "direction in which to spread open layout", 0};
static COMCOMP spreadap = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "distance to spread open layout", "design-rule spacing"};

/* the "system" command */
static KEYWORD systemopt[] =
{
	{"setstatusfont", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#ifdef	WIN32
	{"print",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
#endif
	TERMKEY
};
static COMCOMP systemp = {systemopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "special system-dependent option", 0};

/* the "technology" command */
static COMCOMP technologyetlp = {NOKEYWORD, us_topofedtectech, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "technology to edit", 0};
static COMCOMP technologyeppp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "name of this port", 0};
static KEYWORD technologyepopt[] =
{
	{"port",               1,{&technologyeppp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"highlight",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rectangle-filled",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rectangle-outline",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rectangle-crossed",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"polygon-filled",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"polygon-outline",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lines-solid",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lines-dotted",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lines-dashed",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lines-far-dotted",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"circle-outline",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"circle-filled",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"circle-half",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"circle-arc",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"text",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP technologyepp = {technologyepopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "shape/type of layer to create", 0};
static KEYWORD technologyclcopt[] =
{
	{"none",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"overlappable-1", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"overlappable-2", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"overlappable-3", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"overlappable-4", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"overlappable-5", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"white",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"black",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"red",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"blue",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"green",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cyan",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"magenta",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"yellow",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gray",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"orange",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"purple",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"brown",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-gray",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-gray",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-red",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-red",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-green",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-green",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light-blue",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dark-blue",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP technologyclcp = {technologyclcopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "new color of this layer", 0};
static KEYWORD technologyclsopt[] =
{
	{"solid",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"patterned",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"patterned/outlined", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_technologyclsp = {technologyclsopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "new style of this layer", 0};
COMCOMP us_technologyclip = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "CIF symbol for this layer", 0};
COMCOMP us_technologycldp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "DXF name(s) for this layer", 0};
COMCOMP us_technologyclgp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "Calma GDS-II number for this layer", 0};
COMCOMP us_technologyclsrp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "SPICE resistance for this layer", 0};
COMCOMP us_technologyclscp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "SPICE capacitance for this layer", 0};
COMCOMP us_technologyclsecp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "SPICE edge capacitance for this layer", 0};
static COMCOMP technologycldmw = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "DRC minimum width for this layer", 0};
static COMCOMP technologycl3dh = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "3D height for this layer", 0};
static COMCOMP technologycl3dt = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "3D thickness for this layer", 0};
static KEYWORD technologyclfopt[] =
{
	{"unknown",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"metal-1",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"metal-2",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"metal-3",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"metal-4",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"metal-5",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"metal-6",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"metal-7",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"metal-8",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"poly-1",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"poly-2",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"poly-3",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"gate",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"diffusion",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"implant",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"contact-1",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"contact-2",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"contact-3",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"contact-4",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"contact-5",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"contact-6",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"plug",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"overglass",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"resistor",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"capacitor",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"transistor",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"emitter",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"base",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"collector",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"substrate",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"well",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"guard",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"isolation",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"bus",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"art",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"control",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"p-type",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"n-type",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"depletion",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"enhancement",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"light",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"heavy",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pseudo",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"nonelectrical", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP technologyclfp = {technologyclfopt, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "new function for this layer", 0};
static COMCOMP technologycllp = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "letters to identify this layer", 0};
static COMCOMP technologyclasp = {NOKEYWORD, us_topofedtecslay, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "new layer for this patch", 0};
static COMCOMP technologyclacp = {NOKEYWORD, us_topofedtecclay, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "new layer for this patch", 0};
static COMCOMP technologycafp = {NOKEYWORD, us_topofarcfun, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "new function for this arc", 0};
static COMCOMP technologycamp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "is this arc fixed-angle by default", 0};
static COMCOMP technologycawp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "does this arc erase connecting pins", 0};
static COMCOMP technologycanp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "does this arc extend beyond its endpoints", 0};
static COMCOMP technologycaip = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "prefered angle increment for this arc", 0};
static COMCOMP technologycnfp = {NOKEYWORD, us_topofnodefun, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "new function for this node", 0};
COMCOMP us_technologycnsp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "is this node a serpentine transistor", 0};
static COMCOMP technologycnsqp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "is this node square", 0};
static COMCOMP technologycnwp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "is this node invisible with 1 or 2 arcs", 0};
static COMCOMP technologycnlp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "can this node be locked", 0};
static COMCOMP technologyctlp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "new value of Lambda for this technology", 0};
COMCOMP us_technologyctdp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "new description of this technology", 0};
static COMCOMP technologycpbp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "new value", "       "};
static COMCOMP technologycpynp = {yesnoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "new value", "yes"};
INTSML us_paramsedtecport(char *word, COMCOMP *arr[], char breakc)
{
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER INTSML len, j;
	HIGHLIGHT high;
	static char technologycpynpdef[4];
	static char technologycpbpdef[20];

	if (*word == 0) return(0);

	/* get the currently highlighted node */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(0);
	if (getlength(var) != 1) return(0);
	(void)us_makehighlight(((char **)var->addr)[0], &high);
	if (((high.status&HIGHTYPE) != HIGHFROM && (high.status&HIGHTYPE) != HIGHTEXT)) return(0);
	if (high.fromgeom->entrytype != OBJNODEINST) return(0);
	ni = high.fromgeom->entryaddr.ni;

	if (namesame(word, "PORT-ANGLE") == 0)
	{
		technologycpbp.def = technologycpbpdef;
		var = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_portangle");
		if (var != NOVARIABLE) (void)sprintf(technologycpbp.def, "%ld", var->addr); else
			(void)strcpy(technologycpbp.def, "0");
		arr[0] = &technologycpbp;
		return(1);
	}

	if (namesame(word, "PORT-ANGLE-RANGE") == 0)
	{
		technologycpbp.def = technologycpbpdef;
		var = getval((INTBIG)ni, VNODEINST, VINTEGER, "EDTEC_portrange");
		if (var != NOVARIABLE) (void)sprintf(technologycpbp.def, "%ld", var->addr); else
			(void)strcpy(technologycpbp.def, "180");
		arr[0] = &technologycpbp;
		return(1);
	}

	/* see if this is an arc that was named */
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		if (namesamen(np->cell->cellname, "arc-", 4) == 0 &&
			namesame(&np->cell->cellname[4], word) == 0) break;
	if (np == NONODEPROTO) return(0);

	/* get information about this node */
	technologycpynp.def = technologycpynpdef;
	(void)strcpy(technologycpynp.def, "no");
	arr[0] = &technologycpynp;
	var = getval((INTBIG)ni, VNODEINST, VNODEPROTO|VISARRAY, "EDTEC_connects");
	if (var != NOVARIABLE)
	{
		len = (INTSML)getlength(var);
		for(j=0; j<len; j++) if (((NODEPROTO **)var->addr)[j] == np)
		{
			(void)strcpy(technologycpynp.def, "yes");
			break;
		}
	}
	return(1);
}
static COMCOMP technologycpp = {NOKEYWORD, us_topofedtecport, us_nextparse,
	us_paramsedtecport, NOBACKUP, INPUTOPT|MULTIOPT, " \t", "options for this port", 0};
COMCOMP us_technologyeenp = {NOKEYWORD, us_topofedtecnode, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT|INCLUDENOISE, " \t", "node in technology to edit", 0};
COMCOMP us_technologyeeap = {NOKEYWORD, us_topofedtecarc, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT|INCLUDENOISE, " \t", "arc in technology to edit", 0};
COMCOMP us_technologyeelp = {NOKEYWORD, us_topofedteclay, us_nextparse,
	NOPARAMS, NOBACKUP, INPUTOPT|INCLUDENOISE, " \t", "layer in technology to edit", 0};
COMCOMP us_technologyedlp = {NOKEYWORD,topoflibs,nextlibs,us_nexttechnologyedlp,
	NOBACKUP, 0, " \t", "argument to be echoed", 0};
INTSML us_nexttechnologyedlp(char *i, COMCOMP *j[], char c)
{ j[0] = &us_technologyedlp; return(1); }
static KEYWORD technologyeopt[] =
{
	{"library-to-tech-and-C", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"library-to-tech",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tech-to-library",       1,{&technologyetlp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"inquire-layer",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"place-layer",           1,{&technologyepp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"change",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edit-node",             1,{&us_technologyeenp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edit-arc",              1,{&us_technologyeeap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edit-layer",            1,{&us_technologyeelp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edit-subsequent",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edit-colors",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edit-design-rules",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edit-misc-information", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"identify-layers",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"identify-ports",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dependent-libraries",   1,{&us_technologyedlp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
INTSML paramtechnology(char *word, COMCOMP *arr[], char breakc)
{
	REGISTER NODEINST *ni;
	REGISTER INTSML ind, i, count;
	HIGHLIGHT high;
	REGISTER VARIABLE *var;

	if (*word == 0) return(0);
	if (namesame(word, "change") != 0)
	{
		/* handle normal parameter addition if not the "change" option */
		for(ind=0; technologyeopt[ind].name != 0; ind++)
			if (namesame(word, technologyeopt[ind].name) == 0) break;
		if (technologyeopt[ind].name == 0) return(0);
		count = technologyeopt[ind].params;
		for(i=0; i<count; i++) arr[i] = technologyeopt[ind].par[i];
		return(count);
	}

	/* get the currently highlighted node */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(0);
	if (getlength(var) != 1) return(0);
	(void)us_makehighlight(((char **)var->addr)[0], &high);
	if (((high.status&HIGHTYPE) != HIGHFROM && (high.status&HIGHTYPE) != HIGHTEXT)) return(0);
	if (high.fromgeom->entrytype != OBJNODEINST) return(0);
	ni = high.fromgeom->entryaddr.ni;
	switch (us_tecedgetoption(ni))
	{
		case ARCFIXANG:      arr[0] = &technologycamp;      return(1);
		case ARCFUNCTION:    arr[0] = &technologycafp;      return(1);
		case ARCINC:         arr[0] = &technologycaip;      return(1);
		case ARCNOEXTEND:    arr[0] = &technologycanp;      return(1);
		case ARCWIPESPINS:   arr[0] = &technologycawp;      return(1);
		case LAYER3DHEIGHT:  arr[0] = &technologycl3dh;     return(1);
		case LAYER3DTHICK:   arr[0] = &technologycl3dt;     return(1);
		case LAYERCIF:       arr[0] = &us_technologyclip;   return(1);
		case LAYERCOLOR:     arr[0] = &technologyclcp;      return(1);
		case LAYERDXF:       arr[0] = &us_technologycldp;   return(1);
		case LAYERDRCMINWID: arr[0] = &technologycldmw;     return(1);
		case LAYERFUNCTION:  arr[0] = &technologyclfp;      return(1);
		case LAYERGDS:       arr[0] = &us_technologyclgp;   return(1);
		case LAYERLETTERS:   arr[0] = &technologycllp;      return(1);
		case LAYERPATCH:
			var = getval((INTBIG)ni, VNODEINST, VSTRING, "EDTEC_minbox");
			if (var == NOVARIABLE) arr[0] = &technologyclasp; else
				arr[0] = &technologyclacp;
			return(1);
		case LAYERSPICAP:    arr[0] = &us_technologyclscp;  return(1);
		case LAYERSPIECAP:   arr[0] = &us_technologyclsecp; return(1);
		case LAYERSPIRES:    arr[0] = &us_technologyclsrp;  return(1);
		case LAYERSTYLE:     arr[0] = &us_technologyclsp;   return(1);
		case NODEFUNCTION:   arr[0] = &technologycnfp;      return(1);
		case NODELOCKABLE:   arr[0] = &technologycnlp;      return(1);
		case NODESERPENTINE: arr[0] = &us_technologycnsp;   return(1);
		case NODESQUARE:     arr[0] = &technologycnsqp;     return(1);
		case NODEWIPES:      arr[0] = &technologycnwp;      return(1);
		case PORTOBJ:        arr[0] = &technologycpp;       return(1);
		case TECHDESCRIPT:   arr[0] = &us_technologyctdp;   return(1);
		case TECHLAMBDA:     arr[0] = &technologyctlp;      return(1);
	}
	return(0);
}
static COMCOMP technologyep = {technologyeopt, NOTOPLIST, NONEXTLIST,
	paramtechnology, NOBACKUP, INPUTOPT, " \t", "technology editing option", 0};
COMCOMP us_technologyup = {NOKEYWORD, topoftechs, nexttechs, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "new technology to use", 0};
static COMCOMP technologykp = {NOKEYWORD, topoftechs, nexttechs, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "technology to kill", 0};
static COMCOMP technologycp = {NOKEYWORD, topoftechs, nexttechs, NOPARAMS,
	NOBACKUP, 0, " \t", "technology to which conversion is desired", 0};
static COMCOMP technologyap = {NOKEYWORD, NOTOPLIST, NONEXTLIST,
	us_paramtechnologyb, NOBACKUP, 0, " \t", "message for technology", 0};
INTSML us_paramtechnologyb(char *i, COMCOMP *j[], char c)
{ j[0]= &technologyap; return(1); }
INTSML us_paramtechnologya(char *pt, COMCOMP *j[], char c)
{
	REGISTER TECHNOLOGY *t;

	t = gettechnology(pt);
	if (t != NOTECHNOLOGY)
	{
		j[0] = t->parse;
		if (j[0] != NOCOMCOMP) return(1);
	}
	return(us_paramtechnologyb(pt, j, c));
}
COMCOMP us_technologytp = {NOKEYWORD, topoftechs, nexttechs,
	us_paramtechnologya, NOBACKUP, INPUTOPT, " \t", "technology to direct", 0};
static COMCOMP technologydp = {NOKEYWORD, topoftechs, nexttechs,
	NOPARAMS, NOBACKUP, INPUTOPT, " \t", "technology to document", 0};
static KEYWORD technologyopt[] =
{
	{"use",      1,{&us_technologyup,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"convert",  1,{&technologycp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tell",     1,{&us_technologytp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"edit",     1,{&technologyep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"kill",     1,{&technologykp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"document", 1,{&technologydp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_technologyp = {technologyopt,NOTOPLIST,NONEXTLIST,NOPARAMS,
	NOBACKUP, 0, " \t", "technology option", 0};

/* the "tellaid" command */
static COMCOMP tellaidap = {NOKEYWORD,NOTOPLIST,NONEXTLIST,us_paramtellaidb,
	NOBACKUP, 0, " \t", "message to be sent to aid", 0};
INTSML us_paramtellaidb(char *i, COMCOMP *j[], char c)
{ j[0]= &tellaidap; return(1); }
INTSML us_paramtellaida(char *pt, COMCOMP *j[], char c)
{
	REGISTER INTSML i;

	i = parse(pt, &offaidp, 0);
	if (i >= 0)
	{
		j[0] = el_aids[i].parse;
		if (j[0] != NOCOMCOMP) return(1);
	}
	return(us_paramtellaidb(pt, j, c));
}
COMCOMP us_tellaidp = {NOKEYWORD, topofaids, nextaids, us_paramtellaida,
	NOBACKUP, INPUTOPT, " \t", "aid to instruct", 0};
COMCOMP us_userp = {NOKEYWORD,us_topofcommands,us_nextparse,us_paramcommands,
	NOBACKUP, 0, " \t", "Full Electric command", 0};

/* the "terminal" command */
static COMCOMP terminalvp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "variable letter to fill", 0};
static COMCOMP terminalpp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "prompt message", 0};
static COMCOMP terminaltp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "type of input ('facet', etc.)", 0};
static COMCOMP terminalsfp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "number of keystrokes between recording checkpoints", 0};
static COMCOMP terminalspp = {NOKEYWORD, topoffile, nextfile, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "session playback file", 0};
static KEYWORD terminalspaopt[] =
{
	{"all",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP terminalspap = {terminalspaopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "session playback option", 0};
static KEYWORD terminalsopt[] =
{
	{"playback",             2,{&terminalspp,&terminalspap,NOKEY,NOKEY,NOKEY}},
	{"begin-record",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"end-record",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rewind-record",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"checkpoint-frequency", 1,{&terminalsfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP terminalsp = {terminalsopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "session logging option", 0};
static KEYWORD terminalnopt[] =
{
	{"lock-keys-on-error",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"only-informative-messages",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-electric-commands",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"display-dialogs",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"permanent-menu-highlighting", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"track-cursor-coordinates",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"beep",                        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP terminalnp = {terminalnopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "terminal option to turn off", 0};
static COMCOMP terminalmp = {us_onoffopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "scrolling pause option", 0};
static KEYWORD terminalopt[] =
{
	{"input",                       3,{&terminalvp,&terminalpp,&terminaltp,NOKEY,NOKEY}},
	{"more",                        1,{&terminalmp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"session",                     1,{&terminalsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"not",                         1,{&terminalnp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lock-keys-on-error",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"only-informative-messages",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use-electric-commands",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"display-dialogs",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"permanent-menu-highlighting", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"track-cursor-coordinates",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"beep",                        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP terminalp = {terminalopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "terminal control option", 0};

/* the "text" command */
static KEYWORD textsopt[] =
{
	{"4p",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"6p",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"8p",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"10p",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"12p",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"14p",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"16p",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"18p",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"20p",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"small",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"medium",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"large",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP textsp = {textsopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "New size for highlighted text", 0};
COMCOMP us_textdsp = {textsopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Size for new text", 0};
KEYWORD us_texttopt[] =
{
	{"centered",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"boxed",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"left",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"right",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up-left",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up-right",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down-left",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down-right",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static KEYWORD textssopt[] =
{
	{"none",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"inside",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"outside",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP texttp = {us_texttopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "New style for highlighted text", 0};
static COMCOMP textdtp = {us_texttopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Style for new text", 0};
static COMCOMP textdhp = {textssopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Horizontal style for new text", 0};
static COMCOMP textdvp = {textssopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Vertical style for new text", 0};
static COMCOMP textep = {NOKEYWORD, us_topofeditor, us_nexteditor, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "Editor to use for text", 0};
static COMCOMP textrp = {NOKEYWORD, topoffile, nextfile, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "File to read into this facet", 0};
static COMCOMP textwp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "File to write with this facet", 0};
static COMCOMP textfp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "String to find in text window", 0};
static KEYWORD textopt[] =
{
	{"style",                    1,{&texttp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"size",                     1,{&textsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"default-style",            1,{&textdtp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"default-horizontal-style", 1,{&textdhp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"default-vertical-style",   1,{&textdvp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"default-size",             1,{&us_textdsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"editor",                   1,{&textep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"write",                    1,{&textwp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"read",                     1,{&textrp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cut",                      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"copy",                     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"paste",                    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"find",                     1,{&textfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"easy-text-selection",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"hard-text-selection",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP textp = {textopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "text manipulation option", 0};

/* the "undo" command */
static COMCOMP undorp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "number of changes to redo", "redo 1 major change"};
static COMCOMP undosp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "number of changes to save", 0};
static KEYWORD undoopt[] =
{
	{"redo",          1,{&undorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"save",          1,{&undosp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP undop = {undoopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "number of changes to undo, or undo control option", "undo 1 major change"};

/* the "var" command */
static COMCOMP varmoddxp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "X offset of variable", 0};
static COMCOMP varmoddyp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "Y offset of variable", 0};
static COMCOMP varmoddsp = {us_texttopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "Text style of variable", 0};
static KEYWORD varmodopt[] =
{
	{"lisp",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tcl",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"java",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"display",           3,{&varmoddxp,&varmoddyp,&varmoddsp,NOKEY,NOKEY}},
	{"temporary",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"fractional",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cannot-change",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP varmodp = {varmodopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "modifier to the variable", 0};
static KEYWORD varhcnopt[] =
{
	{"language",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"display",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"temporary",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cannot-change",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP varhcnp = {varhcnopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "how to unchange the variable", 0};
static KEYWORD varhcopt[] =
{
	{"not",               1,{&varhcnp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lisp",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tcl",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"java",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"display",           3,{&varmoddxp,&varmoddyp,&varmoddsp,NOKEY,NOKEY}},
	{"temporary",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cannot-change",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP varhcp = {varhcopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "how to change the variable", 0};
COMCOMP us_varvalp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "new value of variable", 0};
COMCOMP us_varvalcp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "this variable cannot be set!", 0};
static COMCOMP varupdp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "modification to variable", 0};
COMCOMP us_varvsp = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT|INCLUDENOISE, " \t", "variable name to set", 0};
COMCOMP us_varvep = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT, " \t", "variable name to examine", 0};
static COMCOMP varvcp = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS,
	NOBACKUP, NOFILL|INPUTOPT, " \t", "variable name to change", 0};
static COMCOMP varvpp = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "variable to pick in menu", 0};
static COMCOMP varvtp = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS,
	NOBACKUP, INPUTOPT|INCLUDENOISE, " \t", "variable to edit in window", 0};
static COMCOMP varqhp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	0, " \t", "header string for edit window", 0};
static KEYWORD varqopt[] =
{
	{"in-place",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"header",              1,{&varqhp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_varqp = {varqopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "text editing option", 0};
COMCOMP us_varvdp = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS, NOBACKUP,
	NOFILL, " \t", "variable name to delete", 0};
static COMCOMP varvmp = {NOKEYWORD,us_topofvars,us_nextvars,NOPARAMS,NOBACKUP,
	NOFILL|INPUTOPT, " \t", "variable name to modify", 0};
static COMCOMP varbdp = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT|INCLUDENOISE, " \t", "variable name to set", 0};
static COMCOMP varbs1p = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT|INCLUDENOISE, " \t", "first operand variable name", 0};
static COMCOMP varbs2p = {NOKEYWORD, us_topofvars, us_nextvars, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT|INCLUDENOISE, " \t", "second operand variable name", 0};
static COMCOMP varbs2tp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT, " \t", "pattern matching string", 0};
static KEYWORD varbs2tyopt[] =
{
	{"unknown",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"integer",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"short",                 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"fixed-point",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"address",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"character",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"string",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"float",                 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"double",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"nodeinst",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"nodeproto",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cell",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"portarcinst",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"portexpinst",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"portproto",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"arcinst",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"arcproto",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"geometry",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"r-tree",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"library",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"technology",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"aid",                   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"network",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"view",                  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"window",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"graphics",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"constraint",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"general",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_varbs2typ = {varbs2tyopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	NOFILL|INPUTOPT, " \t", "type to request", 0};
static KEYWORD varbopt[] =
{
	{"set",                 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"concat",              1,{&varbs2p,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"type",                1,{&us_varbs2typ,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"select",              1,{&varbs2p,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pattern",             1,{&varbs2tp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP varbop = {varbopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "operator to perform on variables", 0};
static KEYWORD varoopt[] =
{
	{"ignore",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"track",               0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP varop = {varoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "options control", 0};
static KEYWORD varopt[] =
{
	{"options",           1,{&varop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"set",               3,{&us_varvsp,&us_varvalp,&varmodp,NOKEY,NOKEY}},
	{"examine",           1,{&us_varvep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"change",            2,{&varvcp,&varhcp,NOKEY,NOKEY,NOKEY}},
	{"delete",            1,{&us_varvdp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pick",              1,{&varvpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"textedit",          2,{&varvtp,&us_varqp,NOKEY,NOKEY,NOKEY}},
	{"vector",            3,{&varbdp,&varbs1p,&varbop,NOKEY,NOKEY}},
	{"+",                 3,{&varvmp,&varupdp,&varmodp,NOKEY,NOKEY}},
	{"-",                 3,{&varvmp,&varupdp,&varmodp,NOKEY,NOKEY}},
	{"*",                 3,{&varvmp,&varupdp,&varmodp,NOKEY,NOKEY}},
	{"/",                 3,{&varvmp,&varupdp,&varmodp,NOKEY,NOKEY}},
	{"mod",               3,{&varvmp,&varupdp,&varmodp,NOKEY,NOKEY}},
	{"and",               3,{&varvmp,&varupdp,&varmodp,NOKEY,NOKEY}},
	{"or",                3,{&varvmp,&varupdp,&varmodp,NOKEY,NOKEY}},
	{"|",                 3,{&varvmp,&varupdp,&varmodp,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_varp = {varopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "option for manipulating variables", 0};

/* the "view" command */
static COMCOMP viewc1p = {NOKEYWORD, us_topoffacets, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "facet name whose view is to be changed", 0};
COMCOMP us_viewc2p = {NOKEYWORD, topofviews, nextviews, NOPARAMS,
	NOBACKUP, 0, " \t", "new view for the facet", 0};
static COMCOMP viewdp = {NOKEYWORD, topofviews, nextviews, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "view name to delete", 0};
static KEYWORD viewvhopt[] =
{
	{"vertical",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP viewvhp = {viewvhopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "option to rotate frame", 0};
static KEYWORD viewfopt[] =
{
	{"A",                 1,{&viewvhp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"D",                 1,{&viewvhp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"none",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_viewfp = {viewfopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "frame size to use for this facet", 0};
COMCOMP us_viewn1p = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	INPUTOPT, " \t", "new view name", 0};
static COMCOMP viewn2p = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "abbreviation for new view name", 0};
static KEYWORD viewtopt[] =
{
	{"text",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP viewn3p = {viewtopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "type of view (graphical)", 0};
static KEYWORD viewopt[] =
{
	{"new",               3,{&us_viewn1p,&viewn2p,&viewn3p,NOKEY,NOKEY}},
	{"delete",            1,{&viewdp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"change",            2,{&viewc1p,&us_viewc2p,NOKEY,NOKEY,NOKEY}},
	{"frame",             1,{&us_viewfp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_viewp = {viewopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "option for manipulating cell views", 0};

/* the "visiblelayers" command */
static KEYWORD visiblelayersnoptp[] =
{
	{"no-list",           0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP visiblelayersnp = {visiblelayersnoptp, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "option to supress layer listing", 0};
INTSML us_paramvisiblelayers(char *i, COMCOMP *j[], char c)
{ j[0] = &visiblelayersnp; return(1);}
COMCOMP us_visiblelayersp = {NOKEYWORD, us_topoflayers, us_nextparse,
	us_paramvisiblelayers, NOBACKUP, NOFILL|INPUTOPT, " \t",
		"layers to be made visible (* for all)", "show layers"};

/* the "window" command */
static KEYWORD windowspopt[] =
{
	{"horizontal",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"vertical",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP windowspp = {windowspopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "way to split the window", 0};
COMCOMP us_windowmp = {NOKEYWORD, us_topofwindows, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "other window whose scale to match", 0};
COMCOMP us_windowup = {NOKEYWORD, us_topofwindows, us_nextparse, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "window to use", 0};
static COMCOMP windowap = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "amount to move window", 0};
static COMCOMP windowsp = {NOKEYWORD,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "amount to scale", 0};
static KEYWORD windowzopt[] =
{
	{"integral",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"nonintegral",  0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP windowzp = {windowzopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "option to force window scale to align with pixels",
		"show current state"};
static KEYWORD windownopt[] =
{
	{"standard",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pen",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tee",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP windownp = {windownopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "default cursor to use in window", 0};
static KEYWORD windowtopt[] =
{
	{"draw",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"hash-out",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP windowtp = {windowtopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "how to draw tiny facets in window", 0};
static COMCOMP windowdp = {us_onoffopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "mode in which cursor-based operations drag objects",
		"show current state"};
COMCOMP us_windowrnamep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "view name", 0};
static COMCOMP windowsnamep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, NOFILL|INPUTOPT|INCLUDENOISE, " \t", "view name", 0};
static KEYWORD windowoopt[] =
{
	{"on",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"off",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP windowop = {windowoopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "whether to allow overlappable layers", 0};
static KEYWORD windowsbaopt[] =
{
	{"align",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"angle",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"arc",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"facet",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"grid",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lambda",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"network",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"node",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"package",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"part",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"project",      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"root",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"selection",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"size",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"technology",   0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"x",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"y",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP windowsbap = {windowsbaopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "area in status bar", 0};
static COMCOMP windowsblp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "line number (1-based)", 0};
static COMCOMP windowsbsp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "starting percentage of position on line", 0};
static COMCOMP windowsbep = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "ending percentage of position on line", 0};
static COMCOMP windowsbtp = {NOKEYWORD, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "title of status field", 0};
static KEYWORD windowsbcopt[] =
{
	{"persistent",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"temporary",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP windowsbcp = {windowsbcopt,NOTOPLIST,NONEXTLIST,NOPARAMS,NOBACKUP,
	INPUTOPT, " \t", "persistence of current node in status bar", ""};
static KEYWORD windowsbopt[] =
{
	{"add",          5,{&windowsbap,&windowsblp,&windowsbsp,&windowsbep,&windowsbtp}},
	{"delete",       1,{&windowsbap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"current-node", 1,{&windowsbcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
static COMCOMP windowsbp = {windowsbopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "commands for status bar", 0};
static KEYWORD window3dopt[] =
{
	{"begin",        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"end",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rotate",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"zoom",         0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"pan",          0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_window3dp = {window3dopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, INPUTOPT, " \t", "commands for 3D display", 0};
static KEYWORD windowopt[] =
{
	{"1-window",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"3-dimensional",       1,{&us_window3dp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"all-displayed",       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"center-highlight",    0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"cursor-centered",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"delete",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"down",                1,{&windowap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"dragging",            1,{&windowdp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"highlight-displayed", 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"in-zoom",             1,{&windowsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"join",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"kill",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"left",                1,{&windowap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"match",               1,{&us_windowmp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"measure",             0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"name",                1,{&windowsnamep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"new",                 0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"normal-cursor",       2,{&windownp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"out-zoom",            1,{&windowsp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"overlappable-display",1,{&windowop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"overview",            0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"peek",                0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"right",               1,{&windowap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"save",                1,{&us_windowrnamep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"split",               1,{&windowspp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"status-bar",          1,{&windowsbp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tiny-facets",         1,{&windowtp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"trace-displayed",     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"use",                 1,{&us_windowup,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"up",                  1,{&windowap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"zoom-scale",          1,{&windowzp,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP us_windowp = {windowopt, NOTOPLIST, NONEXTLIST, NOPARAMS, NOBACKUP,
	0, " \t", "window display control option", 0};

/* table of commands with routines */
struct commandinfo us_lcommand[] =
{
	{"arc",          us_arc,           1,{&us_arcp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"array",        us_array,         5,{&us_arrayxp,&arrayyp,&arrayxop,&arrayyop, &arrayswitchp}},
	{"bind",         us_bind,          1,{&us_bindp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"color",        us_color,         1,{&us_colorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"commandfile",  us_commandfile,   2,{&us_commandfilep,&commandfileswitchp,NOKEY,NOKEY,NOKEY}},
	{"constraint",   us_constraint,    1,{&constraintp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"copyfacet",    us_copyfacet,     3,{&us_copyfacetp,&copyfacetdp,&copyfacetqp,NOKEY,NOKEY}},
	{"create",       us_create,        1,{&createp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"debug",        us_debug,         1,{&debugp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"defarc",       us_defarc,        2,{&us_defarcsp,&defarcp,NOKEY,NOKEY,NOKEY}},
	{"defnode",      us_defnode,       2,{&us_defnodesp,&us_defnodep,NOKEY,NOKEY,NOKEY}},
	{"duplicate",    us_duplicate,     0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"echo",         us_echo,          1,{&echop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"editfacet",    us_editfacet,     2,{&us_editfacetp,&editfacetwindowp,NOKEY,NOKEY,NOKEY}},
	{"erase",        us_erase,         1,{&erasep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"find",         us_find,          1,{&us_findp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"getproto",     us_getproto,      2,{&us_getproto1p,&getproto2p,NOKEY,NOKEY,NOKEY}},
	{"grid",         us_grid,          1,{&us_gridp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"help",         us_help,          1,{&us_helpp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"if",           us_if,            4,{&iftp,&ifrp,&iftp,&us_userp,NOKEY}},
	{"interpret",    us_interpret,     1,{&interpretp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"iterate",      us_iterate,       2,{&iteratep,&iteratemp,NOKEY,NOKEY,NOKEY}},
	{"killfacet",    us_killfacet,     1,{&us_killfacetp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"lambda",       us_lambda,        1,{&us_lambdap,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"library",      us_library,       1,{&us_libraryp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"macbegin",     us_macbegin,      2,{&us_macbeginnp,&us_macbeginop,NOKEY,NOKEY,NOKEY}},
	{"macdone",      us_macdone,       0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"macend",       us_macend,        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"menu",         us_menu,          1,{&us_menup,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"mirror",       us_mirror,        1,{&us_mirrorp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"move",         us_move,          1,{&movep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"node",         us_node,          1,{&us_nodep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"offaid",       us_offaid,        2,{&offaidp,&offaidep,NOKEY,NOKEY,NOKEY}},
	{"onaid",        us_onaid,         2,{&onaidp,&onaidep,NOKEY,NOKEY,NOKEY}},
	{"outhier",      us_outhier,       1,{&outhierp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"package",      us_package,       1,{&us_packagep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"port",         us_port,          1,{&us_portp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"quit",         us_quit,          1,{&us_quitp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"redraw",       us_redraw,        0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"rename",       us_rename,        3,{&us_renameop,&us_renamenp,&us_renamecp,NOKEY,NOKEY}},
	{"replace",      us_replace,       2,{&us_replacep,&replacesp,NOKEY,NOKEY,NOKEY}},
	{"rotate",       us_rotate,        2,{&rotateap,&rotatep,NOKEY,NOKEY,NOKEY}},
	{"show",         us_show,          1,{&us_showp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"size",         us_size,          1,{&us_sizep,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"spread",       us_spread,        2,{&us_spreaddp,&spreadap,NOKEY,NOKEY,NOKEY}},
	{"system",       us_system,        1,{&systemp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"technology",   us_technology,    1,{&us_technologyp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"tellaid",      us_tellaid,       1,{&us_tellaidp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"terminal",     us_terminal,      1,{&terminalp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"text",         us_text,          1,{&textp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"undo",         us_undo,          1,{&undop,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"var",          us_var,           1,{&us_varp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"view",         us_view,          1,{&us_viewp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"visiblelayers",us_visiblelayers, 1,{&us_visiblelayersp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"window",       us_window,        1,{&us_windowp,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"yanknode",     us_yanknode,      0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{NULL, NULL, 0, {NULL, NULL, NULL, NULL, NULL}} /* 0 */
};
