/*
 * Electric(tm) VLSI Design System
 *
 * File: dblangtcl.c
 * TCL interface module
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "config.h"
#if LANGTCL

#include "global.h"
#include "dblang.h"
#include "database.h"

#define BUFSIZE 500

       char       *tcl_outputbuffer, *tcl_outputloc;
static char       *tcl_inputbuffer;
       Tcl_Interp *tcl_interp = 0;

/* prototypes for local routines */
static void   tcl_dumpoutput(char *str);
static void   tcl_converttoanyelectric(char *tclstr, INTBIG *addr, INTBIG *type);
static void   tcl_converttoscalartcl(void *infstr, INTBIG addr, INTBIG type);
static char  *tcl_converttotcl(INTBIG addr, INTBIG type);
static INTBIG tcl_converttowarnedelectric(char *name, INTBIG want, char *msg);
static int    tcl_tkconsoleclose(ClientData instanceData, Tcl_Interp *interp);
static int    tcl_tkconsoleinput(ClientData instanceData, char *buf, int bufSize, int *errorCode);
static int    tcl_tkconsoleoutput(ClientData instanceData, char *buf, int toWrite, int *errorCode);
static void   tcl_tkconsolewatch(ClientData instanceData, int mask);

static int tcl_curlib(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_curtech(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getval(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getparentval(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_P(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_PD(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_PAR(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_PARD(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_setval(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_setind(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_delval(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_initsearch(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_nextobject(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_gettool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_maxtool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_indextool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_toolturnon(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_toolturnoff(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getlibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_newlibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_killlibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_eraselibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_selectlibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getnodeproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_newnodeproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_killnodeproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_copynodeproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_iconview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_contentsview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_newnodeinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_modifynodeinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_killnodeinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_replacenodeinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_nodefunction(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_newarcinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_modifyarcinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_killarcinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_replacearcinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_newportproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_portposition(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getportproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_killportproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_moveportproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_undoabatch(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_noundoallowed(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_newview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_killview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_telltool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_asktool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getarcproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getcell(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_gettechnology(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getpinproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);
static int tcl_getnetwork(ClientData dummy, Tcl_Interp *interp, int argc, char **argv);

/************************* TCL: ELECTRIC ROUTINES *************************/
/*
 * Routine to free all memory associated with this module.
 */
void db_freetclmemory(void)
{
	efree((char *)tcl_outputbuffer);
	efree((char *)tcl_inputbuffer);
}

void el_tclinterpreter(Tcl_Interp *interp)
{
	Tcl_Channel consoleChannel;
	char *path;
	static Tcl_ChannelType consoleChannelType =
	{
		"console",				/* Type name. */
		NULL,					/* Always non-blocking.*/
		tcl_tkconsoleclose,		/* Close proc. */
		tcl_tkconsoleinput,		/* Input proc. */
		tcl_tkconsoleoutput,	/* Output proc. */
		NULL,					/* Seek proc. */
		NULL,					/* Set option proc. */
		NULL,					/* Get option proc. */
		tcl_tkconsolewatch,		/* Watch for events on console. */
		NULL,					/* Are events present? */
	};
	REGISTER void *infstr;

	/* remember the interpreter */
	tcl_interp = interp;

	/* allocate terminal I/O buffers */
	tcl_outputbuffer = (char *)emalloc(BUFSIZE, db_cluster);
	tcl_inputbuffer = (char *)emalloc(BUFSIZE, db_cluster);
	if (tcl_outputbuffer == 0 || tcl_inputbuffer == 0) return;
	tcl_outputloc = tcl_outputbuffer;

	/* set console channels */
	consoleChannel = Tcl_CreateChannel(&consoleChannelType, "console0", (ClientData)TCL_STDIN, TCL_READABLE);
	if (consoleChannel != NULL)
	{
		Tcl_SetChannelOption(NULL, consoleChannel, "-translation", "lf");
		Tcl_SetChannelOption(NULL, consoleChannel, "-buffering", "none");
	}
	Tcl_SetStdChannel(consoleChannel, TCL_STDIN);
	consoleChannel = Tcl_CreateChannel(&consoleChannelType, "console1", (ClientData)TCL_STDOUT, TCL_WRITABLE);
	if (consoleChannel != NULL)
	{
		Tcl_SetChannelOption(NULL, consoleChannel, "-translation", "lf");
		Tcl_SetChannelOption(NULL, consoleChannel, "-buffering", "none");
	}
	Tcl_SetStdChannel(consoleChannel, TCL_STDOUT);
	consoleChannel = Tcl_CreateChannel(&consoleChannelType, "console2", (ClientData)TCL_STDERR, TCL_WRITABLE);
	if (consoleChannel != NULL)
	{
		Tcl_SetChannelOption(NULL, consoleChannel, "-translation", "lf");
		Tcl_SetChannelOption(NULL, consoleChannel, "-buffering", "none");
	}
	Tcl_SetStdChannel(consoleChannel, TCL_STDERR);

	/* set the system version */
	if (Tcl_SetVar(tcl_interp, "el_version", el_version, TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
	{
		ttyputerr(_("Tcl_SetVar failed: %s"), tcl_interp->result);
		ttyputmsg(_("couldn't set el_version variable"));
	}

	/* set the Electric library directory */
	if (Tcl_SetVar(tcl_interp, "el_libdir", el_libdir, TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
	{
		ttyputerr(_("Tcl_SetVar failed: %s"), tcl_interp->result);
		ttyputmsg(_("couldn't set el_libdir variable"));
	}

	/* set the Electric LISP directory */
	infstr = initinfstr();
	addstringtoinfstr(infstr, el_libdir);
	addstringtoinfstr(infstr, "lisp");
	addtoinfstr(infstr, DIRSEP);
	if (Tcl_SetVar(tcl_interp, "el_lispdir", returninfstr(infstr), TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
	{
		ttyputerr(_("Tcl_SetVar failed: %s"), tcl_interp->result);
		ttyputmsg(_("couldn't set el_lispdir variable"));
	}

	/* set the Electric Tcl directory for extra TCL code */
	infstr = initinfstr();
	addstringtoinfstr(infstr, el_libdir);
	addstringtoinfstr(infstr, "tcl");
	addtoinfstr(infstr, DIRSEP);
	path = returninfstr(infstr);
	if (Tcl_SetVar(tcl_interp, "el_tcldir", path, TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
	{
		ttyputerr(_("Tcl_SetVar failed: %s"), tcl_interp->result);
		ttyputmsg(_("couldn't set el_tcldir variable"));
	}

	/* create the Electric commands */
	Tcl_CreateCommand(tcl_interp, "curlib",          tcl_curlib,          (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "curtech",         tcl_curtech,         (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getval",          tcl_getval,          (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getparentval",    tcl_getparentval,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "P",               tcl_P,               (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "PD",              tcl_PD,              (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "PAR",             tcl_PAR,             (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "PARD",            tcl_PARD,            (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "setval",          tcl_setval,          (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "setind",          tcl_setind,          (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "delval",          tcl_delval,          (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "initsearch",      tcl_initsearch,      (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "nextobject",      tcl_nextobject,      (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "gettool",         tcl_gettool,         (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "maxtool",         tcl_maxtool,         (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "indextool",       tcl_indextool,       (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "toolturnon",      tcl_toolturnon,      (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "toolturnoff",     tcl_toolturnoff,     (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getlibrary",      tcl_getlibrary,      (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "newlibrary",      tcl_newlibrary,      (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "killlibrary",     tcl_killlibrary,     (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "eraselibrary",    tcl_eraselibrary,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "selectlibrary",   tcl_selectlibrary,   (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getnodeproto",    tcl_getnodeproto,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "newnodeproto",    tcl_newnodeproto,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "killnodeproto",   tcl_killnodeproto,   (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "copynodeproto",   tcl_copynodeproto,   (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "iconview",        tcl_iconview,        (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "contentsview",    tcl_contentsview,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "newnodeinst",     tcl_newnodeinst,     (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "modifynodeinst",  tcl_modifynodeinst,  (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "killnodeinst",    tcl_killnodeinst,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "replacenodeinst", tcl_replacenodeinst, (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "nodefunction",    tcl_nodefunction,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "newarcinst",      tcl_newarcinst,      (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "modifyarcinst",   tcl_modifyarcinst,   (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "killarcinst",     tcl_killarcinst,     (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "replacearcinst",  tcl_replacearcinst,  (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "newportproto",    tcl_newportproto,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "portposition",    tcl_portposition,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getportproto",    tcl_getportproto,    (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "killportproto",   tcl_killportproto,   (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "moveportproto",   tcl_moveportproto,   (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "undoabatch",      tcl_undoabatch,      (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "noundoallowed",   tcl_noundoallowed,   (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getview",         tcl_getview,         (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "newview",         tcl_newview,         (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "killview",        tcl_killview,        (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "telltool",        tcl_telltool,        (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "asktool",         tcl_asktool,         (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getarcproto",     tcl_getarcproto,     (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getcell",         tcl_getcell,         (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "gettechnology",   tcl_gettechnology,   (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getpinproto",     tcl_getpinproto,     (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(tcl_interp, "getnetwork",      tcl_getnetwork,      (ClientData)0, (Tcl_CmdDeleteProc *)NULL);
}

void tcl_converttoanyelectric(char *tclstr, INTBIG *addr, INTBIG *type)
{
	REGISTER char *pt;
	REGISTER float fval;
	REGISTER void *infstr;

	/* simple if it is a number */
	if (isanumber(tclstr))
	{
		for(pt = tclstr; *pt != 0; pt++)
			if (*pt == '.')
		{
			/* with a decimal point, use floating point representation */
			fval = (float)atof(tclstr);
			*addr = castint(fval);
			*type = VFLOAT;
			return;
		}

		/* just an integer */
		*addr = myatoi(tclstr);
		*type = VINTEGER;
		return;
	}

	/* check for special Electric object names */
	if (namesamen(tclstr, "#nodeinst",     9) == 0) { *addr = myatoi(&tclstr[9]);  *type = VNODEINST;    return; }
	if (namesamen(tclstr, "#nodeproto",   10) == 0) { *addr = myatoi(&tclstr[10]); *type = VNODEPROTO;   return; }
	if (namesamen(tclstr, "#portarcinst", 12) == 0) { *addr = myatoi(&tclstr[12]); *type = VPORTARCINST; return; }
	if (namesamen(tclstr, "#portexpinst", 12) == 0) { *addr = myatoi(&tclstr[12]); *type = VPORTEXPINST; return; }
	if (namesamen(tclstr, "#portproto",   10) == 0) { *addr = myatoi(&tclstr[10]); *type = VPORTPROTO;   return; }
	if (namesamen(tclstr, "#arcinst",      8) == 0) { *addr = myatoi(&tclstr[8]);  *type = VARCINST;     return; }
	if (namesamen(tclstr, "#arcproto",     9) == 0) { *addr = myatoi(&tclstr[9]);  *type = VARCPROTO;    return; }
	if (namesamen(tclstr, "#geom",         5) == 0) { *addr = myatoi(&tclstr[5]);  *type = VGEOM;        return; }
	if (namesamen(tclstr, "#library",      8) == 0) { *addr = myatoi(&tclstr[8]);  *type = VLIBRARY;     return; }
	if (namesamen(tclstr, "#technology",  11) == 0) { *addr = myatoi(&tclstr[11]); *type = VTECHNOLOGY;  return; }
	if (namesamen(tclstr, "#tool",         5) == 0) { *addr = myatoi(&tclstr[5]);  *type = VTOOL;        return; }
	if (namesamen(tclstr, "#rtnode",       7) == 0) { *addr = myatoi(&tclstr[7]);  *type = VRTNODE;      return; }
	if (namesamen(tclstr, "#network",      8) == 0) { *addr = myatoi(&tclstr[8]);  *type = VNETWORK;     return; }
	if (namesamen(tclstr, "#cell",         5) == 0) { *addr = myatoi(&tclstr[5]);  *type = VCELL;        return; }
	if (namesamen(tclstr, "#view",         5) == 0) { *addr = myatoi(&tclstr[5]);  *type = VVIEW;        return; }
	if (namesamen(tclstr, "#window",       7) == 0) { *addr = myatoi(&tclstr[7]);  *type = VWINDOWPART;  return; }
	if (namesamen(tclstr, "#graphics",     9) == 0) { *addr = myatoi(&tclstr[9]);  *type = VGRAPHICS;    return; }
	if (namesamen(tclstr, "#constraint",  11) == 0) { *addr = myatoi(&tclstr[11]); *type = VCONSTRAINT;  return; }
	if (namesamen(tclstr, "#windowframe", 12) == 0) { *addr = myatoi(&tclstr[7]);  *type = VWINDOWFRAME; return; }

	/* just a string */
	infstr = initinfstr();
	addstringtoinfstr(infstr, tclstr);
	*addr = (INTBIG)returninfstr(infstr);
	*type = VSTRING;
}

INTBIG tcl_converttoelectric(char *tclstr, INTBIG type)
{
	REGISTER void *infstr;

	switch (type&VTYPE)
	{
		case VSTRING:
			infstr = initinfstr();
			addstringtoinfstr(infstr, tclstr);
			return((INTBIG)returninfstr(infstr));

		case VINTEGER:
		case VSHORT:
		case VBOOLEAN:
		case VADDRESS:
			return(myatoi(tclstr));

		case VFLOAT:
		case VDOUBLE:
			return(castint((float)atof(tclstr)));

		case VFRACT:
			return(myatoi(tclstr) * WHOLE);

		case VNODEINST:    if (namesamen(tclstr, "#nodeinst",     9) == 0) return(myatoi(&tclstr[9]));  break;
		case VNODEPROTO:   if (namesamen(tclstr, "#nodeproto",   10) == 0) return(myatoi(&tclstr[10])); break;
		case VPORTARCINST: if (namesamen(tclstr, "#portarcinst", 12) == 0) return(myatoi(&tclstr[12])); break;
		case VPORTEXPINST: if (namesamen(tclstr, "#portexpinst", 12) == 0) return(myatoi(&tclstr[12])); break;
		case VPORTPROTO:   if (namesamen(tclstr, "#portproto",   10) == 0) return(myatoi(&tclstr[10])); break;
		case VARCINST:     if (namesamen(tclstr, "#arcinst",      8) == 0) return(myatoi(&tclstr[8]));  break;
		case VARCPROTO:    if (namesamen(tclstr, "#arcproto",     9) == 0) return(myatoi(&tclstr[9]));  break;
		case VGEOM:        if (namesamen(tclstr, "#geom",         5) == 0) return(myatoi(&tclstr[5]));  break;
		case VLIBRARY:     if (namesamen(tclstr, "#library",      8) == 0) return(myatoi(&tclstr[8]));  break;
		case VTECHNOLOGY:  if (namesamen(tclstr, "#technology",  11) == 0) return(myatoi(&tclstr[11])); break;
		case VTOOL:        if (namesamen(tclstr, "#tool",         5) == 0) return(myatoi(&tclstr[5]));  break;
		case VRTNODE:      if (namesamen(tclstr, "#rtnode",       7) == 0) return(myatoi(&tclstr[7]));  break;
		case VNETWORK:     if (namesamen(tclstr, "#network",      8) == 0) return(myatoi(&tclstr[8]));  break;
		case VCELL:        if (namesamen(tclstr, "#cell",         5) == 0) return(myatoi(&tclstr[5]));  break;
		case VVIEW:        if (namesamen(tclstr, "#view",         5) == 0) return(myatoi(&tclstr[5]));  break;
		case VWINDOWPART:  if (namesamen(tclstr, "#window",       7) == 0) return(myatoi(&tclstr[7]));  break;
		case VGRAPHICS:    if (namesamen(tclstr, "#graphics",     9) == 0) return(myatoi(&tclstr[9]));  break;
		case VCONSTRAINT:  if (namesamen(tclstr, "#constraint",  11) == 0) return(myatoi(&tclstr[11])); break;
		case VWINDOWFRAME: if (namesamen(tclstr, "#windowframe", 12) == 0) return(myatoi(&tclstr[12])); break;
	}
	return(-1);
}

INTBIG tcl_converttowarnedelectric(char *name, INTBIG want, char *msg)
{
	INTBIG addr, type;

	tcl_converttoanyelectric(name, &addr, &type);
	if (type != want)
	{
		Tcl_AppendResult(tcl_interp, msg, _(" argument has the wrong type"), (char *)NULL);
		return(-1);
	}
	return(addr);
}

char *tcl_converttotcl(INTBIG addr, INTBIG type)
{
	REGISTER INTBIG saddr, stype, len, i;
	REGISTER void *infstr;

	/* handle scalars easily */
	if ((type&VISARRAY) == 0)
	{
		infstr = initinfstr();
		tcl_converttoscalartcl(infstr, addr, type);
		return(returninfstr(infstr));
	}

	/* handle arrays */
	infstr = initinfstr();
	len = (type&VLENGTH) >> VLENGTHSH;
	if (len != 0)
	{
		stype = type & VTYPE;
		for(i=0; i<len; i++)
		{
			if ((type&VTYPE) == VGENERAL)
			{
				stype = ((INTBIG *)addr)[i*2+1];
				saddr = ((INTBIG *)addr)[i*2];
			} else
			{
				if ((type&VTYPE) == VCHAR) saddr = ((char *)addr)[i]; else
					if ((type&VTYPE) == VDOUBLE) saddr = (INTBIG)(((double *)addr)[i]); else
						if ((type&VTYPE) == VSHORT) saddr = ((INTSML *)addr)[i]; else
							saddr = ((INTBIG *)addr)[i];
			}
			if (i != 0) addtoinfstr(infstr, ' ');
			tcl_converttoscalartcl(infstr, saddr, stype);
		}
	} else
	{
		for(i=0; ; i++)
		{
			if ((type&VTYPE) == VCHAR)
			{
				if ((((char *)addr)[i]&0377) == 0377) break;
				saddr = ((char *)addr)[i];
			} else if ((type&VTYPE) == VDOUBLE)
			{
				if (((double *)addr)[i] == -1) break;
				saddr = (INTBIG)(((double *)addr)[i]);
			} else if ((type&VTYPE) == VSHORT)
			{
				if (((INTSML *)addr)[i] == -1) break;
				saddr = ((INTSML *)addr)[i];
			} else
			{
				saddr = ((INTBIG *)addr)[i];
				if (saddr == -1) break;
			}
			if (i != 0) addtoinfstr(infstr, ' ');
			tcl_converttoscalartcl(infstr, saddr, type);
		}
	}
	return(returninfstr(infstr));
}

void tcl_converttoscalartcl(void *infstr, INTBIG addr, INTBIG type)
{
	static char line[50];

	switch (type&VTYPE)
	{
		case VSTRING:
			addtoinfstr(infstr, '"');
			addstringtoinfstr(infstr, (char *)addr);
			addtoinfstr(infstr, '"');
			return;
		case VINTEGER:
		case VSHORT:
		case VBOOLEAN:
		case VADDRESS:     sprintf(line, "%ld", addr);                break;
		case VFLOAT:
		case VDOUBLE:      sprintf(line, "%g", castfloat(addr));      break;
		case VFRACT:       sprintf(line, "%g", (float)addr / WHOLE);  break;
		case VNODEINST:    sprintf(line, "#nodeinst%ld",    addr);    break;
		case VNODEPROTO:   sprintf(line, "#nodeproto%ld",   addr);    break;
		case VPORTARCINST: sprintf(line, "#portarcinst%ld", addr);    break;
		case VPORTEXPINST: sprintf(line, "#portexpinst%ld", addr);    break;
		case VPORTPROTO:   sprintf(line, "#portproto%ld",   addr);    break;
		case VARCINST:     sprintf(line, "#arcinst%ld",     addr);    break;
		case VARCPROTO:    sprintf(line, "#arcproto%ld",    addr);    break;
		case VGEOM:        sprintf(line, "#geom%ld",        addr);    break;
		case VLIBRARY:     sprintf(line, "#library%ld",     addr);    break;
		case VTECHNOLOGY:  sprintf(line, "#technology%ld",  addr);    break;
		case VTOOL:        sprintf(line, "#tool%ld",        addr);    break;
		case VRTNODE:      sprintf(line, "#rtnode%ld",      addr);    break;
		case VNETWORK:     sprintf(line, "#network%ld",     addr);    break;
		case VCELL:        sprintf(line, "#cell%ld",        addr);    break;
		case VVIEW:        sprintf(line, "#view%ld",        addr);    break;
		case VWINDOWPART:  sprintf(line, "#window%ld",      addr);    break;
		case VGRAPHICS:    sprintf(line, "#graphics%ld",    addr);    break;
		case VCONSTRAINT:  sprintf(line, "#constraint%ld",  addr);    break;
		case VWINDOWFRAME: sprintf(line, "#windowframe%ld", addr);    break;
		default: return;
	}
	addstringtoinfstr(infstr, line);
}

int tcl_tkconsoleclose(ClientData instanceData, Tcl_Interp *interp) { return 0; }

int tcl_tkconsoleinput(ClientData instanceData, char *buf, int bufSize, int *errorCode)
{
	REGISTER char *pt;
	REGISTER INTBIG len, i;

	*errorCode = 0;
	*tcl_outputloc = 0;
	pt = ttygetlinemessages(tcl_outputbuffer);
	tcl_outputloc = tcl_outputbuffer;
	len = strlen(pt);
	if (len > bufSize) len = bufSize;
	for(i=0; i<len; i++) buf[i] = pt[i];
	return(len);
}

int tcl_tkconsoleoutput(ClientData instanceData, char *buf, int toWrite, int *errorCode)
{
	char save;

	*errorCode = 0;
	Tcl_SetErrno(0);

	save = buf[toWrite];
	buf[toWrite] = 0;
	tcl_dumpoutput(buf);
	buf[toWrite] = save;
	return(toWrite);
}

void tcl_tkconsolewatch(ClientData instanceData, int mask) {}

void tcl_dumpoutput(char *str)
{
	while (*str != 0)
	{
		if (*str == '\n')
		{
			*tcl_outputloc = 0;
			ttyputmsg("%s", tcl_outputbuffer);
			tcl_outputloc = tcl_outputbuffer;
		} else
		{
			if (tcl_outputloc >= tcl_outputbuffer+BUFSIZE-2)
			{
				*tcl_outputloc = 0;
				ttyputmsg("%s", tcl_outputbuffer);
				tcl_outputloc = tcl_outputbuffer;
			}
			*tcl_outputloc++ = *str;
		}
		str++;
	}
}

/************************* DATABASE EXAMINATION ROUTINES *************************/

int tcl_curlib(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	if (argc != 1)
	{
		Tcl_AppendResult(interp, "Usage: curlib", (char *)NULL);
		return TCL_ERROR;
	}

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)el_curlib, VLIBRARY), (char *) NULL);
	return TCL_OK;
}

int tcl_curtech(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	if (argc != 1)
	{
		Tcl_AppendResult(interp, "Usage: curtech", (char *)NULL);
		return TCL_ERROR;
	}

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)el_curtech, VTECHNOLOGY), (char *) NULL);
	return TCL_OK;
}

int tcl_getval(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	INTBIG addr, type;
	char *retval;
	REGISTER VARIABLE *var;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: getval OBJECT QUAL", (char *)NULL);
		return TCL_ERROR;
	}
	tcl_converttoanyelectric(argv[1], &addr, &type);

	/* call Electric */
	var = getval(addr, type, -1, argv[2]);

	/* convert result */
	if (var != NOVARIABLE)
	{
		retval = tcl_converttotcl(var->addr, var->type);
		Tcl_AppendResult(interp, retval, (char *)NULL);
	}
	return TCL_OK;
}

int tcl_getparentval(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	INTBIG height;
	char *retval;
	REGISTER VARIABLE *var;

	/* get input */
	if (argc != 4)
	{
		Tcl_AppendResult(interp, "Usage: getparentval OBJECT DEFAULT HEIGHT", (char *)NULL);
		return TCL_ERROR;
	}
	height = myatoi(argv[3]);

	/* call Electric */
	var = getparentval(argv[1], height);

	/* convert result */
	if (var == NOVARIABLE) retval = argv[2]; else
		retval = tcl_converttotcl(var->addr, var->type);
	Tcl_AppendResult(interp, retval, (char *)NULL);
	return TCL_OK;
}

int tcl_P(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	char *retval, line[300];
	REGISTER VARIABLE *var;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: P OBJECT", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	sprintf(line, "ATTR_%s", argv[1]);
	var = getparentval(line, 1);

	/* convert result */
	if (var != NOVARIABLE)
	{
		retval = tcl_converttotcl(var->addr, var->type);
		Tcl_AppendResult(interp, retval, (char *)NULL);
	}
	return TCL_OK;
}

int tcl_PD(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	char *retval, line[300];
	REGISTER VARIABLE *var;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: PD OBJECT DEFAULT", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	sprintf(line, "ATTR_%s", argv[1]);
	var = getparentval(line, 1);

	/* convert result */
	if (var == NOVARIABLE) retval = argv[2]; else
		retval = tcl_converttotcl(var->addr, var->type);
	Tcl_AppendResult(interp, retval, (char *)NULL);
	return TCL_OK;
}

int tcl_PAR(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	char *retval, line[300];
	REGISTER VARIABLE *var;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: PAR OBJECT", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	sprintf(line, "ATTR_%s", argv[1]);
	var = getparentval(line, 0);

	/* convert result */
	if (var != NOVARIABLE)
	{
		retval = tcl_converttotcl(var->addr, var->type);
		Tcl_AppendResult(interp, retval, (char *)NULL);
	}
	return TCL_OK;
}

int tcl_PARD(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	char *retval, line[300];
	REGISTER VARIABLE *var;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: PARD OBJECT DEFAULT", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	sprintf(line, "ATTR_%s", argv[1]);
	var = getparentval(line, 0);

	/* convert result */
	if (var == NOVARIABLE) retval = argv[2]; else
		retval = tcl_converttotcl(var->addr, var->type);
	Tcl_AppendResult(interp, retval, (char *)NULL);
	return TCL_OK;
}

int tcl_setval(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	INTBIG addr, type, naddr, ntype, i, *naddrlist;
	REGISTER VARIABLE *ret;
	char *retval, **setArgv;
	int setArgc;

	/* get input */
	if (argc != 4 && argc != 5)
	{
		Tcl_AppendResult(interp, "Usage: setval OBJECT QUAL NEWVALUE [OPTIONS]", (char *)NULL);
		return TCL_ERROR;
	}
	tcl_converttoanyelectric(argv[1], &addr, &type);

	/* see how many entries are being set */
	i = Tcl_SplitList(interp, argv[3], &setArgc, &setArgv);
	if (i != TCL_OK) return TCL_ERROR;

	/* see if the "newvalue" is an array */
	if (setArgc > 1)
	{
		/* setting an array */
		naddrlist = (INTBIG *)emalloc(setArgc * SIZEOFINTBIG * 2, el_tempcluster);
		if (naddrlist == 0) return TCL_ERROR;
		for(i=0; i<setArgc; i++)
			tcl_converttoanyelectric(setArgv[i], &naddrlist[i*2], &naddrlist[i*2+1]);
		ckfree((char *)setArgv);

		/* see if the array is uniform */
		for(i=1; i<setArgc; i++)
			if (naddrlist[(i-1)*2+1] != naddrlist[i*2+1]) break;
		if (i < setArgc)
		{
			/* general array */
			ret = setval(addr, type, argv[2], (INTBIG)naddrlist,
				VGENERAL|VISARRAY|((setArgc*2) << VLENGTHSH));
		} else
		{
			/* a uniform array */
			ntype = naddrlist[1];
			for(i=1; i<setArgc; i++) naddrlist[i] = naddrlist[i*2];
			ret = setval(addr, type, argv[2], (INTBIG)naddrlist,
				ntype|VISARRAY|(setArgc << VLENGTHSH));
		}
		efree((char *)naddrlist);
	} else
	{
		/* just setting a single value */
		tcl_converttoanyelectric(argv[3], &naddr, &ntype);

		/* call Electric */
		ret = setval(addr, type, argv[2], naddr, ntype);
	}
	if (argc == 5)
	{
		if (namesame(argv[4], "displayable") == 0) ntype |= VDISPLAY;
	}

	/* convert result */
	if (ret != NOVARIABLE)
	{
		retval = tcl_converttotcl(ret->addr, ret->type);
		Tcl_AppendResult(interp, retval, (char *)NULL);
	}
	return TCL_OK;
}

int tcl_setind(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	INTBIG addr, type, aindex, naddr, ntype;
	BOOLEAN ret;

	/* get input */
	if (argc != 5)
	{
		Tcl_AppendResult(interp, "Usage: setind OBJECT QUAL INDEX NEWVALUE", (char *)NULL);
		return TCL_ERROR;
	}
	tcl_converttoanyelectric(argv[1], &addr, &type);
	aindex = myatoi(argv[3]);
	tcl_converttoanyelectric(argv[4], &naddr, &ntype);

	/* call Electric */
	ret = setind(addr, type, argv[2], aindex, naddr);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_delval(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	INTBIG addr, type;
	BOOLEAN ret;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: delval OBJECT QUAL", (char *)NULL);
		return TCL_ERROR;
	}
	tcl_converttoanyelectric(argv[1], &addr, &type);

	/* call Electric */
	ret = delval(addr, type, argv[2]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_initsearch(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER INTBIG lx, hx, ly, hy, sea;
	REGISTER NODEPROTO *np;

	/* get input */
	if (argc != 6)
	{
		Tcl_AppendResult(interp, "Usage: initsearch LX HX LY HY FACET", (char *)NULL);
		return TCL_ERROR;
	}
	lx = myatoi(argv[1]);
	hx = myatoi(argv[2]);
	ly = myatoi(argv[3]);
	hy = myatoi(argv[4]);
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[5], VNODEPROTO, "fifth");
	if (np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	sea = initsearch(lx, hx, ly, hy, np);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(sea, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_nextobject(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER INTBIG sea;
	REGISTER GEOM *g;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: nextobject SEARCH", (char *)NULL);
		return TCL_ERROR;
	}
	sea = myatoi(argv[1]);

	/* call Electric */
	g = nextobject(sea);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)g, VGEOM), (char *) NULL);
	return TCL_OK;
}

/****************************** TOOL ROUTINES ******************************/

int tcl_gettool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER TOOL *tool;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: gettool NAME", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	tool = gettool(argv[1]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)tool, VTOOL), (char *) NULL);
	return TCL_OK;
}

int tcl_maxtool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	/* get input */
	if (argc != 1)
	{
		Tcl_AppendResult(interp, "Usage: maxtool", (char *)NULL);
		return TCL_ERROR;
	}

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(el_maxtools, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_indextool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER TOOL *tool;
	REGISTER INTBIG aindex;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: indextool INDEX", (char *)NULL);
		return TCL_ERROR;
	}
	aindex = myatoi(argv[1]);

	/* call Electric */
	if (aindex < 0 || aindex >= el_maxtools) tool = NOTOOL; else tool = &el_tools[aindex];

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)tool, VTOOL), (char *) NULL);
	return TCL_OK;
}

int tcl_toolturnon(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER TOOL *tool;
	REGISTER INTBIG nocatchupn;
	REGISTER BOOLEAN nocatchup;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: toolturnon TOOL NOCATCHUP", (char *)NULL);
		return TCL_ERROR;
	}
	tool = (TOOL *)tcl_converttowarnedelectric(argv[1], VTOOL, "first");
	if (tool == NOTOOL) return TCL_ERROR;
	nocatchupn = myatoi(argv[2]);
	nocatchup = nocatchupn != 0 ? TRUE : FALSE;

	/* call Electric */
	toolturnon(tool, nocatchup);
	return TCL_OK;
}

int tcl_toolturnoff(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER TOOL *tool;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: toolturnoff TOOL", (char *)NULL);
		return TCL_ERROR;
	}
	tool = (TOOL *)tcl_converttowarnedelectric(argv[1], VTOOL, "first");
	if (tool == NOTOOL) return TCL_ERROR;

	/* call Electric */
	toolturnoff(tool, TRUE);
	return TCL_OK;
}

/****************************** LIBRARY ROUTINES ******************************/

int tcl_getlibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER LIBRARY *lib;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: getlibrary NAME", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	lib = getlibrary(argv[1]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)lib, VLIBRARY), (char *) NULL);
	return TCL_OK;
}

int tcl_newlibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER LIBRARY *lib;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: newlibrary NAME FILE", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	lib = newlibrary(argv[1], argv[2]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)lib, VLIBRARY), (char *) NULL);
	return TCL_OK;
}

int tcl_killlibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER LIBRARY *lib;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: killlibrary LIBRARY", (char *)NULL);
		return TCL_ERROR;
	}
	lib = (LIBRARY *)tcl_converttowarnedelectric(argv[1], VLIBRARY, "first");
	if (lib == NOLIBRARY) return TCL_ERROR;

	/* call Electric */
	killlibrary(lib);
	return TCL_OK;
}

int tcl_eraselibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER LIBRARY *lib;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: eraselibrary LIBRARY", (char *)NULL);
		return TCL_ERROR;
	}
	lib = (LIBRARY *)tcl_converttowarnedelectric(argv[1], VLIBRARY, "first");
	if (lib == NOLIBRARY) return TCL_ERROR;

	/* call Electric */
	eraselibrary(lib);
	return TCL_OK;
}

int tcl_selectlibrary(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER LIBRARY *lib;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: selectlibrary LIBRARY", (char *)NULL);
		return TCL_ERROR;
	}
	lib = (LIBRARY *)tcl_converttowarnedelectric(argv[1], VLIBRARY, "first");
	if (lib == NOLIBRARY) return TCL_ERROR;

	/* call Electric */
	selectlibrary(lib);
	return TCL_OK;
}

/****************************** NODEPROTO ROUTINES ******************************/

int tcl_getnodeproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: getnodeproto NAME", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	np = getnodeproto(argv[1]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)np, VNODEPROTO), (char *) NULL);
	return TCL_OK;
}

int tcl_newnodeproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER LIBRARY *lib;
	REGISTER NODEPROTO *np;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: newnodeproto NAME LIBRARY", (char *)NULL);
		return TCL_ERROR;
	}
	lib = (LIBRARY *)tcl_converttowarnedelectric(argv[2], VLIBRARY, "second");
	if (lib == NOLIBRARY) return TCL_ERROR;

	/* call Electric */
	np = newnodeproto(argv[1], lib);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)np, VNODEPROTO), (char *) NULL);
	return TCL_OK;
}

int tcl_killnodeproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np;
	REGISTER BOOLEAN ret;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: killnodeproto FACET", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	if (np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	ret = killnodeproto(np);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret?1:0, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_copynodeproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER LIBRARY *lib;
	REGISTER NODEPROTO *np, *nnp;

	/* get input */
	if (argc != 4)
	{
		Tcl_AppendResult(interp, "Usage: copynodeproto FACET LIBRARY NAME", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	lib = (LIBRARY *)tcl_converttowarnedelectric(argv[2], VLIBRARY, "second");
	if (lib == NOLIBRARY || np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	nnp = copynodeproto(np, lib, argv[3]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)nnp, VNODEPROTO), (char *) NULL);
	return TCL_OK;
}

int tcl_iconview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np, *inp;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: iconview FACET", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	if (np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	inp = iconview(np);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)inp, VNODEPROTO), (char *) NULL);
	return TCL_OK;
}

int tcl_contentsview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np, *cnp;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: contentsview FACET", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	if (np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	cnp = contentsview(np);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)cnp, VNODEPROTO), (char *) NULL);
	return TCL_OK;
}

/****************************** NODEINST ROUTINES ******************************/

int tcl_newnodeinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np, *facet;
	REGISTER INTBIG lx, hx, ly, hy, rot, tran;

	/* get input */
	if (argc != 9)
	{
		Tcl_AppendResult(interp, "Usage: newnodeinst PROTO LX HX LY HY TRAN ROT FACET", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	lx = myatoi(argv[2]);
	hx = myatoi(argv[3]);
	ly = myatoi(argv[4]);
	hy = myatoi(argv[5]);
	tran = myatoi(argv[6]);
	rot = myatoi(argv[7]);
	facet = (NODEPROTO *)tcl_converttowarnedelectric(argv[8], VNODEPROTO, "eighth");
	if (np == NONODEPROTO || facet == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	ni = newnodeinst(np, lx, hx, ly, hy, tran, rot, facet);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)ni, VNODEINST), (char *) NULL);
	return TCL_OK;
}

int tcl_modifynodeinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEINST *ni;
	REGISTER INTBIG dlx, dly, dhx, dhy, drot, dtran;

	/* get input */
	if (argc != 8)
	{
		Tcl_AppendResult(interp, "Usage: modifynodeinst NODE dLX dLY dHX dHY dROT dTRAN", (char *)NULL);
		return TCL_ERROR;
	}
	ni = (NODEINST *)tcl_converttowarnedelectric(argv[1], VNODEINST, "first");
	dlx = myatoi(argv[2]);
	dly = myatoi(argv[3]);
	dhx = myatoi(argv[4]);
	dhy = myatoi(argv[5]);
	drot = myatoi(argv[6]);
	dtran = myatoi(argv[7]);
	if (ni == NONODEINST) return TCL_ERROR;

	/* call Electric */
	modifynodeinst(ni, dlx, dly, dhx, dhy, drot, dtran);
	return TCL_OK;
}

int tcl_killnodeinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEINST *ni;
	REGISTER BOOLEAN ret;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: killnodeinst NODE", (char *)NULL);
		return TCL_ERROR;
	}
	ni = (NODEINST *)tcl_converttowarnedelectric(argv[1], VNODEINST, "first");
	if (ni == NONODEINST) return TCL_ERROR;

	/* call Electric */
	ret = killnodeinst(ni);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret?1:0, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_replacenodeinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni, *nni;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: replacenodeinst NODE NEWPROTO", (char *)NULL);
		return TCL_ERROR;
	}
	ni = (NODEINST *)tcl_converttowarnedelectric(argv[1], VNODEINST, "first");
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[2], VNODEPROTO, "second");
	if (ni == NONODEINST || np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	nni = replacenodeinst(ni, np, FALSE, FALSE);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)nni, VNODEINST), (char *) NULL);
	return TCL_OK;
}

int tcl_nodefunction(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEINST *ni;
	REGISTER INTBIG fun;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: nodefunction NODE", (char *)NULL);
		return TCL_ERROR;
	}
	ni = (NODEINST *)tcl_converttowarnedelectric(argv[1], VNODEINST, "first");
	if (ni == NONODEINST) return TCL_ERROR;

	/* call Electric */
	fun = nodefunction(ni);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(fun, VINTEGER), (char *) NULL);
	return TCL_OK;
}

/****************************** ARCINST ROUTINES ******************************/

int tcl_newarcinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER ARCINST *ai;
	REGISTER NODEINST *ni1, *ni2;
	REGISTER PORTPROTO *pp1, *pp2;
	REGISTER ARCPROTO *ap;
	REGISTER NODEPROTO *np;
	REGISTER INTBIG w, bits, x1, y1, x2, y2;

	/* get input */
	if (argc != 13)
	{
		Tcl_AppendResult(interp, "Usage: newarcinst PROTO WID BITS NODE1 PORT1 X1 Y1 NODE2 PORT2 X2 Y2 FACET",
			(char *)NULL);
		return TCL_ERROR;
	}
	ap = (ARCPROTO *)tcl_converttowarnedelectric(argv[1], VARCPROTO, "first");
	w = myatoi(argv[2]);
	bits = myatoi(argv[3]);
	ni1 = (NODEINST *)tcl_converttowarnedelectric(argv[4], VNODEINST, "fourth");
	pp1 = (PORTPROTO *)tcl_converttowarnedelectric(argv[5], VPORTPROTO, "fifth");
	x1 = myatoi(argv[6]);
	y1 = myatoi(argv[7]);
	ni2 = (NODEINST *)tcl_converttowarnedelectric(argv[8], VNODEINST, "eighth");
	pp2 = (PORTPROTO *)tcl_converttowarnedelectric(argv[9], VPORTPROTO, "ninth");
	x2 = myatoi(argv[10]);
	y2 = myatoi(argv[11]);
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[12], VNODEPROTO, "twelvth");
	if (ap == NOARCPROTO || ni1 == NONODEINST || pp1 == NOPORTPROTO || ni2 == NONODEINST ||
		pp2 == NOPORTPROTO || np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	ai = newarcinst(ap, w, bits, ni1, pp1, x1, y1, ni2, pp2, x2, y2, np);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)ai, VARCINST), (char *) NULL);
	return TCL_OK;
}

int tcl_modifyarcinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER ARCINST *ai;
	REGISTER INTBIG dw, dx1, dy1, dx2, dy2;
	REGISTER BOOLEAN ret;

	/* get input */
	if (argc != 7)
	{
		Tcl_AppendResult(interp, "Usage: modifyarcinst ARC dW dX1 dY1 dX2 dY2", (char *)NULL);
		return TCL_ERROR;
	}
	ai = (ARCINST *)tcl_converttowarnedelectric(argv[1], VARCINST, "first");
	if (ai == NOARCINST) return TCL_ERROR;
	dw = myatoi(argv[2]);
	dx1 = myatoi(argv[3]);
	dy1 = myatoi(argv[4]);
	dx2 = myatoi(argv[5]);
	dy2 = myatoi(argv[6]);

	/* call Electric */
	ret = modifyarcinst(ai, dw, dx1, dy1, dx2, dy2);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret?1:0, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_killarcinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER ARCINST *ai;
	REGISTER BOOLEAN ret;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: killarcinst ARC", (char *)NULL);
		return TCL_ERROR;
	}
	ai = (ARCINST *)tcl_converttowarnedelectric(argv[1], VARCINST, "first");
	if (ai == NOARCINST) return TCL_ERROR;

	/* call Electric */
	ret = killarcinst(ai);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret?1:0, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_replacearcinst(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER ARCPROTO *ap;
	REGISTER ARCINST *ai, *nai;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: replacearcinst ARC NEWPROTO", (char *)NULL);
		return TCL_ERROR;
	}
	ai = (ARCINST *)tcl_converttowarnedelectric(argv[1], VARCINST, "first");
	ap = (ARCPROTO *)tcl_converttowarnedelectric(argv[2], VARCPROTO, "second");
	if (ai == NOARCINST || ap == NOARCPROTO) return TCL_ERROR;

	/* call Electric */
	nai = replacearcinst(ai, ap);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)nai, VARCINST), (char *) NULL);
	return TCL_OK;
}

/****************************** PORTPROTO ROUTINES ******************************/

int tcl_newportproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni;
	REGISTER PORTPROTO *pp, *npp;

	/* get input */
	if (argc != 5)
	{
		Tcl_AppendResult(interp, "Usage: newportproto FACET NODE PORT NAME", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	ni = (NODEINST *)tcl_converttowarnedelectric(argv[2], VNODEINST, "second");
	pp = (PORTPROTO *)tcl_converttowarnedelectric(argv[3], VPORTPROTO, "third");
	if (np == NONODEPROTO || ni == NONODEINST || pp == NOPORTPROTO) return TCL_ERROR;

	/* call Electric */
	npp = newportproto(np, ni, pp, argv[4]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)npp, VPORTPROTO), (char *) NULL);
	return TCL_OK;
}

int tcl_portposition(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEINST *ni;
	REGISTER PORTPROTO *pp;
	INTBIG x, y;
	char line[50];

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: portposition NODE PORT", (char *)NULL);
		return TCL_ERROR;
	}
	ni = (NODEINST *)tcl_converttowarnedelectric(argv[1], VNODEINST, "first");
	pp = (PORTPROTO *)tcl_converttowarnedelectric(argv[2], VPORTPROTO, "second");
	if (ni == NONODEINST || pp == NOPORTPROTO) return TCL_ERROR;

	/* call Electric */
	portposition(ni, pp, &x, &y);

	/* convert result */
	(void)sprintf(line, "%ld %ld", x, y);
	Tcl_AppendResult(interp, line, (char *) NULL);
	return TCL_OK;
}

int tcl_getportproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: getportproto FACET NAME", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	if (np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	pp = getportproto(np, argv[2]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)pp, VPORTPROTO), (char *) NULL);
	return TCL_OK;
}

int tcl_killportproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;
	REGISTER BOOLEAN ret;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: killportproto FACET PORT", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	pp = (PORTPROTO *)tcl_converttowarnedelectric(argv[2], VPORTPROTO, "second");
	if (np == NONODEPROTO || pp == NOPORTPROTO) return TCL_ERROR;

	/* call Electric */
	ret = killportproto(np, pp);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret?1:0, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_moveportproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *opp, *npp;
	REGISTER NODEINST *nni;
	REGISTER BOOLEAN ret;

	/* get input */
	if (argc != 5)
	{
		Tcl_AppendResult(interp, "Usage: moveportproto FACET PORT NEWNODE NEWPORT", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[1], VNODEPROTO, "first");
	opp = (PORTPROTO *)tcl_converttowarnedelectric(argv[2], VPORTPROTO, "second");
	nni = (NODEINST *)tcl_converttowarnedelectric(argv[3], VNODEINST, "third");
	npp = (PORTPROTO *)tcl_converttowarnedelectric(argv[4], VPORTPROTO, "fourth");
	if (np == NONODEPROTO || opp == NOPORTPROTO || nni == NONODEINST || npp == NOPORTPROTO) return TCL_ERROR;

	/* call Electric */
	ret = moveportproto(np, opp, nni, npp);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret?1:0, VINTEGER), (char *) NULL);
	return TCL_OK;
}

/*************************** CHANGE CONTROL ROUTINES ***************************/

int tcl_undoabatch(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	TOOL *tool;

	if (argc != 1)
	{
		Tcl_AppendResult(interp, "Usage: undoabatch", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	if (undoabatch(&tool) == 0)
	{
		Tcl_AppendResult(interp, _("Error during undoabatch"), (char *)NULL);
		return TCL_ERROR;
	}

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)tool, VTOOL), (char *) NULL);
	return TCL_OK;
}

int tcl_noundoallowed(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	if (argc != 1)
	{
		Tcl_AppendResult(interp, "Usage: noundoallowed", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	noundoallowed();
	return TCL_OK;
}

/****************************** VIEW ROUTINES ******************************/

int tcl_getview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER VIEW *v;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: getview NAME", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	v = getview(argv[1]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)v, VVIEW), (char *) NULL);
	return TCL_OK;
}

int tcl_newview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER VIEW *v;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: newview NAME SNAME", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	v = newview(argv[1], argv[2]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)v, VVIEW), (char *) NULL);
	return TCL_OK;
}

int tcl_killview(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER VIEW *v;
	REGISTER BOOLEAN ret;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: killview VIEW", (char *)NULL);
		return TCL_ERROR;
	}
	v = (VIEW *)tcl_converttowarnedelectric(argv[1], VVIEW, "first");
	if (v == NOVIEW) return TCL_ERROR;

	/* call Electric */
	ret = killview(v);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret?1:0, VINTEGER), (char *) NULL);
	return TCL_OK;
}

/*************************** MISCELLANEOUS ROUTINES ***************************/

int tcl_telltool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER TOOL *tool;
	int toolArgc, ret;
	char **toolArgv;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: telltool TOOL {PARAMETERS}", (char *)NULL);
		return TCL_ERROR;
	}
	tool = (TOOL *)tcl_converttowarnedelectric(argv[1], VTOOL, "first");
	ret = Tcl_SplitList(interp, argv[2], &toolArgc, &toolArgv);
	if (tool == NOTOOL || ret != TCL_OK) return TCL_ERROR;

	/* call Electric */
	telltool(tool, toolArgc, toolArgv);
	ckfree((char *)toolArgv);

	/* convert result */
	ret = 0;		/* actually should not return a value!!! */
	Tcl_AppendResult(interp, tcl_converttotcl(ret, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_asktool(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER TOOL *tool;
	INTBIG addr, addr0, addr1, addr2, addr3, type;
	int toolArgc, ret, i, argcS, first;
	char **toolArgv, **argvS, *command;
	REGISTER void *infstr;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: asktool TOOL {PARAMETERS}", (char *)NULL);
		return TCL_ERROR;
	}
	tool = (TOOL *)tcl_converttowarnedelectric(argv[1], VTOOL, "first");
	ret = Tcl_SplitList(interp, argv[2], &toolArgc, &toolArgv);
	if (tool == NOTOOL || ret != TCL_OK) return TCL_ERROR;

	command = toolArgv[0];
	for (i=0; i<toolArgc; i++) toolArgv[i] = toolArgv[i+1];
	toolArgv[toolArgc] = NULL;
	toolArgc--;

	/* call Electric */
	if (toolArgc == 0)
	{
		ret = asktool(tool, command);
	} else if (toolArgc == 1)
	{
		/* special case for user_tool: show-multiple command */
		if (strcmp(command, "show-multiple") == 0)
		{
			ret = Tcl_SplitList(interp, toolArgv[0], &argcS, &argvS);
			if (ret == TCL_OK)
			{
				infstr = initinfstr();
				first = 0;
				for (i=0; i<argcS; i++)
				{
					if (first != 0) addtoinfstr(infstr, '\n');
					first++;
					tcl_converttoanyelectric(argvS[i], &addr, &type);
					formatinfstr(infstr, "FACET=%s FROM=0%lo;-1;0",
						describenodeproto(((GEOM *)addr)->entryaddr.ni->parent), addr);
				}
				ret = asktool(tool, command, (INTBIG)returninfstr(infstr));
			}
		} else
		{
			tcl_converttoanyelectric(toolArgv[0], &addr0, &type);
			ret = asktool(tool, command, addr0);
		}
	} else if (toolArgc == 2)
	{
		tcl_converttoanyelectric(toolArgv[0], &addr0, &type);
		tcl_converttoanyelectric(toolArgv[1], &addr1, &type);
		ret = asktool(tool, command, addr0, addr1);
	} else if (toolArgc == 3)
	{
		tcl_converttoanyelectric(toolArgv[0], &addr0, &type);
		tcl_converttoanyelectric(toolArgv[1], &addr1, &type);
		tcl_converttoanyelectric(toolArgv[2], &addr2, &type);
		ret = asktool(tool, command, addr0, addr1, addr2);
	} else if (toolArgc == 4)
	{
		tcl_converttoanyelectric(toolArgv[0], &addr0, &type);
		tcl_converttoanyelectric(toolArgv[1], &addr1, &type);
		tcl_converttoanyelectric(toolArgv[2], &addr2, &type);
		tcl_converttoanyelectric(toolArgv[3], &addr3, &type);
		ret = asktool(tool, command, addr0, addr1, addr2, addr3);
	}
	ckfree((char *)toolArgv);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl(ret, VINTEGER), (char *) NULL);
	return TCL_OK;
}

int tcl_getarcproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER ARCPROTO *ap;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: getarcproto NAME", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	ap = getarcproto(argv[1]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)ap, VARCPROTO), (char *) NULL);
	return TCL_OK;
}

int tcl_getcell(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER CELL *cell;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: getcell NAME", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	cell = getcell(argv[1]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)cell, VCELL), (char *) NULL);
	return TCL_OK;
}

int tcl_gettechnology(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER TECHNOLOGY *tech;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: gettechnology NAME", (char *)NULL);
		return TCL_ERROR;
	}

	/* call Electric */
	tech = gettechnology(argv[1]);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)tech, VTECHNOLOGY), (char *) NULL);
	return TCL_OK;
}

int tcl_getpinproto(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *ap;

	/* get input */
	if (argc != 2)
	{
		Tcl_AppendResult(interp, "Usage: getpinproto ARCPROTO", (char *)NULL);
		return TCL_ERROR;
	}
	ap = (ARCPROTO *)tcl_converttowarnedelectric(argv[1], VARCPROTO, "first");
	if (ap == NOARCPROTO) return TCL_ERROR;

	/* call Electric */
	np = getpinproto(ap);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)np, VNODEPROTO), (char *) NULL);
	return TCL_OK;
}

int tcl_getnetwork(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
	REGISTER NETWORK *net;
	REGISTER NODEPROTO *np;

	/* get input */
	if (argc != 3)
	{
		Tcl_AppendResult(interp, "Usage: getnetwork NAME FACET", (char *)NULL);
		return TCL_ERROR;
	}
	np = (NODEPROTO *)tcl_converttowarnedelectric(argv[2], VNODEPROTO, "second");
	if (np == NONODEPROTO) return TCL_ERROR;

	/* call Electric */
	net = getnetwork(argv[1], np);

	/* convert result */
	Tcl_AppendResult(interp, tcl_converttotcl((INTBIG)net, VNETWORK), (char *) NULL);
	return TCL_OK;
}

#endif  /* LANGTCL - at top */
