/**
 * D-Mod installation wizard

 * Copyright (C) 2004  Andrew Reading
 * Copyright (C) 2005, 2006  Dan Walma
 * Copyright (C) 2008  Sylvain Beucler

 * This file is part of GNU FreeDink

 * GNU FreeDink is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.

 * GNU FreeDink is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <wx/msgdlg.h>
#include <wx/filename.h>
#include <wx/progdlg.h>

#include "InstallVerifyFrame.hpp"
#include "Config.hpp"
#include "BZip.hpp"
#include "Tar.hpp"

BEGIN_EVENT_TABLE(InstallVerifyFrame, wxDialog)
EVT_BUTTON(wxID_OK, InstallVerifyFrame::onInstall)
EVT_BUTTON(wxID_CANCEL, InstallVerifyFrame::onCancel)
END_EVENT_TABLE()

InstallVerifyFrame::InstallVerifyFrame(const wxString& aFilePath) :
InstallVerifyFrame_Base(NULL, wxID_ANY, _T("")),
  mExtracting(true),
  mDmodFilePath(aFilePath),
  mTar(NULL)
{
  mConfig = Config::GetConfig();

  prepareDialog();
  
  int flags = wxPD_AUTO_HIDE | wxPD_APP_MODAL;
#if defined _WIN32 || defined __WIN32__ || defined __CYGWIN__
  // remaining time is buggy under Woe, displaying
  // "407890879:49879874:4987987 remaining" or so
#else
  flags |= wxPD_REMAINING_TIME;
#endif
  wxProgressDialog* lPrepareProgress = new wxProgressDialog(_("Preparing"),
    _("The D-Mod archive is being decompressed in a temporary file."), 100, this, flags);
  
  // Extract
  BZip lBZip(mDmodFilePath);
  mTarFilePath = lBZip.Extract( lPrepareProgress );
  lPrepareProgress->Destroy();
  
  if ( mTarFilePath.Len() == 0 )
    {
      mCancelButton->Enable(true);
      mExtracting = false;
    }
  else
    {
      // Prepare the tar file for reading
      mTar = new Tar(mTarFilePath);
      mTar->ReadHeaders();
     
      mSelectedDmod = mTar->getInstalledDmodDirectory();
      
      // Get and display the dmod description
      wxString lDmodDescription = mTar->getmDmodDescription();
      if ( lDmodDescription.Len() == 0 )
        {
	  lDmodDescription = wxString::Format(_("No Description Available.\n"
						"\n"
						"The D-Mod will be installed in subdirectory '%s'."),
					      mTar->getInstalledDmodDirectory().c_str());
        }
      else
        {
	  int lBreakChar = lDmodDescription.Find( '\r' );
	  if ( lBreakChar <= 0 )
            {
	      lBreakChar = lDmodDescription.Find( '\n' );
            }
	  mDmodName = lDmodDescription.SubString( 0, lBreakChar - 1 );
	  this->SetTitle(_("DFArc - Install D-Mod - ") + mDmodName);
        }
      mDmodDescription->SetValue(lDmodDescription);
      
      // Enable the cancel buttons and such
      mCancelButton->Enable(true);
      mInstallButton->Enable(true);
      mExtracting = false;
    }
}

InstallVerifyFrame::~InstallVerifyFrame()
{
  if (mTar != NULL)
    delete mTar;
}

void InstallVerifyFrame::prepareDialog(void)
{
  if (mConfig->mDModDir.IsEmpty())
    // force install in Dink dir
    mDestdirBox->Disable();
  else
    // default install in D-Mods dir
    mDestdirBox->SetSelection(1);

  // Disable installation button until the tar is properly decompressed
  mInstallButton->Enable(false);

  this->SetMinSize(this->GetBestSize());
  this->SetSize(this->GetMinSize());
}

void InstallVerifyFrame::onCancel(wxCommandEvent &Event)
{
  if (mExtracting == false)
    ::wxRemoveFile(mTarFilePath);
  this->EndModal(wxID_CANCEL);
}

void InstallVerifyFrame::onInstall(wxCommandEvent &Event)
{
  int flags = wxPD_AUTO_HIDE | wxPD_APP_MODAL;
#if defined _WIN32 || defined __WIN32__ || defined __CYGWIN__
  // remaining time is buggy under Woe, displaying
  // "407890879:49879874:4987987 remaining" or so
#else
  flags |= wxPD_REMAINING_TIME;
#endif
  wxProgressDialog* lInstallProgress = new wxProgressDialog(_("DFArc - Installing"), _T(""),
    100, this, flags);

  // Install in the main dink dir, or in the specified DMod dir if any
  wxString destdir = mConfig->GetDinkrefDir();
  if (mDestdirBox->GetSelection() == 1)
    destdir = mConfig->mDModDir;

  int lError = mTar->Extract(destdir, lInstallProgress);
  lInstallProgress->Destroy();
  if (lError == 0)
    {
      if (mDmodName.Len() <= 0)
        {
	  mDmodName = _("The D-Mod you selected");
        }
      ::wxMessageBox(mDmodName + _(" was successfully installed."),
		   _("Success"), wxOK | wxICON_INFORMATION, this);
      // Preselect this new D-Mod
      wxString dmod_dir = mTar->getInstalledDmodDirectory();
      if (dmod_dir.Last() == _T('/'))
	dmod_dir = dmod_dir.RemoveLast();
      mConfig->mSelectedDmod = destdir + wxFileName::GetPathSeparator() + dmod_dir;
    }
  else
    {
      ::wxLogError(_("An error occured while extracting the .dmod file."));
    }
  ::wxRemoveFile(mTarFilePath);
  // TODO: return error code instead of wxID_OK if something goes wrong
  this->EndModal(wxID_OK);
}

wxString InstallVerifyFrame::getSelectedDmod(void)
{
  return mSelectedDmod;
}
