/*
  Copyright (C) 2003-2009 FreeIPMI Core Team

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software Foundation,
  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
*/

#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#if STDC_HEADERS
#include <string.h>
#endif /* STDC_HEADERS */
#include <assert.h>

#include "bmc-config.h"
#include "bmc-config-map.h"
#include "bmc-config-validate.h"
#include "bmc-config-utils.h"

#include "freeipmi-portability.h"
#include "pstdout.h"

/* convenience struct */
struct bad_password_threshold
{
  uint8_t user_disabled_event_message;
  uint8_t bad_password_threshold_number;
  uint16_t attempt_count_reset_interval;
  uint16_t user_lockout_interval;
};

static config_err_t
_get_bad_password_threshold (bmc_config_state_data_t *state_data,
                             struct bad_password_threshold *bpt)
{
  fiid_obj_t obj_cmd_rs = NULL;
  uint64_t val;
  config_err_t rv = CONFIG_ERR_FATAL_ERROR;
  config_err_t ret;
  uint8_t channel_number;

  assert (state_data);
  assert (bpt);

  if (!(obj_cmd_rs = fiid_obj_create (tmpl_cmd_get_lan_configuration_parameters_bad_password_threshold_rs)))
    {
      pstdout_fprintf (state_data->pstate,
                       stderr,
                       "fiid_obj_create: %s\n",
                       strerror (errno));
      goto cleanup;
    }
  
  if ((ret = get_lan_channel_number (state_data, &channel_number)) != CONFIG_ERR_SUCCESS)
    {
      rv = ret;
      goto cleanup;
    }
  
  if (ipmi_cmd_get_lan_configuration_parameters_bad_password_threshold (state_data->ipmi_ctx,
                                                                        channel_number,
                                                                        IPMI_GET_LAN_PARAMETER,
                                                                        CONFIG_SET_SELECTOR,
                                                                        CONFIG_BLOCK_SELECTOR,
                                                                        obj_cmd_rs) < 0)
    {
      if (state_data->prog_data->args->config_args.common.debug)
        pstdout_fprintf (state_data->pstate,
                         stderr,
                         "ipmi_cmd_get_lan_configuration_parameters_bad_password_threshold: %s\n",
                         ipmi_ctx_errormsg (state_data->ipmi_ctx));
      
      if (config_is_config_param_non_fatal_error (state_data->ipmi_ctx,
                                                  obj_cmd_rs,
                                                  &ret))
        rv = ret;
      
      goto cleanup;
    }
  
  if (FIID_OBJ_GET (obj_cmd_rs, "user_disabled_event_message", &val) < 0)
    {
      pstdout_fprintf (state_data->pstate,
                       stderr,
                       "fiid_obj_get: 'user_disabled_event_message': %s\n",
                       fiid_obj_errormsg (obj_cmd_rs));
      goto cleanup;
    }
  bpt->user_disabled_event_message = val;

  if (FIID_OBJ_GET (obj_cmd_rs, "bad_password_threshold_number", &val) < 0)
    {
      pstdout_fprintf (state_data->pstate,
                       stderr,
                       "fiid_obj_get: 'bad_password_threshold_number': %s\n",
                       fiid_obj_errormsg (obj_cmd_rs));
      goto cleanup;
    }
  bpt->bad_password_threshold_number = val;

  if (FIID_OBJ_GET (obj_cmd_rs, "attempt_count_reset_interval", &val) < 0)
    {
      pstdout_fprintf (state_data->pstate,
                       stderr,
                       "fiid_obj_get: 'attempt_count_reset_interval': %s\n",
                       fiid_obj_errormsg (obj_cmd_rs));
      goto cleanup;
    }
  bpt->attempt_count_reset_interval = val;

  if (FIID_OBJ_GET (obj_cmd_rs, "user_lockout_interval", &val) < 0)
    {
      pstdout_fprintf (state_data->pstate,
                       stderr,
                       "fiid_obj_get: 'user_lockout_interval': %s\n",
                       fiid_obj_errormsg (obj_cmd_rs));
      goto cleanup;
    }
  bpt->user_lockout_interval = val;

  rv = CONFIG_ERR_SUCCESS;
 cleanup:
  fiid_obj_destroy (obj_cmd_rs);
  return (rv);
}

static config_err_t
_set_bad_password_threshold (bmc_config_state_data_t *state_data,
                             struct bad_password_threshold *bpt)
{
  fiid_obj_t obj_cmd_rs = NULL;
  config_err_t rv = CONFIG_ERR_FATAL_ERROR;
  config_err_t ret;
  uint8_t channel_number;

  assert (state_data);
  assert (bpt);

  if (!(obj_cmd_rs = fiid_obj_create (tmpl_cmd_set_lan_configuration_parameters_rs)))
    {
      pstdout_fprintf (state_data->pstate,
                       stderr,
                       "fiid_obj_create: %s\n",
                       strerror (errno));
      goto cleanup;
    }

  if ((ret = get_lan_channel_number (state_data, &channel_number)) != CONFIG_ERR_SUCCESS)
    {
      rv = ret;
      goto cleanup;
    }

  if (ipmi_cmd_set_lan_configuration_parameters_bad_password_threshold (state_data->ipmi_ctx,
                                                                        channel_number,
                                                                        bpt->user_disabled_event_message,
                                                                        bpt->bad_password_threshold_number,
                                                                        bpt->attempt_count_reset_interval,
                                                                        bpt->user_lockout_interval,
                                                                        obj_cmd_rs) < 0)
    {
      if (state_data->prog_data->args->config_args.common.debug)
        pstdout_fprintf (state_data->pstate,
                         stderr,
                         "ipmi_cmd_set_lan_configuration_parameters_bad_password_threshold: %s\n",
                         ipmi_ctx_errormsg (state_data->ipmi_ctx));

      if (config_is_config_param_non_fatal_error (state_data->ipmi_ctx,
                                                  obj_cmd_rs,
                                                  &ret))
        rv = ret;

      goto cleanup;
    }

  rv = CONFIG_ERR_SUCCESS;
 cleanup:
  fiid_obj_destroy (obj_cmd_rs);
  return (rv);

}

static config_err_t
bad_password_threshold_checkout (const char *section_name,
                                                  struct config_keyvalue *kv,
                                                  void *arg)
{
  bmc_config_state_data_t *state_data = (bmc_config_state_data_t *)arg;
  struct bad_password_threshold bpt;
  config_err_t ret;
  
  if ((ret = _get_bad_password_threshold (state_data, &bpt)) != CONFIG_ERR_SUCCESS)
    return (ret);
  
  if (config_section_update_keyvalue_output_unsigned_int (state_data->pstate,
                                                          kv,
                                                          bpt.bad_password_threshold_number) < 0)
    return (CONFIG_ERR_FATAL_ERROR);
  
  return (CONFIG_ERR_SUCCESS);
}

static config_err_t
bad_password_threshold_commit (const char *section_name,
                                                const struct config_keyvalue *kv,
                                                void *arg)
{
  bmc_config_state_data_t *state_data = (bmc_config_state_data_t *)arg;
  struct bad_password_threshold bpt;
  config_err_t ret;

  if ((ret = _get_bad_password_threshold (state_data, &bpt)) != CONFIG_ERR_SUCCESS)
    return (ret);
  
  bpt.bad_password_threshold_number = atoi (kv->value_input);
  return (_set_bad_password_threshold (state_data, &bpt));
}

static config_err_t
attempt_count_reset_interval_checkout (const char *section_name,
                                       struct config_keyvalue *kv,
                                       void *arg)
{
  bmc_config_state_data_t *state_data = (bmc_config_state_data_t *)arg;
  struct bad_password_threshold bpt;
  config_err_t ret;
  
  if ((ret = _get_bad_password_threshold (state_data, &bpt)) != CONFIG_ERR_SUCCESS)
    return (ret);
  
  if (config_section_update_keyvalue_output_unsigned_int (state_data->pstate,
                                                          kv,
                                                          bpt.attempt_count_reset_interval) < 0)
    return (CONFIG_ERR_FATAL_ERROR);
  
  return (CONFIG_ERR_SUCCESS);
}

static config_err_t
attempt_count_reset_interval_commit (const char *section_name,
                                     const struct config_keyvalue *kv,
                                     void *arg)
{
  bmc_config_state_data_t *state_data = (bmc_config_state_data_t *)arg;
  struct bad_password_threshold bpt;
  config_err_t ret;

  if ((ret = _get_bad_password_threshold (state_data, &bpt)) != CONFIG_ERR_SUCCESS)
    return (ret);
  
  bpt.attempt_count_reset_interval = atoi (kv->value_input);
  return (_set_bad_password_threshold (state_data, &bpt));
}

static config_err_t
user_lockout_interval_checkout (const char *section_name,
                                struct config_keyvalue *kv,
                                void *arg)
{
  bmc_config_state_data_t *state_data = (bmc_config_state_data_t *)arg;
  struct bad_password_threshold bpt;
  config_err_t ret;
  
  if ((ret = _get_bad_password_threshold (state_data, &bpt)) != CONFIG_ERR_SUCCESS)
    return (ret);
  
  if (config_section_update_keyvalue_output_unsigned_int (state_data->pstate,
                                                          kv,
                                                          bpt.user_lockout_interval) < 0)
    return (CONFIG_ERR_FATAL_ERROR);
  
  return (CONFIG_ERR_SUCCESS);
}

static config_err_t
user_lockout_interval_commit (const char *section_name,
                              const struct config_keyvalue *kv,
                              void *arg)
{
  bmc_config_state_data_t *state_data = (bmc_config_state_data_t *)arg;
  struct bad_password_threshold bpt;
  config_err_t ret;

  if ((ret = _get_bad_password_threshold (state_data, &bpt)) != CONFIG_ERR_SUCCESS)
    return (ret);
  
  bpt.user_lockout_interval = atoi (kv->value_input);
  return (_set_bad_password_threshold (state_data, &bpt));
}

static config_err_t
enable_event_message_when_user_disabled_checkout (const char *section_name,
                                                  struct config_keyvalue *kv,
                                                  void *arg)
{
  bmc_config_state_data_t *state_data = (bmc_config_state_data_t *)arg;
  struct bad_password_threshold bpt;
  config_err_t ret;
  
  if ((ret = _get_bad_password_threshold (state_data, &bpt)) != CONFIG_ERR_SUCCESS)
    return (ret);
  
  if (config_section_update_keyvalue_output (state_data->pstate,
                                             kv,
                                             bpt.user_disabled_event_message ? "Yes" : "No") < 0)
    return (CONFIG_ERR_FATAL_ERROR);
  
  return (CONFIG_ERR_SUCCESS);
}

static config_err_t
enable_event_message_when_user_disabled_commit (const char *section_name,
                                                const struct config_keyvalue *kv,
                                                void *arg)
{
  bmc_config_state_data_t *state_data = (bmc_config_state_data_t *)arg;
  struct bad_password_threshold bpt;
  config_err_t ret;

  if ((ret = _get_bad_password_threshold (state_data, &bpt)) != CONFIG_ERR_SUCCESS)
    return (ret);
  
  bpt.user_disabled_event_message = same (kv->value_input, "yes");
  return (_set_bad_password_threshold (state_data, &bpt));
}

struct config_section *
bmc_config_lan_conf_user_security_section_get (bmc_config_state_data_t *state_data)
{
  struct config_section *section = NULL;
  char *section_comment =
    "The following user security configuration options are optionally "
    "implemented by the vendor.  They may not be available your system and "
    "may not be visible below."
    "\n"
    "The following configuration supports the ability for the BMC to "
    "disable a user if a number of bad passwords are entered sequentially. "
    "\"Bad_Password_Threshold\" determines the number of bad passwords that "
    "must be entered sequentially.  \"Attempt_Count_Reset_Interval\" determines "
    "the range of time the bad passwords must occur in.  \"User_Lockout_Interval\" "
    "determines the time a user will be locked off if the bad password "
    "threshold is reached.  If set to \"Yes\", \"Enable_Event_Message_When_User_Disabled\" "
    "will inform the BMC to log an event message when a user is disabled.";

  if (!(section = config_section_create (state_data->pstate,
                                         "Lan_Conf_User_Security",
                                         "Lan_Conf_User_Security",
                                         section_comment,
                                         0,
                                         NULL,
                                         NULL)))
    goto cleanup;

  if (config_section_add_key (state_data->pstate,
                              section,
                              "Bad_Password_Threshold",
                              "Possible values: 0-255, 0 indicates no limit",
                              0,
                              bad_password_threshold_checkout,
                              bad_password_threshold_commit,
                              config_number_range_one_byte) < 0)
    goto cleanup;

  if (config_section_add_key (state_data->pstate,
                              section,
                              "Attempt_Count_Reset_Interval",
                              "Possible values: 0-65535, in 10 second increments (e.g. 2 = 20 sec)\n"
                              "                 0 indicates no interval (i.e. don't reset counter)",
                              0,
                              attempt_count_reset_interval_checkout,
                              attempt_count_reset_interval_commit,
                              config_number_range_two_bytes) < 0)
    goto cleanup;

  if (config_section_add_key (state_data->pstate,
                              section,
                              "User_Lockout_Interval",
                              "Possible values: 0-65535, in 10 second increments (e.g. 2 = 20 sec)\n"
                              "                 0 indicates no interval (i.e. don't re-enable user)",
                              0,
                              user_lockout_interval_checkout,
                              user_lockout_interval_commit,
                              config_number_range_two_bytes) < 0)
    goto cleanup;

  if (config_section_add_key (state_data->pstate,
                              section,
                              "Enable_Event_Message_When_User_Disabled",
                              "Possible values: Yes/No",
                              0,
                              enable_event_message_when_user_disabled_checkout,
                              enable_event_message_when_user_disabled_commit,
                              config_yes_no_validate) < 0)
    goto cleanup;

  return (section);

 cleanup:
  if (section)
    config_section_destroy (state_data->pstate, section);
  return (NULL);
}
