/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <gr_fir_sysconfig_x86.h>
#include <gr_cpu.h>

#include <gr_fir_CCF.h>
#include <gr_fir_FCC.h>
#include <gr_fir_FFF.h>
#include <gr_fir_FFF_generic.h>
#include <gr_fir_FFF_x86.h>
#include <gr_fir_FSF.h>
#include <gr_fir_FSF_generic.h>
#include <gr_fir_FSF_x86.h>
#include <gr_fir_SCC.h>
#include <gr_fir_SCC_generic.h>
#include <gr_fir_SCC_x86.h>
// #include <gr_fir_SSS.h>
// #include <gr_fir_SSS_generic.h>
// #include <gr_fir_SSS_mmx.h>
// #include <gr_fir_SSS_sse2.h>

#include <iostream>
using std::cerr;

/*
 * ----------------------------------------------------------------
 * static functions that serve as constructors...
 * Is it possible to take the address of a normal constructor?
 * ----------------------------------------------------------------
 */

static gr_fir_FFF *
make_gr_fir_FFF_3dnow (const std::vector<float> &taps)
{
  return new gr_fir_FFF_3dnow (taps);
}

static gr_fir_FFF *
make_gr_fir_FFF_sse (const std::vector<float> &taps)
{
  return new gr_fir_FFF_sse (taps);
}

static gr_fir_FSF *
make_gr_fir_FSF_3dnow (const std::vector<float> &taps)
{
  return new gr_fir_FSF_3dnow (taps);
}

static gr_fir_FSF *
make_gr_fir_FSF_sse (const std::vector<float> &taps)
{
  return new gr_fir_FSF_sse (taps);
}

#if 0
static gr_fir_SSS *
make_gr_fir_SSS_mmx (const std::vector<short> &taps)
{
  return new gr_fir_SSS_mmx (taps);
}

static gr_fir_SSS *
make_gr_fir_SSS_sse2 (const std::vector<short> &taps)
{
  return new gr_fir_SSS_sse2 (taps);
}
#endif

static gr_fir_SCC *
make_gr_fir_SCC_3dnow(const std::vector<VrComplex> &taps)
{
  return new gr_fir_SCC_3dnow(taps);
}

static gr_fir_SCC *
make_gr_fir_SCC_3dnowext(const std::vector<VrComplex> &taps)
{
  return new gr_fir_SCC_3dnowext(taps);
}

static gr_fir_SCC *
make_gr_fir_SCC_sse(const std::vector<VrComplex> &taps)
{
  return new gr_fir_SCC_sse(taps);
}

/*
 * ----------------------------------------------------------------
 * Return instances of the fastest x86 versions of these classes.
 *
 * check CPUID, if has 3DNowExt, return 3DNow!Ext version,
 *              else if 3DNow, return 3DNow! version,
 *              else if SSE2, return SSE2 version,
 *		else if SSE, return SSE version,
 *		else if MMX, return MMX version,
 *		else return generic version.
 *
 * FIXME: benchmark, store result, use stored result to
 *   select the fastest version.
 * ----------------------------------------------------------------
 */

gr_fir_FFF *
gr_fir_sysconfig_x86::create_gr_fir_FFF (const std::vector<float> &taps)
{
  static bool first = true;

  if (gr_cpu::has_3dnow ()){
    if (first) {
      cerr << ">>> gr_fir_FFF: using 3DNow!\n";
      first = false;
    }
    return make_gr_fir_FFF_3dnow (taps);
  }

  if (gr_cpu::has_sse ()){
    if (first){
      cerr << ">>> gr_fir_FFF: using SSE\n";
      first = false;
    }
    return make_gr_fir_FFF_sse (taps);
  }
  
  if (first){
    cerr << ">>> gr_fir_FFF: handing off to parent class\n";
    first = false;
  }
  return gr_fir_sysconfig_generic::create_gr_fir_FFF (taps);
}

gr_fir_FSF *
gr_fir_sysconfig_x86::create_gr_fir_FSF (const std::vector<float> &taps)
{
  static bool first = true;

  if (gr_cpu::has_3dnow ()){
    if (first) {
      cerr << ">>> gr_fir_FSF: using 3DNow!\n";
      first = false;
    }
    return make_gr_fir_FSF_3dnow (taps);
  }

  if (gr_cpu::has_sse ()){
    if (first){
      cerr << ">>> gr_fir_FSF: using SSE\n";
      first = false;
    }
    return make_gr_fir_FSF_sse (taps);
  }
  
  if (first){
    cerr << ">>> gr_fir_FSF: handing off to parent class\n";
    first = false;
  }
  return gr_fir_sysconfig_generic::create_gr_fir_FSF (taps);
}

#if 0
gr_fir_SSS *
gr_fir_sysconfig_x86::create_gr_fir_SSS (const std::vector<short> &taps)
{
  // FIXME -- probably want to figure out best answer for Athlon and code
  // add code to select it here...

  if (gr_cpu::has_sse2 ()){
    cerr << ">>> gr_fir_SSS: using SSE2\n";
    return make_gr_fir_SSS_sse2 (taps);
  }
  
  if (gr_cpu::has_mmx ()){
    cerr << ">>> gr_fir_SSS: using MMX\n";
    return make_gr_fir_SSS_mmx (taps);
  }
  
  cerr << ">>> gr_fir_SSS: handing off to parent class\n";
  return gr_fir_sysconfig_generic::create_gr_fir_SSS (taps);
}
#endif

gr_fir_SCC *
gr_fir_sysconfig_x86::create_gr_fir_SCC (const std::vector<VrComplex> &taps)
{
  static bool first = true;

  if (gr_cpu::has_3dnowext ()){
    if (first){
      cerr << ">>> gr_fir_SCC: using 3DNow!Ext\n";
      first = false;
    }
    return make_gr_fir_SCC_3dnowext (taps);
  }

  if (gr_cpu::has_3dnow ()){
    if (first){
      cerr << ">>> gr_fir_SCC: using 3DNow!\n";
      first = false;
    }
    return make_gr_fir_SCC_3dnow (taps);
  }

  if (gr_cpu::has_sse ()){
    if (first){
      cerr << ">>> gr_fir_SCC: using SSE\n";
      first = false;
    }
    return make_gr_fir_SCC_sse (taps);
  }

  if (first){
    cerr << ">>> gr_fir_SCC: handing off to parent class\n";
    first = false;
  }
  return gr_fir_sysconfig_generic::create_gr_fir_SCC (taps);
}

/*
 * ----------------------------------------------------------------
 *         Return info about available implementations
 * ----------------------------------------------------------------
 */

void 
gr_fir_sysconfig_x86::get_gr_fir_CCF_info (std::vector<gr_fir_CCF_info> *info)
{
  // invoke parent..
  gr_fir_sysconfig_generic::get_gr_fir_CCF_info (info);

  // ...add our stuff
}

void 
gr_fir_sysconfig_x86::get_gr_fir_FCC_info (std::vector<gr_fir_FCC_info> *info)
{
  // invoke parent..
  gr_fir_sysconfig_generic::get_gr_fir_FCC_info (info);

  // add our stuff...
}

void 
gr_fir_sysconfig_x86::get_gr_fir_FFF_info (std::vector<gr_fir_FFF_info> *info)
{
  gr_fir_FFF_info	t;
  
  // invoke parent..
  gr_fir_sysconfig_generic::get_gr_fir_FFF_info (info);

  // add our stuff...
  if (gr_cpu::has_3dnow ()){
    t.name = "3DNow!";
    t.create = make_gr_fir_FFF_3dnow;
    (*info).push_back (t);
  }

  if (gr_cpu::has_sse ()){
    t.name = "SSE";
    t.create = make_gr_fir_FFF_sse;
    (*info).push_back (t);
  }
}

void 
gr_fir_sysconfig_x86::get_gr_fir_FSF_info (std::vector<gr_fir_FSF_info> *info)
{
  gr_fir_FSF_info	t;
  
  // invoke parent..
  gr_fir_sysconfig_generic::get_gr_fir_FSF_info (info);

  // add our stuff...
  if (gr_cpu::has_3dnow ()){
    t.name = "3DNow!";
    t.create = make_gr_fir_FSF_3dnow;
    (*info).push_back (t);
  }

  if (gr_cpu::has_sse ()){
    t.name = "SSE";
    t.create = make_gr_fir_FSF_sse;
    (*info).push_back (t);
  }
}

void 
gr_fir_sysconfig_x86::get_gr_fir_SCC_info (std::vector<gr_fir_SCC_info> *info)
{
  gr_fir_SCC_info	t;

  // invoke parent..
  gr_fir_sysconfig_generic::get_gr_fir_SCC_info (info);

  // add our stuff...
  if (gr_cpu::has_3dnowext ()){
    t.name = "3DNow!Ext";
    t.create = make_gr_fir_SCC_3dnowext;
    (*info).push_back (t);
  }

  if (gr_cpu::has_3dnow ()){
    t.name = "3DNow!";
    t.create = make_gr_fir_SCC_3dnow;
    (*info).push_back (t);
  }

  if (gr_cpu::has_sse ()){
    t.name = "SSE";
    t.create = make_gr_fir_SCC_sse;
    (*info).push_back (t);
  }
}

#if 0
void 
gr_fir_sysconfig_x86::get_gr_fir_SSS_info (std::vector<gr_fir_SSS_info> *info)
{
  gr_fir_SSS_info	t;
  
  // invoke parent..
  gr_fir_sysconfig_generic::get_gr_fir_SSS_info (info);

  // add our stuff...
  if (gr_cpu::has_mmx ()){
    t.name = "MMX";
    t.create = make_gr_fir_SSS_mmx;
    (*info).push_back (t);
  }

  if (gr_cpu::has_sse2 ()){
    t.name = "SSE2";
    t.create = make_gr_fir_SSS_sse2;
    (*info).push_back (t);
  }
}
#endif
