/*
 *  rtpcommon.h
 *
 * Copyright (c) 1995-98 University College London
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, is permitted provided that the following conditions 
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Computer Science
 *      Department at University College London
 * 4. Neither the name of the University nor of the Department may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <netinet/in.h> 
#include <sys/socket.h> 
#include <arpa/inet.h>
#include <assert.h>

#define DIFF_BYTE_ORDER  1

typedef short sample;       /* Sample representation 16 bit signed */
typedef struct {
	unsigned long ticks:25;
	unsigned long check:3;
	unsigned long xxidx:4;
} ts_t;
typedef struct {
	ts_t    last_ts;
	unsigned long last_32;
} ts_sequencer;
typedef struct s_fast_time {
} ft_t;

typedef struct s_time {
	struct s_fast_time *ft;
} frtime_t;
typedef struct {
	unsigned long	 ssrc;
	unsigned long	 addr;
	char	*cname;
	char	*name;
	char	*email;
	char	*phone;
	char	*note;
	char	*loc;
	char	*txt;
	char	*tool;
} ssrc_entry;

typedef struct {
	unsigned long	ssrc;		/* SSRC this report is about	         */
	unsigned long	loss;		/* frac(8) + cumulative(24)                  */
	unsigned long	last_seq;	/* Extended highest sequence number received */
	unsigned long	jitter;		/* interarrival jitter                       */
	unsigned long	lsr;		/* last SR packet from this source           */
	unsigned long	dlsr;		/* delay since last SR packet	        */
} rtcp_rr_t;

typedef struct {
	unsigned long	ssrc;		/* source this RTCP packet refers to */
	unsigned long	ntp_sec;	/* NTP timestamp                     */
	unsigned long	ntp_frac;
	unsigned long	rtp_ts;		/* RTP timestamp */
	unsigned long	sender_pcount;
	unsigned long	sender_bcount;
} sender_rpt;

#define RTP_SEQ_MOD (1<<16)
#define RTP_TS_MOD  (0xffffffff)
#define RTP_MAX_SDES 256	/* maximum text length for SDES */ 
#define RTP_NUM_SDES 		6

typedef enum {
	RTCP_SR = 200, RTCP_RR, RTCP_SDES, RTCP_BYE, RTCP_APP
} rtcp_type_t;

typedef enum {
	RTCP_SDES_INVALID, RTCP_SDES_CNAME, RTCP_SDES_NAME, RTCP_SDES_EMAIL, RTCP_SDES_PHONE,
	RTCP_SDES_LOC, RTCP_SDES_TOOL, RTCP_SDES_NOTE, RTCP_SDES_PRIV
} rtcp_sdes_type_t;

typedef struct {
#ifndef DIFF_BYTE_ORDER
	unsigned short  type:2;	/* packet type            */
	unsigned short  p:1;	/* padding flag           */
	unsigned short  x:1;	/* header extension flag  */
	unsigned short  cc:4;	/* CSRC count	     */
	unsigned short  m:1;	/* marker bit	     */
	unsigned short  pt:7;	/* payload type           */
#else
	unsigned short  cc:4;	/* CSRC count	     */
	unsigned short  x:1;	/* header extension flag  */
	unsigned short  p:1;	/* padding flag           */
	unsigned short  type:2;	/* packet type            */
	unsigned short  pt:7;	/* payload type           */
	unsigned short  m:1;	/* marker bit	     */
#endif
	unsigned short         seq;	/* sequence number        */
	unsigned long	 ts;	/* timestamp              */
	unsigned long	 ssrc;	/* synchronization source */
	unsigned long	 csrc[16];/* optional CSRC list     */
} rtp_hdr_t;

typedef struct {
#ifndef DIFF_BYTE_ORDER
	unsigned short  type:2;	/* packet type            */
	unsigned short  p:1;	/* padding flag           */
	unsigned short  count:5;/* varies by payload type */
#else
	unsigned short  count:5;/* varies by payload type */
	unsigned short  p:1;	/* padding flag           */
	unsigned short  type:2;	/* packet type            */
#endif
	unsigned short  pt:8;	/* payload type           */
	unsigned short         length;	/* packet length          */
} rtcp_common_t;

typedef struct {
	unsigned char          type;	/* type of SDES item (rtcp_sdes_type_t) */
	unsigned char          length;	/* length of SDES item (in bytes)       */
	char	    data[1];/* text, not zero-terminated            */
} rtcp_sdes_item_t;

typedef struct {
	rtcp_common_t   common;	
	union {
		struct {
			unsigned long	 ssrc;		/* source this RTCP packet refers to */
			unsigned long	 ntp_sec;	/* NTP timestamp */
			unsigned long	 ntp_frac;
			unsigned long	 rtp_ts;		/* RTP timestamp */
			unsigned long	 sender_pcount;
			unsigned long	 sender_bcount;
			rtcp_rr_t       rr[1];		/* variable-length list */
		} sr;
		struct {
			unsigned long	 ssrc;		/* source this RTCP packet is coming from */
			rtcp_rr_t       rr[1];		/* variable-length list */
		} rr;
		struct {
			unsigned long	 src[1];		/* list of sources */
		} bye;
		struct rtcp_sdes_t {
			rtcp_sdes_item_t s[1];		/* list of SDES */
		} sdes;
	} r;
} rtcp_t;

typedef struct _rr {
	struct _rr *next;
	unsigned long	  ssrc;
	unsigned char    fraction_lost;
	unsigned long	  pckts_lost;
	unsigned long	  ext_seq_num;
	unsigned long	  jitter;
	unsigned long	  lsr;
	unsigned long	  dlsr;
} rtcp_user_rr;

typedef struct s_rtcp_dbentry {
	struct s_rtcp_dbentry *next; 
	unsigned long	ssrc;
	ssrc_entry     *sentry;
	unsigned long	pckts_recv;
	unsigned short lastseqno;
	unsigned long	cycles;
	rtcp_user_rr	*rr; 
	unsigned long	misordered;		
	unsigned long	duplicates;
	unsigned char	first_pckt_flag:1;
	struct s_time  *dbclock;
	ts_t	last_mixed;             /* Used to check mixing */
	unsigned short last_seq;		/* Last packet sequence number */
	unsigned long	expected_prior;
	unsigned long	received_prior;
	unsigned long	lost_frac;
	long	lost_tot; 
	unsigned long	bad_seq;
	char	probation;
	unsigned long	last_sr;
	unsigned long	last_sr_rx;	
	unsigned char	is_sender;
	int	    mapping_valid;
	unsigned long	last_ntp_sec;	/* NTP timestamp */
	unsigned long	last_ntp_frac;
	unsigned long	last_rtp_ts;	/* RTP timestamp */
	unsigned long age;
#define BUFFER_LENGTH	3000
	unsigned char *buffer, *inputp, *outputp;
	unsigned long last_ts;
} rtcp_dbentry;

typedef struct rtp_db_tag {
	unsigned long	myssrc;
	unsigned long	pkt_count, byte_count, pckts_received, misordered, duplicates;
	rtcp_dbentry *my_dbe;	/* For my info in UI 			*/
	unsigned long	last_rpt;		/* Time of last RTCP report we sent	*/
	unsigned long	report_interval;/* RTCP reporting interval              */
	unsigned long	map_rtp_time;
	int		initial_rtcp;	/* TRUE until we've sent an RTCP packet */
	unsigned long	sdes_pri_count, sdes_sec_count, sdes_ter_count;
	unsigned short		rtp_seq;
} rtp_db;

static int rtcp_check_rtcp_pkt(unsigned char *packet, int len);
static unsigned char * rtcp_packet_fmt_sdes(unsigned char * ptr);
static void rtcp_decode_rtcp_pkt(unsigned char *packet, int len);
static struct s_rtcp_dbentry   *get_source(unsigned long ssrc);
static unsigned char * rtcp_packet_fmt_srrr(unsigned char *ptr);
static struct s_rtcp_dbentry *find_source(unsigned long ssrc);
