#!/usr/bin/env python

from gnuradio import gr, eng_notation
from gnuradio import audio
from gnuradio import usrp
from gnuradio.eng_option import eng_option
from optparse import OptionParser
import sys
import math

from gnuradio.wxgui import stdgui, fftsink
import wx

class wfm_rx_graph (stdgui.gui_flow_graph):
    def __init__(self,frame,panel,vbox,argv):
        stdgui.gui_flow_graph.__init__ (self,frame,panel,vbox,argv)

        IF_freq = parseargs(argv[1:])
        adc_rate = 64e6

        decim = 250
        quad_rate = adc_rate / decim               # 256 kHz
        audio_decimation = 8
        audio_rate = quad_rate / audio_decimation  # 32 kHz
        
        #decim = 125
        #quad_rate = adc_rate / decim               # 512 kHz
        #audio_decimation = 16
        #audio_rate = quad_rate / audio_decimation  # 32 kHz
        
        # usrp is data source
        src = usrp.source_c (0, decim)
        src.set_rx_freq (0, IF_freq)
        src.set_pga(0,20)
        (head, tail) = self.build_pipeline (quad_rate, audio_decimation, vbox, panel)
        # sound card as final sink
        audio_sink = audio.sink (int (audio_rate))
        
        # now wire it all together
        self.connect (src, head)
        self.connect (tail, (audio_sink, 0))
        
        pre_demod, fft_win1 = fftsink.make_fft_sink_c (self, panel, "Pre-Demodulation", 512, quad_rate)
        self.connect (src, pre_demod)
        vbox.Add (fft_win1, 1, wx.EXPAND)

    def build_pipeline (self, quad_rate, audio_decimation, vbox, panel):
        '''Given a flow_graph, fg, construct a pipeline
        for demodulating a broadcast FM signal.  The
        input is the downconverteed complex baseband
        signal. The output is the demodulated audio.
        
        build_pipeline returns a two element tuple
        containing the input and output endpoints.
        '''
        fm_demod_gain = 2200.0/32768.0
        audio_rate = quad_rate / audio_decimation
        volume = 1.0
        
        # input: complex; output: float
        fm_demod = gr.quadrature_demod_cf (volume*fm_demod_gain)
        
        # compute FIR filter taps for audio filter
        width_of_transition_band = audio_rate / 32
        audio_coeffs = gr.firdes.low_pass (1.0,            # gain
                                           quad_rate,      # sampling rate
                                           audio_rate/2 - width_of_transition_band,
                                           width_of_transition_band,
                                           gr.firdes.WIN_HAMMING)
        
        TAU  = 75e-6  # 75us in US, 50us in EUR
        fftaps = [ 1 - math.exp(-1/TAU/quad_rate), 0]
        fbtaps= [ 0 , math.exp(-1/TAU/quad_rate) ]
        deemph = gr.iir_filter_ffd(fftaps,fbtaps)

        # input: float; output: float
        audio_filter = gr.fir_filter_fff (audio_decimation, audio_coeffs)
        
        self.connect (fm_demod, deemph)
        self.connect (deemph, audio_filter)
        
        #post_demod, fft_win2 = fftsink.make_fft_sink_f (self, panel, "Post Demodulation", 128, quad_rate)
        #self.connect (fm_demod, post_demod)
        #vbox.Add (fft_win2, 1, wx.EXPAND)
        
        post_deemph, fft_win3 = fftsink.make_fft_sink_f (self, panel, "With Deemph", 512, quad_rate)
        self.connect (deemph,post_deemph)
        vbox.Add (fft_win3, 1, wx.EXPAND)

        post_filt, fft_win4 = fftsink.make_fft_sink_f (self, panel, "Post Filter", 512, audio_rate)
        self.connect (audio_filter,post_filt)
        vbox.Add (fft_win4, 1, wx.EXPAND)
        
        return ((fm_demod, 0), (audio_filter, 0))

def parseargs (args):
    nargs = len (args)
    if nargs == 1:
        freq1 = float (args[0]) * 1e6
    else:
        sys.stderr.write ('usage: wfm_rcv freq1\n')
        sys.exit (1)

    return freq1 - 128e6

if __name__ == '__main__':
    app = stdgui.stdapp (wfm_rx_graph, "WFM RX")
    app.MainLoop ()

    # connect to RF front end
    #rf_front_end = gr.microtune_4937_eval_board ()
    #if not rf_front_end.board_present_p ():
    #    raise IOError, 'RF front end not found'

    # set front end gain
    #rf_front_end.set_AGC (300)
    # rf_front_end.get_output_freq ()
    #IF_freq = 5.75e6

    #rf_front_end.set_RF_freq (freq1)

    #fg = build_graph (IF_freq)


    #fg.start ()        # fork thread(s) and return
    #raw_input ('Press Enter to quit: ')
    #fg.stop ()

