#!/usr/bin/env python

# GMSK modulation and demodulation test.
#
#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 
# $Id: ggmsk-test.py,v 1.1 2005/03/29 22:15:45 eb Exp $

import sys
from math import pi
from time import time
from optparse import OptionParser

import wx
from gnuradio import gr
from gnuradio.wxgui import stdgui, fftsink, scopesink
from gnuradio.blks import gmsk_mod, gmsk_demod
from gnuradio.eng_option import eng_option

class gmsk_mod_demod_gui(stdgui.gui_flow_graph):
	def __init__(self, frame, panel, vbox, argv):
		stdgui.gui_flow_graph.__init__(self, frame, panel, vbox, argv)
		frame.SetSize((1200, 1000))
		self.frame = frame
		self.panel = panel

		parser = OptionParser(option_class = eng_option)
		parser.add_option("-s", "--samples-per-symbol", type = "intx",
		   default = 8, help = "samples per symbol (integer)")
		parser.add_option("-r", "--symbols-per-second",
		   type = "eng_float", default = 1625000.0 / 6.0,
		   help = "symbols per second")
		parser.add_option("-p", "--packet-size", type = "intx",
		   default = 1024, help = "data block size")
		parser.add_option("-i", "--if-freq", type = "eng_float",
		   default = 7.5e5, help = "intermediate frequency")
		parser.add_option("-w", "--filter-width", type = "eng_float",
		   default = 2.5e5, help = "low pass filter width")
		parser.add_option("-t", "--transition-width",
		   type = "eng_float", default = 5e5,
		   help = "transition width of low pass filter")
		parser.add_option("-n", "--noise", type = "eng_float",
		   default = 0.0, help = "add AWGN noise (mag)")
		parser.add_option("-f", "--filename", type = "string",
		   default = "./payload.dat", help = "file name of input data")
		(options, args) = parser.parse_args()

		sps = options.samples_per_symbol
		symbol_rate = options.symbols_per_second
		sample_rate = sps * symbol_rate
		p_size = options.packet_size
		lo_freq = options.if_freq
		lp_cutoff = options.filter_width
		lp_tw = options.transition_width
		noise_mag = options.noise
		filename = options.filename

		print "sps:\t\t" + str(sps)
		print "symbol_rate:\t" + str(symbol_rate)
		print "sample_rate:\t" + str(sample_rate)
		print "p_size:\t\t" + str(p_size)
		print "lo_freq:\t" + str(lo_freq)
		print "lp_cutoff:\t" + str(lp_cutoff)
		print "lp_tw:\t\t" + str(lp_tw)
		print "noise_mag:\t" + str(noise_mag)
		print "filename:\t" + filename

		# File source (repeats)
		src = gr.file_source(1, filename, 1)

		# GMSK modulate input bytes to baseband
		self.mod = gmsk_mod(self, sps, symbol_rate, 0.3, p_size)

		# Display GMSK modulated bits
		gmsk_data_tx_block, gmsk_data_tx_win = \
		   scopesink.make_scope_sink_f(self, panel, "GMSK tx data",
		      sample_rate)
		vbox.Add(gmsk_data_tx_win, 1, wx.EXPAND)
		self.connect(self.mod.gaussian_filter, gmsk_data_tx_block)

		# Display GMSK baseband
		gmsk_fft_block, gmsk_fft_win = fftsink.make_fft_sink_c(self,
		   panel, "Baseband FFT", 1024, sample_rate)
		vbox.Add(gmsk_fft_win, 1, wx.EXPAND)
		self.connect(self.mod.fmmod, gmsk_fft_block)

		# Mix to IF
		lo = gr.sig_source_c(sample_rate, gr.GR_SIN_WAVE, lo_freq,
		   1.0, 0)
		mixer = gr.multiply_cc()
		self.connect(lo, (mixer, 1))

		# Take real part as transmit
		ctof = gr.complex_to_float()

		# Simulate noise in the channel
		noise = gr.noise_source_f(gr.GR_GAUSSIAN, noise_mag)
		air_noise = gr.add_ff()
		self.connect(noise, (air_noise, 1))

		# IF spectrum
		if_fft_block, if_fft_win = fftsink.make_fft_sink_f(self,
		   panel, "IF FFT", 1024, sample_rate)
		vbox.Add(if_fft_win, 1, wx.EXPAND)
		self.connect(air_noise, if_fft_block)

		# Mix to baseband
		chan_taps = gr.firdes.low_pass(1.0, sample_rate, lp_cutoff,
		   lp_tw, gr.firdes.WIN_HAMMING)
		chan_filter = gr.freq_xlating_fir_filter_fcf(1, chan_taps,
		   -lo_freq, sample_rate)

		# GMSK demodulate baseband to bytes
		self.demod = gmsk_demod(self, sps, symbol_rate, p_size)

		# Display GMSK demodulated bits
		gmsk_data_rx_block, gmsk_data_rx_win = \
		   scopesink.make_scope_sink_f(self, panel, "GMSK rx data",
		      sample_rate)
		vbox.Add(gmsk_data_rx_win, 1, wx.EXPAND)
		self.connect(self.demod.integrate_filter, gmsk_data_rx_block)

		# Sink
		dst = gr.null_sink(gr.sizeof_char)

		# Connect
		self.connect(src, self.mod.head)
		self.connect(self.mod.tail, mixer, ctof, air_noise, chan_filter,
		   self.demod.head)
		self.connect(self.demod.tail, dst)


if __name__ == '__main__':
	app = stdgui.stdapp(gmsk_mod_demod_gui, "gmsk_mod_demod_gui")
	app.MainLoop()

# vim:ts=8
