#!/usr/bin/env python
#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

from gnuradio import gr, gru, blks
from gnuradio import usrp

# from current dir
from pick_bitrate import pick_rx_bitrate

# /////////////////////////////////////////////////////////////////////////////
#                              receive path
# /////////////////////////////////////////////////////////////////////////////

class receive_path(gr.hier_block):
    def __init__(self, fg, subdev_spec, bitrate, decim, spb, rx_callback, log_p):

        gain_mu = 0.03     # default
        threshold = -1     # use system default
        
        self.u = usrp.source_c ()
        adc_rate = self.u.adc_rate()

        (self._bitrate, self._spb, self._decim) = pick_rx_bitrate(bitrate, spb,
                                                                  decim, adc_rate)

        self.u.set_decim_rate(self._decim)
        sw_decim = 1

        if subdev_spec is None:
            subdev_spec = usrp.pick_rx_subdevice(self.u)
        self.subdev = usrp.selected_subdev(self.u, subdev_spec)
        print "Using RX d'board %s" % (self.subdev.side_and_name(),)

        self.u.set_mux(usrp.determine_rx_mux_value(self.u, subdev_spec))

        # Create filter to get actual channel we want
        chan_coeffs = gr.firdes.low_pass (1.0,                  # gain
                                          sw_decim * self._spb, # sampling rate
                                          1.0,                  # midpoint of trans. band
                                          0.1,                  # width of trans. band
                                          gr.firdes.WIN_HANN)   # filter type 

        print "len(rx_chan_coeffs) =", len(chan_coeffs)

        # Decimating Channel filter
        # complex in and out, float taps
        self.chan_filt = gr.fft_filter_ccc(sw_decim, chan_coeffs)
        #self.chan_filt = gr.fir_filter_ccf(sw_decim, chan_coeffs)

        # receiver
        self.packet_receiver = blks.gmsk2_demod_pkts(fg, callback=rx_callback, spb=self._spb,
                                                     gain_mu=gain_mu,
                                                     threshold=threshold)

        fg.connect(self.u, self.chan_filt, self.packet_receiver)
        gr.hier_block.__init__(self, fg, None, None)

        g = self.subdev.gain_range()
        #self.set_gain((g[0] + g[1])/2)        # set gain to midpoint
        self.set_gain(g[1])                    # set gain to max
        self.set_auto_tr(True)                 # enable Auto Transmit/Receive switching

        if log_p:
            self._add_logging(fg)

    def set_freq(self, target_freq):
        """
        Set the center frequency we're interested in.

        @param target_freq: frequency in Hz
        @rypte: bool

        Tuning is a two step process.  First we ask the front-end to
        tune as close to the desired frequency as it can.  Then we use
        the result of that operation and our target_frequency to
        determine the value for the digital up converter.  Finally, we could
        feed any residual_freq to the s/w freq translater.
        """
        r = self.u.tune(0, self.subdev, target_freq)
        if r:
            # Could use residual_freq in s/w freq translator
            return True

        return False

    def set_gain(self, gain):
        if gain is None:
            r = self.subdev.gain_range()
            gain = (r[0] + r[1])/2               # set gain to midpoint
        self.gain = gain
        return self.subdev.set_gain(gain)

    def set_auto_tr(self, enable):
        return self.subdev.set_auto_tr(enable)
        
    def bitrate(self):
        return self._bitrate

    def spb(self):
        return self._spb

    def decim(self):
        return self._decim

    def carrier_sensed(self):
        """
        Return True if we think carrier is present
        """
        return self.packet_receiver.carrier_sensed()
    
    def _add_logging(self, fg):
        fg.connect(self.packet_receiver.gmsk_demod.freq_offset,
                   gr.file_sink(gr.sizeof_float, "freq_offset.dat"))

