/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <usrp1_sink_base.h>
#include <gr_io_signature.h>
#include <usrp_standard.h>
#include <assert.h>

static const int OUTPUT_MULTIPLE_BYTES = 4 * 1024;

usrp1_sink_base::usrp1_sink_base (const std::string &name,
				  gr_io_signature_sptr input_signature,
				  int which_board,
				  unsigned int interp_rate
				  ) throw (std::runtime_error)
  : gr_sync_block (name,
		   input_signature,
		   gr_make_io_signature (0, 0, 0)),
    d_nunderruns (0)
{
  d_usrp = usrp_standard_tx::make (which_board,  interp_rate);
  if (d_usrp == 0)
    throw std::runtime_error ("can't open usrp1");

  // All calls to d_usrp->write must be multiples of 512 bytes.
  // We jack this up to 4k to reduce overhead.

  set_output_multiple (OUTPUT_MULTIPLE_BYTES / input_signature->sizeof_stream_item (0));
}

usrp1_sink_base::~usrp1_sink_base ()
{
  delete d_usrp;
}

int
usrp1_sink_base::work (int noutput_items,
		       gr_vector_const_void_star &input_items,
		       gr_vector_void_star &output_items)
{
  static const int BUFSIZE = 4 * OUTPUT_MULTIPLE_BYTES;
  unsigned char outbuf[BUFSIZE];
  int		obi = 0;
  int		input_index = 0;
  int		input_items_consumed;
  int		bytes_written;
  bool		underrun;
  

  while (input_index < noutput_items){
  
    copy_to_usrp_buffer (input_items,
			 input_index,
			 noutput_items - input_index,	// input_items_available
			 input_items_consumed,		// [out]
			 &outbuf[obi],			// [out] usrp_buffer
			 BUFSIZE - obi,			// usrp_buffer_length
			 bytes_written);		// [out]

    assert (input_index + input_items_consumed <= noutput_items);
    assert (obi + bytes_written <= BUFSIZE);
    
    if (obi >= BUFSIZE){	// flush
      if (d_usrp->write (outbuf, obi, &underrun) != obi)
	return -1;		// indicate we're done

      if (underrun){
	d_nunderruns++;
	fprintf (stderr, "usrp1_sink: underrun\n");
      }
      obi = 0;
    }

    input_index += input_items_consumed;
    obi += bytes_written;
  }

  if (obi != 0){
    assert (obi % 512 == 0);
    if (d_usrp->write (outbuf, obi, &underrun) != obi)
      return -1;	// indicate we're done

    if (underrun){
      d_nunderruns++;
      fprintf (stderr, "usrp1_sink: underrun");
    }
  }

  return noutput_items;
}

bool
usrp1_sink_base::set_interp_rate (unsigned int rate)
{
  return d_usrp->set_interp_rate (rate);
}

bool
usrp1_sink_base::set_tx_freq (int channel, double freq)
{
  return d_usrp->set_tx_freq (channel, freq);
}

long
usrp1_sink_base::dac_freq () const
{
  return d_usrp->dac_freq ();
}

unsigned int
usrp1_sink_base::interp_rate () const
{
  return d_usrp->interp_rate ();
}

double
usrp1_sink_base::tx_freq (int channel) const
{
  return d_usrp->tx_freq (channel);
}

void
usrp1_sink_base::set_verbose (bool verbose)
{  
  d_usrp->set_verbose (verbose);
}

bool
usrp1_sink_base::write_aux_dac (int slot, int which_dac, int value)
{
  return d_usrp->write_aux_dac (slot, which_dac, value);
}

int
usrp1_sink_base::read_aux_adc (int slot, int which_adc)
{
  return d_usrp->read_aux_adc (slot, which_adc);
}

bool
usrp1_sink_base::write_eeprom (int i2c_addr, int eeprom_offset, const std::string buf)
{
  return d_usrp->write_eeprom (i2c_addr, eeprom_offset, buf);
}

std::string
usrp1_sink_base::read_eeprom (int i2c_addr, int eeprom_offset, int len)
{
  return d_usrp->read_eeprom (i2c_addr, eeprom_offset, len);
}

// internal routines...

bool
usrp1_sink_base::_write_fpga_reg (int regno, int value)
{
  return d_usrp->_write_fpga_reg (regno, value);
}

int
usrp1_sink_base::_read_fpga_reg (int regno)
{
  return d_usrp->_read_fpga_reg (regno);
}
