/*  Motti -- a strategy game
    Copyright (C) 1999 Free Software Foundation

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
#include <config.h>

#ifdef HAVE_LIBX11

/* This file has general X init routines. The O'Reilly's Xlib
   Programming manual by Adrian Nye has been used as a guide.  */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/xpm.h>

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STRING_H
# include <string.h>
#else
# include <strings.h>
#endif

#ifdef HAVE_GETHOSTNAME
# include <unistd.h>
#endif

#include "xwin.h"
#include "map.h"
#include "xinit.h"
#include "wrappers.h"

Display *display;
int screen_num;
Colormap colormap;
int depth;

struct db_vals db_val_tab = {0};
XrmDatabase DB = {0};

XColor *player_col;
XColor *occupied_col;
XColor misc_col[last_col_num];

Window main_win;
Window but_win[NBUTS];
Window turn_win;
Window map_win;

GC mapgc, capitalgc, crossgc, turngc;
Pixmap att_pix[6], def_pix, gue_pix;
Pixmap d_att_pix, d_def_pix, d_gue_pix;

Coord turnwin_num_base[CROSS_MAX+2] = {0};

static Visual *visual;
static GC tempgc;

static enum mouse_action parse_mouse_string (const char *);
static void init_DB_tab (void);
static int alloc_Xrm_color (const char *, const char *, const char *,
			    XColor *);
static void make_map_stipples (Pixmap, Pixmap, int, int);
static void make_att_buttons (void);
static void make_att_pixs (void);
static void make_def_pixs (void);
static void make_gue_pixs (void);
static void calculate_extents (const Font);
static void alloc_gcs (void);
static void set_hints (const int, const int, const int, char **, int);

static enum mouse_action
parse_mouse_string (act_name)
     const char *act_name;
{
  register enum mouse_action i;
  const char *all_acts[] = {
    "mark", "select", "act", "last_place", "attack", "defend",
    "guerilla", "reset", "none"
  };
  for (i = mark; i <= no_act; i++)
    if (!strcasecmp (all_acts[i], act_name))
      return i;
}

static void
init_DB_tab ()
{
  register int i;

  enum types 
  {
    type_short, type_string, type_mouse, type_geometry, end
  };
  struct val_tab
  {
    const enum types type;
    void *ptr;
    const char *name, *class;
    union
    {
      const int val;
      const char *str;
      const enum mouse_action mouse;
    } def;
  };
  struct val_tab vals[] =
  {
    {type_short, &db_val_tab.but_width, "motti.button.width",
     "Motti.Button.Width", 80},
    {type_short, &db_val_tab.but_height, "motti.button.height",
     "Motti.Button.Height", 20},
    {type_geometry, &db_val_tab.geom_str, "motti.geometry",
     "Motti.Geometry", 0},
    {type_short, &db_val_tab.turnwin_width, "motti.turnwindow.width",
     "Motti.Turnwindow.Width", 40},
    {type_short, &db_val_tab.turnwin_height,
     "motti.turnwindow.height", "Motti.Turnwindow.Height", 20},
    {type_string, &db_val_tab.cross_left_font,
     "motti.turnwindow.font", "Motti.Font", 0},
    {type_short, &db_val_tab.map_square_size, "motti.map.squaresize",
     "Motti.Map.Squaresize", 20},
    {type_short, &db_val_tab.x_margin, "motti.x.margin",
     "Motti.Margin", 10},
    {type_short, &db_val_tab.y_margin, "motti.y.margin",
     "Motti.Margin", 10},
    {type_short, &db_val_tab.sel_width, "motti.selected.width",
     "Motti.Width", 3},
    {type_mouse, &db_val_tab.mouse[0], "motti.mousebutton1",
     "Motti.Mousebutton1", mark},
    {type_mouse, &db_val_tab.mouse[1], "motti.mousebutton2",
     "Motti.Mousebutton2", sel},
    {type_mouse, &db_val_tab.mouse[2], "motti.mousebutton3",
     "Motti.Mousebutton3", act},
    {type_mouse, &db_val_tab.mouse[3], "motti.mousebutton4",
     "Motti.Mousebutton4", no_act},
    {type_mouse, &db_val_tab.mouse[4], "motti.mousebutton5",
     "Motti.Mousebutton5", no_act},
    {end, NULL, "", "", 0}
  };
  struct val_tab *val_ptr = vals;
  vals[5].def.str = "-adobe-helvetica-medium-r-*-*-20-*-*-*-*-*-*-*";

  while (val_ptr->type != end)
    {
      char *str_type;
      XrmValue value;
      if (XrmGetResource (DB, val_ptr->name, val_ptr->class, &str_type,
			  &value))
	{
	  switch (val_ptr->type)
	    {
	      char *str;
	    case type_short:
	      *((short *) val_ptr->ptr) = (short) atoi (value.addr);
	      break;
	    case type_string:
	      str = (char *) my_malloc ((strlen (value.addr)+1) *
					 sizeof (char));
	      strcpy (str, value.addr);
	      *((char **) val_ptr->ptr) = str;
	      break;
	    case type_mouse:
	      *((enum mouse_action *) val_ptr->ptr) =
		parse_mouse_string (value.addr);
	      if (*((enum mouse_action *) val_ptr->ptr) > no_act)
		die (alloc_err_str (ERR_MOUSE, value.addr));
	      break;
	    }
	}
      else
	{
	  switch (val_ptr->type)
	    {
	    case type_short:
	      *((short *) val_ptr->ptr) = (short) val_ptr->def.val;
	      break;
	    case type_string:
	    case type_geometry:
	      *((const char **) val_ptr->ptr) = val_ptr->def.str;
	      break;
	    case type_mouse:
	      *((enum mouse_action *) val_ptr->ptr) =
		val_ptr->def.mouse;
	      break;
	    }
	  if (val_ptr->type != type_geometry)
	    fprintf (stderr, "motti: using hard-coded default resource value for %s\n", val_ptr->name);
	}
      val_ptr++;
    }
}

/* This function searches for possible display name in any files.  */
extern int
open_display (display_name)
     char *display_name;
{
  XrmValue value;
  XrmDatabase displayDB = {0};
  char *str_type;
  XrmInitialize ();

  if (display_name)
    display = XOpenDisplay (display_name);
  else
    {
      XrmCombineFileDatabase ("~/.Xdefaults", &displayDB, False);
      XrmCombineFileDatabase (APPDEF, &displayDB, False);
      if (XrmGetResource (displayDB, "motti.display",
			  "Motti.Display", &str_type, &value))
	display = XOpenDisplay (value.addr);
      else
	display = XOpenDisplay ((char *) NULL);
      XrmDestroyDatabase (displayDB);
    }
  if (!display)
    return 0;
  /* These are defined now for later use.  */
  screen_num = DefaultScreen (display);
  visual = DefaultVisual (display, screen_num);
  colormap = DefaultColormap (display, screen_num);
  depth = DefaultDepth (display, screen_num);
  return 1;
}

extern void
create_DB ()
{
  char filename[1024] = APPDEF;
  char *environment;

  /* First get the application defaults.  */
  DB = XrmGetFileDatabase (filename);

  /* Next get the server defaults.  */
  if (XResourceManagerString (display) != NULL)
    {
      XrmDatabase serverDB;
      serverDB = XrmGetStringDatabase (XResourceManagerString
				       (display));
      XrmMergeDatabases(serverDB, &DB);
    }
  else
    {
      XrmCombineFileDatabase ("~/.Xdefaults", &DB, True);
    }

  environment = getenv ("XENVIRONMENT");
#if HAVE_GETHOSTNAME
  if (!environment)
    {
      environment = filename;
      strcpy (filename, "~/.Xdefaults-");
      gethostname (filename+13, 1011);
    }
#else
  if (environment)
#endif
    XrmCombineFileDatabase (environment, &DB, True);
  init_DB_tab ();
}

static int
alloc_Xrm_color (instance, class, deflt, color)
     const char *instance, *class;
     const char *deflt;
     XColor *color;
{
  char *msg = NULL;
  const char *color_name;
  XrmValue value;
  XColor nullcol;
  char *str_type;

  if (XrmGetResource (DB, instance, class, &str_type, &value))
    color_name = value.addr;
  else
    {
      if (deflt)
	{
	  color_name = deflt;
	  fprintf (stderr, "motti: using hard-coded default color \"%s\" for %s\n",
		   deflt, instance);
	}
      else
	msg = alloc_err_str (ERR_COLOR_FIND, class);
    }

  if (!msg && !XAllocNamedColor (display, colormap, color_name, color,
				 &nullcol))
    msg = alloc_err_str (ERR_COLOR_ALLOC, color_name);
  else
    return 1;
  fprintf (stderr, "motti: %s\n", msg);
  free (msg);
  return 0;
}

extern int
alloc_colors ()
{
  char *str_type;
  XrmValue value;
  int i;
  XVisualInfo visual_info;

  /* Search for the best visual, use color if possible */
  for (i = 5; !XMatchVisualInfo (display, screen_num, depth, i,
				 &visual_info); i--);
  if (i < StaticColor)
    return 0;

  player_col = (XColor *) my_malloc (sizeof(XColor)*game_map.players);
  occupied_col = (XColor *) my_malloc (sizeof(XColor)*game_map.players);

  if (!(alloc_Xrm_color ("motti.background", "Motti.Background",
			"white", &misc_col[bg_col])
	&& alloc_Xrm_color ("motti.foreground", "Motti.Foreground",
			    "black", &misc_col[fg_col])
	&& alloc_Xrm_color ("motti.color.map.background",
			    "Motti.Map.Background", "RGBi:0/0/0.05",
			    &misc_col[map_bg_col])
	&& alloc_Xrm_color ("motti.color.capital",
			    "Motti.Color.Capital", "gray50",
			    &misc_col[capital_col])
	&& alloc_Xrm_color ("motti.color.cross",
			    "Motti.Color.Cross", "gray70",
			    &misc_col[cross_col])
	&& alloc_Xrm_color ("motti.button.foreground",
			    "Motti.Foreground", "black",
			    &misc_col[but_fg])
	&& alloc_Xrm_color ("motti.button.background",
			    "Motti.Background", "white",
			    &misc_col[but_bg])
	&& alloc_Xrm_color ("motti.color.cross_left",
			    "Motti.Foreground", "black",
			    &misc_col[cross_left])
	&& alloc_Xrm_color ("motti.color.turnwin_border",
			    "Motti.Background", "black",
			    &misc_col[turn_win_border])))
    return 0;

#define MAXDEFCOLPLAYER 4
  for (i = 1; i <= game_map.players; i++)
    {
      const char *def_col[] =
      {
	"DarkRed", "Red",
	"DarkBlue", "Blue",
	"DarkGreen", "Green",
	"DarkGoldenrod4", "DarkGoldenrod1",
	""
      };
      char instancename[25];
      sprintf (instancename, "motti.player%i.color", i);
      if (!alloc_Xrm_color (instancename, "Motti.Player.Color",
			    def_col[i <= MAXDEFCOLPLAYER ? (i-1)*2 :
				   4], &player_col[i-1]))
	return 0;

      sprintf (instancename, "motti.player%iOcc.color", i);
      if (!alloc_Xrm_color (instancename, "Motti.PlayerOcc.Color",
			    def_col[i <= MAXDEFCOLPLAYER ? (i-1)*2+1 :
				   4], &occupied_col[i-1]))
	return 0;
    }
  return 1;
#undef MAXDEFCOLPLAYER
}

static void
make_map_stipples (capital, cross, width, height)
     Pixmap capital, cross;
     int width, height;
{
  XGCValues gcvalues;

  gcvalues.line_width = 1;
  tempgc = XCreateGC (display, capital, GCLineWidth, &gcvalues);

  XSetForeground (display, tempgc, 1);
  XFillRectangle (display, capital, tempgc, 0, 0, width+1, height+1);
  XSetForeground (display, tempgc, 0);
  XFillRectangle (display, capital, tempgc, width/3, height/3,
		  width/2, height/2);

  XFillRectangle (display, cross, tempgc, 0, 0, width+1, height+1);
  XSetForeground (display, tempgc, 1);
  XDrawLine (display, cross, tempgc, width/6, height/6, (5*width)/6,
	     (5*height)/6);
  XDrawLine (display, cross, tempgc, (5*width)/6, height/6, width/6,
	     (5*height)/6);
}

#include "bitmaps/att"
#include "bitmaps/def"
#include "bitmaps/gue"

static void
make_att_buttons ()
{
  int mark_pos, mark_top, mark_height, mark_space, mark_width;
  int pos_fourth; 
  int i = 0;

  mark_top = db_val_tab.but_height / 3;
  mark_height = mark_top;
  mark_space = (db_val_tab.but_width/2 - att_width/2) / 5;
  mark_pos = mark_space;
  mark_width = 4;
  pos_fourth = db_val_tab.but_width/2 + att_width/2 + mark_space;

  XSetFunction (display, tempgc, GXcopy);
  XSetFillStyle (display, tempgc, FillSolid);
  XSetForeground (display, tempgc, misc_col[but_fg].pixel);
  while (1)
    {
      XFillRectangle (display, att_pix[i], tempgc, mark_pos,
		      mark_top, mark_width, mark_height);
      if (i < 5)
	XCopyArea (display, att_pix[i], att_pix[i+1], tempgc, 0, 0,
		   db_val_tab.but_width, db_val_tab.but_height, 0, 0);
      else
	return;
      if (++i == 3)
	mark_pos = pos_fourth;
      else
	mark_pos += mark_space;
    }
}

static void
make_att_pixs ()
{
  Pixmap pix;

  att_pix[0] = XCreatePixmap (display, main_win, db_val_tab.but_width,
			      db_val_tab.but_height, depth);
  att_pix[1] = XCreatePixmap (display, main_win, db_val_tab.but_width,
			      db_val_tab.but_height, depth);
  att_pix[2] = XCreatePixmap (display, main_win, db_val_tab.but_width,
			      db_val_tab.but_height, depth);
  att_pix[3] = XCreatePixmap (display, main_win, db_val_tab.but_width,
			      db_val_tab.but_height, depth);
  att_pix[4] = XCreatePixmap (display, main_win, db_val_tab.but_width,
			      db_val_tab.but_height, depth);
  att_pix[5] = XCreatePixmap (display, main_win, db_val_tab.but_width,
			      db_val_tab.but_height, depth);

  d_att_pix = XCreatePixmap (display, main_win, db_val_tab.but_width,
			     db_val_tab.but_height, depth);

  /* Draw the disabled attack button pixmap.  */
  XFreeGC (display, tempgc);
  tempgc = XCreateGC (display, main_win, 0, NULL);
  XSetFillStyle (display, tempgc, FillSolid);
  XSetForeground (display, tempgc, misc_col[but_fg].pixel);
  XFillRectangle (display, d_att_pix, tempgc, 0, 0,
		  db_val_tab.but_width+1,
		  db_val_tab.but_height+1);
  pix = XCreateBitmapFromData (display, main_win, att_bits, att_width,
			       att_height);
  XSetFillStyle (display, tempgc, FillStippled);
  XSetStipple (display, tempgc, pix);
  XSetTSOrigin (display, tempgc, db_val_tab.but_width/2 - att_width/2,
		db_val_tab.but_height/2 - att_height/2);
  XSetForeground (display, tempgc, misc_col[but_bg].pixel);
  XFillRectangle (display, d_att_pix, tempgc, db_val_tab.but_width/2 -
		  att_width/2, db_val_tab.but_height/2 - att_height/2,
		  att_width, att_height);

  /* Draw the attack button pixmap.  */
  XSetFillStyle (display, tempgc, FillSolid);
  XSetForeground (display, tempgc, misc_col[but_bg].pixel);
  XFillRectangle (display, att_pix[0], tempgc, 0, 0,
		  db_val_tab.but_width+1,
		  db_val_tab.but_height+1);
  XSetFillStyle (display, tempgc, FillStippled);
  XSetForeground (display, tempgc, misc_col[but_fg].pixel);
  XFillRectangle (display, att_pix[0], tempgc, db_val_tab.but_width/2 -
		  att_width/2, db_val_tab.but_height/2 - att_height/2,
		  att_width, att_height);

  XFreePixmap (display, pix);

  /* Draw attack buttons with number of remaining attacks.  */
  make_att_buttons ();
}

static void
make_def_pixs ()
{
  Pixmap pix;

  def_pix = XCreatePixmap (display, main_win, db_val_tab.but_width,
			   db_val_tab.but_height, depth);
  d_def_pix = XCreatePixmap (display, main_win, db_val_tab.but_width,
			     db_val_tab.but_height, depth);

  /* Draw the defend button.  */
  XSetFillStyle (display, tempgc, FillSolid);
  XSetForeground (display, tempgc, misc_col[but_bg].pixel);
  XFillRectangle (display, def_pix, tempgc, 0, 0,
		  db_val_tab.but_width+1,
		  db_val_tab.but_height+1);
  pix = XCreateBitmapFromData (display, main_win, def_bits, def_width,
			       def_height);
  XSetTSOrigin (display, tempgc, db_val_tab.but_width/2 - def_width/2,
		db_val_tab.but_height/2 - def_height/2);
  XSetFillStyle (display, tempgc, FillStippled);
  XSetForeground (display, tempgc, misc_col[but_fg].pixel);
  XSetStipple (display, tempgc, pix);
  XFillRectangle (display, def_pix, tempgc, db_val_tab.but_width/2 -
		  def_width/2, db_val_tab.but_height/2 - def_height/2,
		  def_width, def_height);

  /* Draw the disabled defend button.  */
  XSetFillStyle (display, tempgc, FillSolid);
  XSetForeground (display, tempgc, misc_col[but_fg].pixel);
  XFillRectangle (display, d_def_pix, tempgc, 0, 0,
		  db_val_tab.but_width+1,
		  db_val_tab.but_height+1);
  XSetFillStyle (display, tempgc, FillStippled);
  XSetForeground (display, tempgc, misc_col[but_bg].pixel);
  XFillRectangle (display, d_def_pix, tempgc, db_val_tab.but_width/2 -
		  def_width/2, db_val_tab.but_height/2 - def_height/2,
		  def_width, def_height);

  XFreePixmap (display, pix);
}

static void
make_gue_pixs ()
{
  Pixmap pix;

  gue_pix = XCreatePixmap (display, main_win, db_val_tab.but_width,
			   db_val_tab.but_height, depth);
  d_gue_pix = XCreatePixmap (display, main_win, db_val_tab.but_width,
			     db_val_tab.but_height, depth);

  /* Draw the defend button.  */
  XSetFillStyle (display, tempgc, FillSolid);
  XSetForeground (display, tempgc, misc_col[but_bg].pixel);
  XFillRectangle (display, gue_pix, tempgc, 0, 0,
		  db_val_tab.but_width+1,
		  db_val_tab.but_height+1);
  pix = XCreateBitmapFromData (display, main_win, gue_bits, gue_width,
			       gue_height);
  XSetFillStyle (display, tempgc, FillStippled);
  XSetTSOrigin (display, tempgc, db_val_tab.but_width/2 - gue_width/2,
		db_val_tab.but_height/2 - gue_height/2);
  XSetForeground (display, tempgc, misc_col[but_fg].pixel);
  XSetStipple (display, tempgc, pix);
  XFillRectangle (display, gue_pix, tempgc, db_val_tab.but_width/2 -
		  gue_width/2, db_val_tab.but_height/2 - gue_height/2,
		  gue_width, gue_height);

  /* Draw the disabled defend button.  */
  XSetFillStyle (display, tempgc, FillSolid);
  XSetForeground (display, tempgc, misc_col[but_fg].pixel);
  XFillRectangle (display, d_gue_pix, tempgc, 0, 0,
		  db_val_tab.but_width+1,
		  db_val_tab.but_height+1);
  XSetFillStyle (display, tempgc, FillStippled);
  XSetForeground (display, tempgc, misc_col[but_bg].pixel);
  XFillRectangle (display, d_gue_pix, tempgc, db_val_tab.but_width/2 -
		  gue_width/2, db_val_tab.but_height/2 - gue_height/2,
		  gue_width, gue_height);

  XFreePixmap (display, pix);
}

static void
calculate_extents (font)
     const Font font;
{
  register int i;
  for (i = 0; i <= CROSS_MAX+1; i++)
    {
      int null;
      const char *nums[] = {"6", "5", "4", "3", "2", "1", "0", "*"};
      XCharStruct charstruct;
      XQueryTextExtents (display, font, nums[i], 1, &null, &null,
			 &null, &charstruct);
      turnwin_num_base[i].x = db_val_tab.turnwin_width/2 -
	(charstruct.width-charstruct.lbearing)/2;
      turnwin_num_base[i].y = db_val_tab.turnwin_height/2 -
	(charstruct.ascent+charstruct.descent)/2 + charstruct.ascent;
    }      
}

static void
alloc_gcs ()
{
  XGCValues gcval = {0};
  Pixmap capital_stipple, cross_stipple;

  capital_stipple = XCreatePixmap (display, main_win,
				   db_val_tab.map_square_size,
				   db_val_tab.map_square_size, 1);
  cross_stipple = XCreatePixmap (display, main_win,
				 db_val_tab.map_square_size,
				 db_val_tab.map_square_size, 1);

  make_map_stipples (capital_stipple, cross_stipple,
		     db_val_tab.map_square_size,
		     db_val_tab.map_square_size);

  /* These three are set to draw capitals. */
  gcval.foreground = misc_col[capital_col].pixel;
  gcval.fill_style = FillStippled;
  gcval.stipple = capital_stipple;
  capitalgc = XCreateGC (display, main_win, GCForeground | GCFillStyle
			 | GCStipple, &gcval);
  gcval.foreground = misc_col[cross_col].pixel;
  gcval.stipple = cross_stipple;
  crossgc = XCreateGC (display, main_win, GCForeground | GCFillStyle |
		       GCStipple, &gcval);


  /* No fields need to be set, since foreground color is changed every
   * time */
  /* Note to myself: change this to make BW window. */
  mapgc = XCreateGC (display, main_win, 0, &gcval);

  /* This gc is used to draw the amount of crosses left to the turn
     window.  */
  gcval.foreground = misc_col[cross_left].pixel;
  gcval.background = occupied_col[game_map.turn-1].pixel;
  gcval.font = XLoadFont (display, db_val_tab.cross_left_font);

  if (!gcval.font)
    die (alloc_err_str (ERR_NOFONT, db_val_tab.cross_left_font));
  calculate_extents (gcval.font);
  turngc = XCreateGC (display, main_win, GCBackground | GCForeground |
		      GCFont, &gcval);
}

int but_width[3];
int but_height;

static void
set_hints (min_width, min_height, geom_specified, argv, argc)
     const int min_width, min_height, geom_specified;
     char **argv;
     int argc;
{
  XWMHints *wm_hints;
  XClassHint *class_hints;
  XTextProperty window_name_prop, icon_name_prop;
  /* Add version number.  */
  char *window_name = VERSION_STR;
  char *icon_name = "motti";
  XSizeHints *size_hints;

  if (!XStringListToTextProperty (&window_name, 1, &window_name_prop)
      || !XStringListToTextProperty (&icon_name, 1, &icon_name_prop))
    die (alloc_err_str(ERR_MALLOC, (char *) NULL));

  /* Allocate needed structures for properties */
  size_hints = XAllocSizeHints ();
  wm_hints = XAllocWMHints ();
  class_hints = XAllocClassHint ();

  if (geom_specified)
    size_hints->flags = PPosition | PSize | PMinSize;
  else
    size_hints->flags = PSize | PMinSize;
  size_hints->min_width = min_width;
  size_hints->min_height = min_height;

  wm_hints->initial_state = NormalState;
  wm_hints->input = False;

  /* TODO: IIRC these shouldn't be hardcoded.  */
  class_hints->res_name = "motti";
  class_hints->res_class = "motti";

  XSetWMProperties (display, main_win, &window_name_prop,
		    &icon_name_prop, argv, argc, size_hints, wm_hints,
		    class_hints);

  /* Free all allocated structures */
  XFree (size_hints);
  XFree (wm_hints);
  XFree (class_hints);
  XFree (window_name_prop.value);
  XFree (icon_name_prop.value);
}

/* TODO: Change this to make BW support. */
extern void
open_windows (argc, argv)
     int argc;
     char **argv;
{
  XSetWindowAttributes win_attribs;
  int x = 0, y = 0, width = 1, height = 1, geom_specified = 1;
  char *str_type;
  XrmValue value;

  if (db_val_tab.geom_str)
    XParseGeometry (db_val_tab.geom_str, &x, &y, &width, &height);
  else
    geom_specified = 0;

  win_attribs.background_pixel = misc_col[bg_col].pixel;
  main_win = XCreateWindow (display, RootWindow(display, screen_num),
			    x, y, width, height, 0, CopyFromParent,
			    InputOutput, CopyFromParent, CWBackPixel,
			    &win_attribs);

  alloc_gcs ();
  make_att_pixs ();
  make_att_buttons ();
  make_def_pixs ();
  make_gue_pixs ();
  XFreeGC (display, tempgc);

  win_attribs.border_pixel = misc_col[but_fg].pixel;
  but_win[ATT] = XCreateWindow (display, main_win,
				db_val_tab.x_margin,
				db_val_tab.y_margin,
				db_val_tab.but_width,
				db_val_tab.but_height, 1,
				CopyFromParent, InputOutput,
				CopyFromParent, CWBorderPixel,
				&win_attribs);

  but_win[DEF] = XCreateWindow (display, main_win,
				db_val_tab.but_width +
				db_val_tab.x_margin*2,
				db_val_tab.y_margin,
				db_val_tab.but_width,
				db_val_tab.but_height, 1,
				CopyFromParent, InputOutput,
				CopyFromParent, CWBorderPixel,
				&win_attribs);

  but_win[GUE] = XCreateWindow (display, main_win,
				db_val_tab.but_width*2 +
				db_val_tab.x_margin*3,
				db_val_tab.y_margin,
				db_val_tab.but_width,
				db_val_tab.but_height, 1,
				CopyFromParent, InputOutput,
				CopyFromParent, CWBorderPixel,
				&win_attribs);

  turn_win = XCreateSimpleWindow (display, main_win,
				  db_val_tab.but_width*3 +
				  db_val_tab.x_margin*4,
				  db_val_tab.y_margin,
				  db_val_tab.turnwin_width,
				  db_val_tab.turnwin_height, 1,
				  misc_col[turn_win_border].pixel,
				  player_col[game_map.turn-1].pixel);

  x = db_val_tab.x_margin;
  /* There must be room for action buttons above the map.  */
  y = db_val_tab.y_margin*2 + (db_val_tab.but_height >
			       db_val_tab.turnwin_height
			       ? db_val_tab.but_height :
			       db_val_tab.turnwin_height);
  /* TODO: Scrollable and resizeable map window.  Propably too much
     trouble anyway, at least with bare Xlib!  */
  win_attribs.border_pixel = misc_col[fg_col].pixel;
  win_attribs.background_pixel = misc_col[map_bg_col].pixel;
  map_win = XCreateWindow (display, main_win, x, y, game_map.width *
			   db_val_tab.map_square_size, game_map.height *
			   db_val_tab.map_square_size, 2, CopyFromParent,
			   InputOutput, CopyFromParent, CWBorderPixel
			   | CWBackPixel, &win_attribs);

  set_hints (x + game_map.width * db_val_tab.map_square_size + 2 *
	     db_val_tab.x_margin, y + game_map.height *
	     db_val_tab.map_square_size + db_val_tab.y_margin,
	     geom_specified, argv, argc);
}

#include "bitmaps/map_pointer_active"
#include "bitmaps/map_pointer_disabled"
#include "bitmaps/map_pointer_mask"
Cursor active_cursor, disabled_cursor;

/* TODO: Make this work.  */
extern void
create_cursors ()
{
  Pixmap mask, active, disabled;
  XColor white, black;
  Colormap default_colormap;

  default_colormap = DefaultColormap (display, screen_num);
  white.pixel = WhitePixel (display, screen_num);
  black.pixel = BlackPixel (display, screen_num);
  XQueryColor (display, default_colormap, &white);
  XQueryColor (display, default_colormap, &black);

  mask = XCreateBitmapFromData (display, main_win,
				map_pointer_mask_bits,
				map_pointer_mask_width,
				map_pointer_mask_height);
  active = XCreateBitmapFromData (display, main_win,
				  map_pointer_active_bits,
				  map_pointer_active_width,
				  map_pointer_active_height);
  disabled = XCreateBitmapFromData (display, main_win,
				    map_pointer_disabled_bits,
				    map_pointer_disabled_width,
				    map_pointer_disabled_height);

  active_cursor = XCreatePixmapCursor (display, active, mask, &black,
				       &white,
				       map_pointer_active_x_hot,
				       map_pointer_active_y_hot);
  disabled_cursor = XCreatePixmapCursor (display, disabled, mask,
					 &black, &white,
					 map_pointer_disabled_x_hot,
					 map_pointer_disabled_y_hot);

  XFreePixmap (display, mask);
  XFreePixmap (display, active);
  XFreePixmap (display, disabled);
}

extern void
kill_x ()
{
  XCloseDisplay (display);
}

#endif /* have_libx11 */
