/* PSPP - a program for statistical analysis.
   Copyright (C) 2010, 2011 Free Sonftware Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#ifndef OUTPUT_MESSAGE_ITEM_H
#define OUTPUT_MESSAGE_ITEM_H 1

/* Message items.

   A message item is a subclass of an output item (see
   output/output-item.h).

   A message item is an error, warning, or note to the user.

   Message items should not be submitted directly to the output subsystem.
   Instead, use the msg() function in libpspp/message.h, which will ensure that
   the message gets routed properly for the PSPP user interface in use. */

#include <stdbool.h>
#include "output/output-item.h"

/* A message item. */
struct message_item
  {
    struct output_item output_item;
    struct msg *msg;
  };

struct message_item *message_item_create (const struct msg *);

const struct msg *message_item_get_msg (const struct message_item *);

/* This boilerplate for message_item, a subclass of output_item, was
   autogenerated by mk-class-boilerplate. */

#include <assert.h>
#include "libpspp/cast.h"

extern const struct output_item_class message_item_class;

/* Returns true if SUPER is a message_item, otherwise false. */
static inline bool
is_message_item (const struct output_item *super)
{
  return super->class == &message_item_class;
}

/* Returns SUPER converted to message_item.  SUPER must be a message_item, as
   reported by is_message_item. */
static inline struct message_item *
to_message_item (const struct output_item *super)
{
  assert (is_message_item (super));
  return UP_CAST (super, struct message_item, output_item);
}

/* Returns INSTANCE converted to output_item. */
static inline struct output_item *
message_item_super (const struct message_item *instance)
{
  return CONST_CAST (struct output_item *, &instance->output_item);
}

/* Increments INSTANCE's reference count and returns INSTANCE. */
static inline struct message_item *
message_item_ref (const struct message_item *instance)
{
  return to_message_item (output_item_ref (&instance->output_item));
}

/* Decrements INSTANCE's reference count, then destroys INSTANCE if
   the reference count is now zero. */
static inline void
message_item_unref (struct message_item *instance)
{
  output_item_unref (&instance->output_item);
}

/* Returns true if INSTANCE's reference count is greater than 1,
   false otherwise. */
static inline bool
message_item_is_shared (const struct message_item *instance)
{
  return output_item_is_shared (&instance->output_item);
}

void message_item_submit (struct message_item *);

#endif /* output/message-item.h */
