/* This file is part of GNU RADIUS.
   Copyright (C) 2000,2001, Sergey Poznyakoff
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#define RADIUS_MODULE_EXEC_C
#ifndef lint
static char rcsid[] =
"@(#) $Id: exec.c,v 1.18 2001/11/27 12:13:36 gray Exp $"; 
#endif

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <sys/types.h>
#include <sys/time.h>
#include <sys/file.h>

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <syslog.h>
#include <radiusd.h>
#include <obstack1.h>

/*
 *	Execute a program on successful authentication.
 *	Return 0 if exec_wait == 0.
 *	Return the exit code of the called program if exec_wait != 0.
 *
 */
int
radius_exec_program(cmd, req, reply, exec_wait, user_msg)
	char *cmd;
	RADIUS_REQ *req;
	VALUE_PAIR **reply;
	int exec_wait;
	char **user_msg;
{
	int p[2];
	RETSIGTYPE (*oldsig)();
	pid_t pid;
	int n;
	char *ptr, *errp;
	int status;
	VALUE_PAIR *vp;
	FILE *fp;
	int line_num;
	char buffer[RAD_BUFFER_SIZE];
	struct passwd *pwd;
#if 0
	int saved_uid, saved_gid;
#endif
	
	if (cmd[0] != '/') {
		radlog(L_ERR,
   _("radius_exec_program(): won't execute, not an absolute pathname: %s"),
		       cmd);
		return -1;
	}

	/* Check user/group
	 * FIXME: This should be checked *once* after re-reading the
	 *        configuration
	 */
	pwd = getpwnam(config.exec_user);
	if (!pwd) {
		radlog(L_ERR,
    _("radius_exec_program(): won't execute, no such user: %s"),
		       config.exec_user);
		return -1;
	}
	
	if (exec_wait) {
		if (pipe(p) != 0) {
			radlog(L_ERR|L_PERROR, _("couldn't open pipe"));
			p[0] = p[1] = 0;
			return -1;
		}

		if ((oldsig = signal(SIGCHLD, SIG_DFL)) == SIG_ERR) {
			radlog(L_ERR|L_PERROR, _("can't reset SIGCHLD"));
			return -1;
		}
	}

	if ((pid = fork()) == 0) {
		int argc;
		char **argv;
		struct obstack s;

		obstack_init(&s);
		
		/* child branch */
		ptr = radius_xlate(&s, cmd, req, reply ? *reply : NULL);

		debug(1,
			("command line: %s", ptr));

		argcv_get(ptr, "", &argc, &argv);
		
		if (exec_wait) {
			if (close(p[0]))
				radlog(L_ERR|L_PERROR, _("can't close pipe"));
			if (dup2(p[1], 1) != 1)
				radlog(L_ERR|L_PERROR, _("can't dup stdout"));
		}

		for(n = getmaxfd(); n >= 3; n--)
			close(n);

		chdir("/tmp");
		
		#if 0
		saved_uid = geteuid();
		saved_gid = getegid();
		#endif
		
		if (pwd->pw_gid != 0 && setgid(pwd->pw_gid)) {
			radlog(L_ERR|L_PERROR,
			       _("setgid(%d) failed"), pwd->pw_gid);
		}
		if (pwd->pw_uid != 0) {
#if defined(HAVE_SETEUID)
			if (seteuid(pwd->pw_uid)) 
				radlog(L_ERR|L_PERROR,
				       _("seteuid(%d) failed (ruid=%d, euid=%d)"),
					 pwd->pw_uid, getuid(), geteuid());
#elif defined(HAVE_SETREUID)
			if (setreuid(0, pwd->pw_uid)) 
				radlog(L_ERR|L_PERROR,
				       _("setreuid(0,%d) failed (ruid=%d, euid=%d)"),
					 pwd->pw_uid, getuid(), geteuid());
#else
# warning "*** NO WAY TO SET EFFECTIVE UID IN radius_exec_program() ***"
#endif
		}
		execvp(argv[0], argv);

		/*
		 * Report error via syslog: we might not be able
		 * to restore initial privileges if we were started
		 * as non-root.
		 */
		openlog("radiusd", LOG_PID, LOG_USER);
		syslog(LOG_ERR, "can't run %s (ruid=%d, euid=%d): %m",
		       argv[0], getuid(), geteuid());
		exit(2);
	}

	/* Parent branch */ 
	if (pid < 0) {
		radlog(L_ERR|L_PERROR, _("can't fork"));
		return -1;
	}
	if (!exec_wait)
		return 0;

	if (close(p[1]))
		radlog(L_ERR|L_PERROR, _("can't close pipe"));

	fp = fdopen(p[0], "r");

	vp = NULL;
	line_num = 0;
	while (ptr = fgets(buffer, sizeof(buffer), fp)) {
		line_num++;
		debug(1,
			("got `%s'", buffer));
		if (userparse(ptr, &vp, &errp)) {
			radlog(L_ERR,
			    _("<stdout of %s>:%d: %s"),
			    cmd, line_num, errp);
			avl_free(vp);
			vp = NULL;
		}
	}

	fclose(fp);
	/*close(p[0]);*/

	if (vp) {
		avl_merge(reply, &vp);
	}

	
	while (waitpid(pid, &status, 0) != pid)
		;

	if (signal(SIGCHLD, oldsig) == SIG_ERR)
		radlog(L_CRIT|L_PERROR,
			_("can't restore SIGCHLD"));
	sig_cleanup(SIGCHLD);

	if (WIFEXITED(status)) {
		status = WEXITSTATUS(status);
		debug(1, ("returned: %d", status));
		if (status == 2) {
			radlog(L_ERR,
			       _("can't run external program (reason reported via syslog channel user.err)"));
		}
		return status;
	}
	radlog(L_ERR, _("radius_exec_program(): abnormal child exit"));

	return 1;
}
