/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

package tech.libeufin.nexus.cli

import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.parameters.arguments.argument
import com.github.ajalt.clikt.parameters.arguments.multiple
import com.github.ajalt.clikt.parameters.arguments.unique
import com.github.ajalt.clikt.parameters.groups.provideDelegate
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.parameters.types.enum
import kotlinx.coroutines.*
import tech.libeufin.common.*
import tech.libeufin.nexus.*
import tech.libeufin.nexus.db.Database
import tech.libeufin.nexus.db.PaymentDAO.IncomingRegistrationResult
import tech.libeufin.nexus.ebics.*
import java.io.IOException
import java.io.InputStream
import java.time.Duration
import java.time.Instant
import kotlin.time.toKotlinDuration

/** Ingests an outgoing [payment] into [db] */
suspend fun ingestOutgoingPayment(
    db: Database,
    payment: OutgoingPayment
) {
    val metadata: Pair<ShortHashCode, ExchangeUrl>? = payment.wireTransferSubject?.let { 
        runCatching { parseOutgoingTxMetadata(it) }.getOrNull()
    }
    val result = db.payment.registerOutgoing(payment, metadata?.first, metadata?.second)
    if (result.new) {
        if (result.initiated)
            logger.info("$payment")
        else 
            logger.warn("$payment recovered")
    } else {
        logger.debug("{} already seen", payment)
    }
}

/** 
 * Ingest an incoming [payment] into [db]
 * Stores the payment into valid talerable ones or bounces it, according to [accountType] .
 */
suspend fun ingestIncomingPayment(
    db: Database,
    payment: IncomingPayment,
    accountType: AccountType
) {
    suspend fun bounce(msg: String) {
        if (payment.bankId == null) {
            logger.debug("{} ignored: missing bank ID", payment)
            return;
        }
        when (accountType) {
            AccountType.exchange -> {
                val result = db.payment.registerMalformedIncoming(
                    payment,
                    payment.amount, 
                    Instant.now()
                )
                if (result.new) {
                    logger.info("$payment bounced in '${result.bounceId}': $msg")
                } else {
                    logger.debug("{} already seen and bounced in '{}': {}", payment, result.bounceId, msg)
                }
            }
            AccountType.normal -> {
                val res = db.payment.registerIncoming(payment)
                if (res.new) {
                    logger.info("$payment")
                } else {
                    logger.debug("{} already seen", payment)
                }
            }
        }
    }
    runCatching { parseIncomingTxMetadata(payment.wireTransferSubject) }.fold(
        onSuccess = { metadata -> 
            when (val res = db.payment.registerTalerableIncoming(payment, metadata)) {
                IncomingRegistrationResult.ReservePubReuse -> bounce("reverse pub reuse")
                is IncomingRegistrationResult.Success -> {
                    if (res.new) {
                        logger.info("$payment")
                    } else {
                        logger.debug("{} already seen", payment)
                    }
                }
            }
        },
        onFailure = { e -> bounce(e.fmt())}
    )
}

/** Ingest an EBICS [payload] of [doc] into [db] */
private suspend fun ingestPayload(
    db: Database,
    cfg: NexusEbicsConfig,
    payload: InputStream,
    doc: OrderDoc
) {
    /** Ingest a single EBICS [xml] [document] into [db] */
    suspend fun ingest(xml: InputStream) {
        when (doc) {
            OrderDoc.report, OrderDoc.statement, OrderDoc.notification -> {
                try {
                    parseTx(xml, cfg.currency, cfg.dialect).forEach {
                        if (cfg.fetch.ignoreBefore != null && it.executionTime < cfg.fetch.ignoreBefore) {
                            logger.debug("IGNORE {}", it)
                        } else {
                            when (it) {
                                is IncomingPayment -> ingestIncomingPayment(db, it, cfg.accountType)
                                is OutgoingPayment -> ingestOutgoingPayment(db, it)
                                is TxNotification.Reversal -> {
                                    logger.error("BOUNCE '${it.msgId}': ${it.reason}")
                                    db.initiated.reversal(it.msgId, "Payment bounced: ${it.reason}")
                                }
                            }
                        }
                    }
                } catch (e: Exception) {
                    throw Exception("Ingesting notifications failed", e)
                }
            }
            OrderDoc.acknowledgement -> {
                val acks = parseCustomerAck(xml)
                for (ack in acks) {
                    when (ack.actionType) {
                        HacAction.ORDER_HAC_FINAL_POS -> {
                            logger.debug("{}", ack)
                            db.initiated.logSuccess(ack.orderId!!)?.let { requestUID ->
                                logger.info("Payment '$requestUID' accepted at ${ack.timestamp.fmtDateTime()}")
                            }
                        }
                        HacAction.ORDER_HAC_FINAL_NEG -> {
                            logger.debug("{}", ack)
                            db.initiated.logFailure(ack.orderId!!)?.let { (requestUID, msg) ->
                                logger.error("Payment '$requestUID' refused at ${ack.timestamp.fmtDateTime()}${if (msg != null) ": $msg" else ""}")
                            }
                        }
                        else -> {
                            logger.debug("{}", ack)
                            if (ack.orderId != null) {
                                db.initiated.logMessage(ack.orderId, ack.msg())
                            }
                        }
                    }
                }
            }
            OrderDoc.status -> {
                val status = parseCustomerPaymentStatusReport(xml)
                val msg = status.msg()
                logger.debug("{}", status)
                if (status.paymentCode == ExternalPaymentGroupStatusCode.RJCT) {
                    db.initiated.bankFailure(status.msgId, msg)
                    logger.error("Transaction '${status.msgId}' was rejected : $msg")
                } else {
                    db.initiated.bankMessage(status.msgId, msg)
                }
            }
        }
    }
    
    // Unzip payload if necessary
    when (doc) {
        OrderDoc.status,
        OrderDoc.report,
        OrderDoc.statement, 
        OrderDoc.notification -> {
            try {
                payload.unzipEach { fileName, xml ->
                    logger.trace("parse $fileName")
                    ingest(xml)
                }
            } catch (e: IOException) {
                throw Exception("Could not open any ZIP archive", e)
            }
        }
        OrderDoc.acknowledgement -> ingest(payload)
    }
}

/** 
 * Fetch and ingest banking records from [orders] using EBICS [client] starting from [pinnedStart]
 * 
 * If [pinnedStart] is null fetch new records.
 * 
 * Return true if successful 
 */
private suspend fun fetchEbicsDocuments(
    client: EbicsClient,
    orders: List<EbicsOrder>,
    pinnedStart: Instant?,
): Boolean {
    val lastExecutionTime: Instant? = pinnedStart
    return orders.all { order ->
        val doc = order.doc()
        if (doc == null) {
            logger.debug("Skip unsupported order {}", order)
            true
        } else {
            try {
                if (lastExecutionTime == null) {
                    logger.info("Fetching new '${doc.fullDescription()}'")
                } else {
                    logger.info("Fetching '${doc.fullDescription()}' from timestamp: $lastExecutionTime")
                }
                // downloading the content
                client.download(
                    order,
                    lastExecutionTime,
                    null
                ) { payload ->
                    ingestPayload(client.db, client.cfg, payload, doc)
                }
                true
            } catch (e: Exception) {
                e.fmtLog(logger)
                false
            }
        }
    }
}

class EbicsFetch: CliktCommand("Downloads and parse EBICS files from the bank and ingest them into the database") {
    private val common by CommonOption()
    private val transient by transientOption()
    private val documents: Set<OrderDoc> by argument(
        help = "Which documents should be fetched? If none are specified, all supported documents will be fetched",
        helpTags = OrderDoc.entries.associate { Pair(it.name, it.shortDescription()) },
    ).enum<OrderDoc>().multiple().unique()
    private val pinnedStart by option(
        help = "Only supported in --transient mode, this option lets specify the earliest timestamp of the downloaded documents",
        metavar = "YYYY-MM-DD"
    )
    private val ebicsLog by ebicsLogOption()

    override fun run() = cliCmd(logger, common.log) {
        nexusConfig(common.config).withDb { db, nexusCgf ->
            val cfg = nexusCgf.ebics
            val (clientKeys, bankKeys) = expectFullKeys(cfg)
            val client = EbicsClient(
                cfg,
                httpClient(),
                db,
                EbicsLogger(ebicsLog),
                clientKeys,
                bankKeys
            )
            val docs = if (documents.isEmpty()) OrderDoc.entries else documents.toList()
            val orders = docs.map { cfg.dialect.downloadDoc(it, false) }
            if (transient) {
                logger.info("Transient mode: fetching once and returning.")
                val pinnedStartVal = pinnedStart
                val pinnedStartArg = if (pinnedStartVal != null) {
                    logger.debug("Pinning start date to: {}", pinnedStartVal)
                    dateToInstant(pinnedStartVal)
                } else null
                if (!fetchEbicsDocuments(client, orders, pinnedStartArg)) {
                    throw Exception("Failed to fetch documents")
                }
            } else {
                val wssNotification = listenForNotification(client)
                logger.info("Running with a frequency of ${cfg.fetch.frequencyRaw}")
                var nextFullRun = 0L
                while (true) {
                    val now = System.currentTimeMillis()
                    if (nextFullRun < now) {
                        fetchEbicsDocuments(client, orders, null)
                        nextFullRun = now + cfg.fetch.frequency.toMillis()
                    }
                    val delay = nextFullRun - now
                    if (wssNotification == null) {
                        logger.info("Running at frequency")
                        delay(delay)
                    } else {
                        val notifications = withTimeoutOrNull(delay) {
                            wssNotification.receive()
                        }
                        if (notifications != null) {
                            logger.info("Running at real-time notifications reception")
                            fetchEbicsDocuments(client, notifications, null)
                        }
                    }
                }
            }
        }
    }
}
