-- Copyright (c) 2015, Oracle and/or its affiliates. All rights reserved.
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; version 2 of the License.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

DROP FUNCTION IF EXISTS list_drop;

DELIMITER $$

CREATE DEFINER='mariadb.sys'@'localhost' FUNCTION list_drop (
        in_list TEXT,
        in_drop_value TEXT
    )
    RETURNS TEXT
    COMMENT '
             Description
             -----------

             Takes a list, and a value to attempt to remove from the list, and returns the resulting list.

             Useful for altering certain session variables, like sql_mode or optimizer_switch for instance.

             Parameters
             -----------

             in_list (TEXT):
               The comma separated list to drop a value from

             in_drop_value (TEXT):
               The value to drop from the input list

             Returns
             -----------

             TEXT

             Example
             --------

             mysql> select @@sql_mode;
             +-----------------------------------------------------------------------------------------------+
             | @@sql_mode                                                                                    |
             +-----------------------------------------------------------------------------------------------+
             | ANSI_QUOTES,ONLY_FULL_GROUP_BY,STRICT_TRANS_TABLES,NO_AUTO_CREATE_USER,NO_ENGINE_SUBSTITUTION |
             +-----------------------------------------------------------------------------------------------+
             1 row in set (0.00 sec)

             mysql> set sql_mode = sys.list_drop(@@sql_mode, ''ONLY_FULL_GROUP_BY'');
             Query OK, 0 rows affected (0.03 sec)

             mysql> select @@sql_mode;
             +----------------------------------------------------------------------------+
             | @@sql_mode                                                                 |
             +----------------------------------------------------------------------------+
             | ANSI_QUOTES,STRICT_TRANS_TABLES,NO_AUTO_CREATE_USER,NO_ENGINE_SUBSTITUTION |
             +----------------------------------------------------------------------------+
             1 row in set (0.00 sec)

            '
    SQL SECURITY INVOKER
    DETERMINISTIC
    CONTAINS SQL
BEGIN

    IF (in_drop_value IS NULL) THEN
        SIGNAL SQLSTATE '02200'
           SET MESSAGE_TEXT = 'Function sys.list_drop: in_drop_value input variable should not be NULL',
               MYSQL_ERRNO = 1138;
    END IF;

    IF (in_list IS NULL OR LENGTH(in_list) = 0) THEN
        -- return the list as it was passed in
        RETURN in_list;
    END IF;

    -- ensure that leading / trailing commas are remove, support values with either spaces or not between commas
    RETURN (SELECT TRIM(BOTH ',' FROM REPLACE(REPLACE(CONCAT(',', in_list), CONCAT(',', in_drop_value), ''), CONCAT(', ', in_drop_value), '')));

END$$

DELIMITER ;
