/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.transport;

import java.io.Closeable;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.CheckedBiConsumer;
import org.elasticsearch.common.lease.Releasable;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.KeyedLock;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.ConnectionProfile;
import org.elasticsearch.transport.NodeNotConnectedException;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportConnectionListener;

public class ConnectionManager
implements Closeable {
    private static final Logger logger = LogManager.getLogger(ConnectionManager.class);
    private final ConcurrentMap<DiscoveryNode, Transport.Connection> connectedNodes = ConcurrentCollections.newConcurrentMap();
    private final KeyedLock<String> connectionLock = new KeyedLock();
    private final Transport transport;
    private final ThreadPool threadPool;
    private final ConnectionProfile defaultProfile;
    private final AtomicBoolean isClosed = new AtomicBoolean(false);
    private final ReadWriteLock closeLock = new ReentrantReadWriteLock();
    private final DelegatingNodeConnectionListener connectionListener = new DelegatingNodeConnectionListener();

    public ConnectionManager(Settings settings, Transport transport, ThreadPool threadPool) {
        this(ConnectionProfile.buildDefaultConnectionProfile(settings), transport, threadPool);
    }

    public ConnectionManager(ConnectionProfile connectionProfile, Transport transport, ThreadPool threadPool) {
        this.transport = transport;
        this.threadPool = threadPool;
        this.defaultProfile = connectionProfile;
    }

    public void addListener(TransportConnectionListener listener) {
        this.connectionListener.listeners.addIfAbsent(listener);
    }

    public void removeListener(TransportConnectionListener listener) {
        this.connectionListener.listeners.remove(listener);
    }

    public Transport.Connection openConnection(DiscoveryNode node, ConnectionProfile connectionProfile) {
        ConnectionProfile resolvedProfile = ConnectionProfile.resolveConnectionProfile(connectionProfile, this.defaultProfile);
        return this.internalOpenConnection(node, resolvedProfile);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void connectToNode(DiscoveryNode node, ConnectionProfile connectionProfile, CheckedBiConsumer<Transport.Connection, ConnectionProfile, IOException> connectionValidator) throws ConnectTransportException {
        ConnectionProfile resolvedProfile = ConnectionProfile.resolveConnectionProfile(connectionProfile, this.defaultProfile);
        if (node == null) {
            throw new ConnectTransportException(null, "can't connect to a null node");
        }
        this.closeLock.readLock().lock();
        try {
            this.ensureOpen();
            try (Releasable ignored = this.connectionLock.acquire(node.getId());){
                Transport.Connection connection = (Transport.Connection)this.connectedNodes.get(node);
                if (connection != null) {
                    return;
                }
                boolean success = false;
                try {
                    connection = this.internalOpenConnection(node, resolvedProfile);
                    connectionValidator.accept(connection, resolvedProfile);
                    this.connectedNodes.put(node, connection);
                    if (logger.isDebugEnabled()) {
                        logger.debug("connected to node [{}]", (Object)node);
                    }
                    try {
                        this.connectionListener.onNodeConnected(node);
                    }
                    finally {
                        Transport.Connection finalConnection = connection;
                        connection.addCloseListener(ActionListener.wrap(() -> {
                            this.connectedNodes.remove(node, finalConnection);
                            this.connectionListener.onNodeDisconnected(node);
                        }));
                    }
                    if (connection.isClosed()) {
                        throw new NodeNotConnectedException(node, "connection concurrently closed");
                    }
                    success = true;
                    if (success) return;
                }
                catch (ConnectTransportException e) {
                    try {
                        throw e;
                        catch (Exception e2) {
                            throw new ConnectTransportException(node, "general node connection failure", e2);
                        }
                    }
                    catch (Throwable throwable) {
                        if (success) throw throwable;
                        logger.trace(() -> new ParameterizedMessage("failed to connect to [{}], cleaning dangling connections", (Object)node));
                        IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{connection});
                        throw throwable;
                    }
                }
                logger.trace(() -> new ParameterizedMessage("failed to connect to [{}], cleaning dangling connections", (Object)node));
                IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{connection});
                return;
            }
        }
        finally {
            this.closeLock.readLock().unlock();
        }
    }

    public Transport.Connection getConnection(DiscoveryNode node) {
        Transport.Connection connection = (Transport.Connection)this.connectedNodes.get(node);
        if (connection == null) {
            throw new NodeNotConnectedException(node, "Node not connected");
        }
        return connection;
    }

    public boolean nodeConnected(DiscoveryNode node) {
        return this.connectedNodes.containsKey(node);
    }

    public void disconnectFromNode(DiscoveryNode node) {
        Transport.Connection nodeChannels = (Transport.Connection)this.connectedNodes.remove(node);
        if (nodeChannels != null) {
            nodeChannels.close();
        }
    }

    public int size() {
        return this.connectedNodes.size();
    }

    @Override
    public void close() {
        if (this.isClosed.compareAndSet(false, true)) {
            CountDownLatch latch = new CountDownLatch(1);
            this.threadPool.generic().execute(() -> {
                this.closeLock.writeLock().lock();
                try {
                    Iterator iterator = this.connectedNodes.entrySet().iterator();
                    while (iterator.hasNext()) {
                        Map.Entry next = iterator.next();
                        try {
                            IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{(Closeable)next.getValue()});
                        }
                        finally {
                            iterator.remove();
                        }
                    }
                }
                finally {
                    this.closeLock.writeLock().unlock();
                    latch.countDown();
                }
            });
            try {
                latch.await(30L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Transport.Connection internalOpenConnection(DiscoveryNode node, ConnectionProfile connectionProfile) {
        Transport.Connection connection;
        PlainActionFuture<Transport.Connection> future = PlainActionFuture.newFuture();
        Releasable pendingConnection = this.transport.openConnection(node, connectionProfile, future);
        try {
            connection = (Transport.Connection)future.actionGet();
        }
        catch (IllegalStateException e) {
            if (e.getCause() instanceof InterruptedException) {
                pendingConnection.close();
            }
            throw e;
        }
        try {
            this.connectionListener.onConnectionOpened(connection);
        }
        finally {
            connection.addCloseListener(ActionListener.wrap(() -> this.connectionListener.onConnectionClosed(connection)));
        }
        if (connection.isClosed()) {
            throw new ConnectTransportException(node, "a channel closed while connecting");
        }
        return connection;
    }

    private void ensureOpen() {
        if (this.isClosed.get()) {
            throw new IllegalStateException("connection manager is closed");
        }
    }

    ConnectionProfile getConnectionProfile() {
        return this.defaultProfile;
    }

    private static final class DelegatingNodeConnectionListener
    implements TransportConnectionListener {
        private final CopyOnWriteArrayList<TransportConnectionListener> listeners = new CopyOnWriteArrayList();

        private DelegatingNodeConnectionListener() {
        }

        @Override
        public void onNodeDisconnected(DiscoveryNode key) {
            for (TransportConnectionListener listener : this.listeners) {
                listener.onNodeDisconnected(key);
            }
        }

        @Override
        public void onNodeConnected(DiscoveryNode node) {
            for (TransportConnectionListener listener : this.listeners) {
                listener.onNodeConnected(node);
            }
        }

        @Override
        public void onConnectionOpened(Transport.Connection connection) {
            for (TransportConnectionListener listener : this.listeners) {
                listener.onConnectionOpened(connection);
            }
        }

        @Override
        public void onConnectionClosed(Transport.Connection connection) {
            for (TransportConnectionListener listener : this.listeners) {
                listener.onConnectionClosed(connection);
            }
        }
    }
}

