/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.bin.format.elf;

import ghidra.app.cmd.refs.RemoveReferenceCmd;
import ghidra.app.util.bin.format.elf.ElfHeader;
import ghidra.app.util.bin.format.elf.ElfLoadHelper;
import ghidra.framework.model.DomainObject;
import ghidra.program.disassemble.Disassembler;
import ghidra.program.disassemble.DisassemblerMessageListener;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOutOfBoundsException;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataTypeConflictException;
import ghidra.program.model.data.Pointer32DataType;
import ghidra.program.model.data.Pointer64DataType;
import ghidra.program.model.data.PointerDataType;
import ghidra.program.model.listing.Data;
import ghidra.program.model.listing.Listing;
import ghidra.program.model.listing.Program;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.symbol.Reference;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolTable;
import ghidra.program.model.util.CodeUnitInsertionException;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.util.Iterator;

public class ElfDefaultGotPltMarkup {
    private ElfLoadHelper elfLoadHelper;
    private ElfHeader elf;
    private Program program;
    private Listing listing;
    private Memory memory;

    public ElfDefaultGotPltMarkup(ElfLoadHelper elfLoadHelper) {
        this.elfLoadHelper = elfLoadHelper;
        this.elf = elfLoadHelper.getElfHeader();
        this.program = elfLoadHelper.getProgram();
        this.listing = this.program.getListing();
        this.memory = this.program.getMemory();
    }

    private void log(String msg) {
        this.elfLoadHelper.log(msg);
    }

    private void log(Throwable t) {
        this.elfLoadHelper.log(t);
    }

    public void process(TaskMonitor monitor) throws CancelledException {
        this.processGOT(monitor);
        this.processPLT(monitor);
    }

    private void processGOT(TaskMonitor monitor) throws CancelledException {
        MemoryBlock[] blocks = this.memory.getBlocks();
        boolean imageBaseAlreadySet = this.elf.isPreLinked();
        for (int i = 0; i < blocks.length; ++i) {
            monitor.checkCanceled();
            MemoryBlock gotBlock = blocks[i];
            if (!gotBlock.getName().startsWith(".got")) continue;
            gotBlock.setWrite(false);
            Address newImageBase = null;
            Address addr = gotBlock.getStart();
            try {
                while (addr.compareTo((Object)gotBlock.getEnd()) < 0) {
                    Data data = this.createPointer(addr, true);
                    try {
                        addr = data.getMaxAddress().add(1L);
                    }
                    catch (AddressOutOfBoundsException e) {
                        break;
                    }
                    newImageBase = this.UglyImageBaseCheck(data, newImageBase);
                }
                if (newImageBase == null) continue;
                this.log("Invalid Address found in .got table.  Suspect Prelinked shared object file");
                if (imageBaseAlreadySet) {
                    this.log("ERROR: Unable to adjust image base for pre-link - retaining existing image base of " + this.program.getImageBase());
                    continue;
                }
                this.program.setImageBase(newImageBase, true);
                this.log("Setting Image base to: " + newImageBase);
                imageBaseAlreadySet = true;
                continue;
            }
            catch (Exception e) {
                this.log(e);
            }
        }
    }

    private void processPLT(TaskMonitor monitor) throws CancelledException {
        if (this.elf.isRelocatable()) {
            return;
        }
        MemoryBlock pltBlock = this.memory.getBlock(".plt");
        if (pltBlock == null || !pltBlock.isExecute() || pltBlock.getSize() <= 16L) {
            return;
        }
        int skipPointers = 16;
        if (this.elf.e_machine() == 40 || this.elf.e_machine() == 183) {
            skipPointers = 0;
        }
        Address minAddress = pltBlock.getStart().add((long)skipPointers);
        Address maxAddress = pltBlock.getEnd();
        this.processLinkageTable(".plt", minAddress, maxAddress, monitor);
    }

    public void processLinkageTable(String name, Address minAddress, Address maxAddress, TaskMonitor monitor) throws CancelledException {
        this.disassemble(minAddress, maxAddress, this.program, monitor);
        int count = this.convertSymbolsToExternalFunctions(minAddress, maxAddress);
        if (count > 0) {
            this.log("Converted " + count + " " + name + " section symbols to external thunks");
        }
    }

    private int convertSymbolsToExternalFunctions(Address minAddress, Address maxAddress) {
        Symbol s;
        Address symAddr;
        AddressSet set = new AddressSet();
        SymbolTable symbolTable = this.program.getSymbolTable();
        Iterator iterator = symbolTable.getPrimarySymbolIterator(minAddress, true).iterator();
        while (iterator.hasNext() && (symAddr = (s = (Symbol)iterator.next()).getAddress()).compareTo((Object)maxAddress) <= 0) {
            if (s.getSource() == SourceType.DEFAULT || this.listing.getDataAt(symAddr) != null) continue;
            set.add(symAddr);
        }
        if (set.isEmpty()) {
            return 0;
        }
        for (Address addr : set.getAddresses(true)) {
            Symbol s2 = symbolTable.getPrimarySymbol(addr);
            this.elfLoadHelper.createExternalFunctionLinkage(s2.getName(), addr, null);
        }
        return (int)set.getNumAddresses();
    }

    private void disassemble(Address start, Address end, Program prog, TaskMonitor monitor) throws CancelledException {
        DisassemblerMessageListener dml = msg -> {};
        AddressSet set = new AddressSet(start, end);
        Disassembler disassembler = Disassembler.getDisassembler((Program)prog, (TaskMonitor)monitor, (DisassemblerMessageListener)dml);
        while (!set.isEmpty()) {
            monitor.checkCanceled();
            AddressSet disset = disassembler.disassemble(set.getMinAddress(), (AddressSetView)set, true);
            if (disset.isEmpty()) {
                prog.getBookmarkManager().removeBookmarks((AddressSetView)set, "Error", "Bad Instruction", monitor);
                break;
            }
            set.delete((AddressSetView)disset);
        }
    }

    private Data createPointer(Address addr, boolean keepRefWhenValid) throws CodeUnitInsertionException, DataTypeConflictException {
        MemoryBlock block = this.memory.getBlock(addr);
        if (block == null || !block.isInitialized()) {
            return null;
        }
        int pointerSize = this.program.getDataTypeManager().getDataOrganization().getPointerSize();
        PointerDataType pointer = PointerDataType.dataType;
        if (this.elf.is32Bit() && pointerSize != 4) {
            pointer = Pointer32DataType.dataType;
        } else if (this.elf.is64Bit() && pointerSize != 8) {
            pointer = Pointer64DataType.dataType;
        }
        Data data = this.listing.getDataAt(addr);
        if (data == null || !pointer.isEquivalent(data.getDataType())) {
            if (data != null) {
                this.listing.clearCodeUnits(addr, addr.add((long)(pointerSize - 1)), false);
            }
            data = this.listing.createData(addr, (DataType)pointer);
        }
        Address refAddr = (Address)data.getValue();
        if (keepRefWhenValid) {
            if (this.memory.contains(refAddr)) {
                return data;
            }
            Symbol[] syms = this.program.getSymbolTable().getSymbols(refAddr);
            if (syms != null && syms.length > 0 && syms[0].getSource() != SourceType.DEFAULT) {
                return data;
            }
        }
        this.removeMemRefs(data);
        return data;
    }

    private void removeMemRefs(Data data) {
        if (data != null) {
            Reference[] refs = data.getValueReferences();
            for (int i = 0; i < refs.length; ++i) {
                RemoveReferenceCmd cmd = new RemoveReferenceCmd(refs[i]);
                cmd.applyTo((DomainObject)data.getProgram());
            }
        }
    }

    private Address UglyImageBaseCheck(Data data, Address imageBase) {
        if (this.elf.e_machine() != 40) {
            return null;
        }
        if (!this.elf.isSharedObject()) {
            return null;
        }
        if (imageBase != null) {
            return imageBase;
        }
        Object dValue = data.getValue();
        if (dValue == null || !(dValue instanceof Address)) {
            return null;
        }
        Address daddr = (Address)dValue;
        if (this.memory.contains(daddr)) {
            return null;
        }
        if (daddr.getOffset() < 4L) {
            return null;
        }
        if (this.program.getImageBase().getOffset() != 0L) {
            return null;
        }
        if (this.program.getRelocationTable().getRelocation(data.getAddress()) != null) {
            return null;
        }
        MemoryBlock tBlock = this.memory.getBlock(".text");
        if (tBlock == null) {
            return null;
        }
        Address topAddr = tBlock.getEnd();
        long byteMask = -1L;
        for (long topVal = topAddr.getOffset(); topVal != 0L; topVal >>>= 8) {
            byteMask <<= 8;
        }
        long newBase = daddr.getOffset() & byteMask;
        if (newBase == 0L) {
            return null;
        }
        return daddr.getNewAddress(newBase);
    }
}

