/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.app.util;

import java.util.ConcurrentModificationException;

import ghidra.docking.settings.Settings;
import ghidra.program.database.ProgramDB;
import ghidra.program.database.data.DataTypeManagerDB;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.data.*;
import ghidra.program.model.listing.Data;
import ghidra.program.model.listing.Program;
import ghidra.program.model.mem.MemBuffer;
import ghidra.program.model.mem.MemoryAccessException;
import ghidra.program.model.scalar.Scalar;
import ghidra.program.model.symbol.*;

/**
 * "Fake" data generated by the PseudoDisassembler.
 */
public class PseudoData extends PseudoCodeUnit implements Data {

	protected DataType dataType;
	protected DataType baseDataType;

	protected static final int OP_INDEX = 0;
	protected int level = 0;
	protected DataTypeManagerDB dataMgr;

	private static final int[] EMPTY_PATH = new int[0];

	public PseudoData(Program program, Address address, DataType dataType, MemBuffer memBuffer)
			throws AddressOverflowException {
		super(program, address, computeLength(dataType, address), memBuffer);
		if (dataType == null) {
			dataType = DataType.DEFAULT;
		}
		this.dataType = dataType;
		baseDataType = getBaseDataType(dataType);
		if (program instanceof ProgramDB) {
			dataMgr = ((ProgramDB) program).getDataManager();
		}
	}

	public PseudoData(Address address, DataType dataType, MemBuffer memBuffer)
			throws AddressOverflowException {
		this(null, address, dataType, memBuffer);
	}

	protected static DataType getBaseDataType(DataType dataType) {
		DataType baseDataType = dataType;
		if (baseDataType instanceof TypeDef) {
			baseDataType = ((TypeDef) baseDataType).getBaseDataType();
		}
		return baseDataType;
	}

	protected static int computeLength(DataType dataType, Address address) {
		if (dataType == null) {
			return 1;
		}
		int length = dataType.getLength();
		if (length < 1) {
			if (getBaseDataType(dataType) instanceof Pointer) {
				length = address.getPointerSize();
			}
			else {
				length = 1;
			}
		}
		return length;
	}

	/**
	 * @see ghidra.program.model.listing.Data#addValueReference(ghidra.program.model.address.Address, ghidra.program.model.symbol.RefType)
	 */
	@Override
	public void addValueReference(Address refAddr, RefType type) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see ghidra.program.model.listing.Data#removeValueReference(ghidra.program.model.address.Address)
	 */
	@Override
	public void removeValueReference(Address refAddr) {
		throw new UnsupportedOperationException();

	}

	/** 
	 * @see ghidra.program.model.listing.Data#getComponent(int)
	 */
	@Override
	public Data getComponent(int index) {
		if (index < 0 || index >= getNumComponents()) {
			return null;
		}

		Data data;
		try {
			data = null;
			if (baseDataType instanceof Array) {
				Array array = (Array) baseDataType;
				int elementLength = array.getElementLength();
				data = new PseudoDataComponent(program, address.add(index * elementLength), this,
					array.getDataType(), index, index * elementLength, elementLength, this);

			}
			else if (baseDataType instanceof Composite) {
				Composite struct = (Composite) baseDataType;
				DataTypeComponent dtc = struct.getComponent(index);
				data =
					new PseudoDataComponent(program, address.add(dtc.getOffset()), this, dtc, this);

			}
			else if (baseDataType instanceof DynamicDataType) {
				DynamicDataType ddt = (DynamicDataType) baseDataType;
				DataTypeComponent dtc = ddt.getComponent(index, this);
				data =
					new PseudoDataComponent(program, address.add(dtc.getOffset()), this, dtc, this);
			}
		}
		catch (MemoryAccessException | AddressOverflowException e) {
			throw new ConcurrentModificationException("Data type length changed");
		}
		return data;
	}

	/**
	 * @see ghidra.program.model.listing.CodeUnit#getAddress(int)
	 */
	@Override
	public Address getAddress(int opIndex) {
		if (opIndex == 0) {
			Object obj = getValue();
			if (obj instanceof Address) {
				return (Address) obj;
			}
		}
		return null;
	}

	/**
	 * @see ghidra.program.model.listing.CodeUnit#getByteCodeString()
	 */
	public String getByteCodeString() {
		StringBuffer bytesStr = new StringBuffer();

		for (int i = 0; i < length; i++) {
			if (i != 0) {
				bytesStr.append(" ");
			}
			String hex;
			try {
				hex = Integer.toHexString(getByte(i));
			}
			catch (MemoryAccessException e) {
				hex = "??";
			}
			if (hex.length() == 1) {
				bytesStr.append("0");
			}
			if (hex.length() > 2) {
				bytesStr.append(hex.substring(hex.length() - 2));
			}
			else {
				bytesStr.append(hex);
			}
		}
		return bytesStr.toString();
	}

	/**
	 * 
	 * @see ghidra.program.model.listing.CodeUnit#toString()
	 */
	@Override
	public String toString() {
		String valueRepresentation = getDefaultValueRepresentation();
		String mnemonicString = getMnemonicString();
		if (valueRepresentation == null) {
			return mnemonicString;
		}
		return mnemonicString + " " + valueRepresentation;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getDefaultValueRepresentation()
	 */
	@Override
	public String getDefaultValueRepresentation() {
		if (getLength() < dataType.getLength()) {
			return "TooBig: " + dataType.getDisplayName() + " need " + dataType.getLength() +
				" have " + getLength();
		}
		return dataType.getRepresentation(this, this, getLength());
	}

	/**
	 * @see ghidra.program.model.listing.CodeUnit#getMnemonicString()
	 */
	@Override
	public String getMnemonicString() {
		return dataType.getMnemonic(this);
	}

	/**
	 * @see ghidra.program.model.listing.CodeUnit#getNumOperands()
	 */
	@Override
	public int getNumOperands() {
		return 1;
	}

	/**
	 * @see ghidra.program.model.listing.CodeUnit#getScalar(int)
	 */
	@Override
	public Scalar getScalar(int opIndex) {
		if (opIndex == 0) {
			Object obj = getValue();
			if (obj instanceof Scalar) {
				return (Scalar) obj;
			}
			else if (obj instanceof Address) {
				Address addrObj = (Address) obj;
				long offset = addrObj.getAddressableWordOffset();
				return new Scalar(addrObj.getAddressSpace().getPointerSize() * 8, offset, false);
			}
		}
		return null;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getBaseDataType()
	 */
	@Override
	public DataType getBaseDataType() {
		return baseDataType;
	}

	/*
	 *  (non-Javadoc)
	 * @see ghidra.program.model.data.Settings#clearSetting(java.lang.String)
	 */
	@Override
	public void clearSetting(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see ghidra.docking.settings.Settings#getByteArray(java.lang.String)
	 */
	@Override
	public byte[] getByteArray(String name) {
		return null;
	}

	/**
	 * @see ghidra.docking.settings.Settings#getLong(java.lang.String)
	 */
	@Override
	public Long getLong(String name) {
		return null;
	}

	/**
	 * @see ghidra.docking.settings.Settings#getNames()
	 */
	@Override
	public String[] getNames() {
		return new String[0];
	}

	/**
	 * @see ghidra.docking.settings.Settings#getString(java.lang.String)
	 */
	@Override
	public String getString(String name) {
		return null;
	}

	/**
	 * @see ghidra.docking.settings.Settings#getValue(java.lang.String)
	 */
	@Override
	public Object getValue(String name) {
		if (baseDataType != null) {
			return baseDataType.getValue(this, this, length);
		}
		return null;
	}

	/**
	 * @see ghidra.docking.settings.Settings#setByteArray(java.lang.String, byte[])
	 */
	@Override
	public void setByteArray(String name, byte[] value) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see ghidra.docking.settings.Settings#setLong(java.lang.String, long)
	 */
	@Override
	public void setLong(String name, long value) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see ghidra.docking.settings.Settings#setString(java.lang.String, java.lang.String)
	 */
	@Override
	public void setString(String name, String value) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see ghidra.docking.settings.Settings#setValue(java.lang.String, java.lang.Object)
	 */
	@Override
	public void setValue(String name, Object value) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see ghidra.program.model.listing.Data#getComponent(int[])
	 */
	@Override
	public Data getComponent(int[] componentPath) {
		if (componentPath == null || componentPath.length <= level) {
			return this;
		}
		Data component = getComponent(componentPath[level]);
		return (component == null ? null : component.getComponent(componentPath));
	}

	/**
	 * @see ghidra.program.model.listing.Data#getComponentAt(int)
	 */
	@Override
	public Data getComponentAt(int offset) {
		if (offset < 0 || offset >= length) {
			return null;
		}

		if (baseDataType instanceof Array) {
			Array array = (Array) baseDataType;
			int elementLength = array.getElementLength();
			int index = offset / elementLength;
			return getComponent(index);

		}
		else if (baseDataType instanceof Union) {
			return getComponent(0);
		}
		else if (baseDataType instanceof Structure) {
			Structure struct = (Structure) baseDataType;
			DataTypeComponent dtc = struct.getComponentAt(offset);
			return getComponent(dtc.getOrdinal());

		}
		else if (baseDataType instanceof DynamicDataType) {
			DynamicDataType ddt = (DynamicDataType) baseDataType;
			DataTypeComponent dtc = ddt.getComponentAt(offset, this);
			return getComponent(dtc.getOrdinal());
		}
		return null;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getComponentIndex()
	 */
	@Override
	public int getComponentIndex() {
		return -1;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getComponentLevel()
	 */
	@Override
	public int getComponentLevel() {
		return level;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getComponentPath()
	 */
	@Override
	public int[] getComponentPath() {
		return EMPTY_PATH;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getComponentPathName()
	 */
	@Override
	public String getComponentPathName() {
		return null;
	}

//	/**
//	 * @see ghidra.program.model.listing.Data#getComponents()
//	 */
//	public Data[] getComponents() {
//        if (length < dataType.getLength()) {
//            return null;
//        }
//        Data[] retData = EMPTY_COMPONENTS;
//        if (baseDataType instanceof Composite) {
//			Composite composite = (Composite)baseDataType;
//			int n = composite.getNumComponents();
//			retData = new Data[n];
//			for(int i=0;i<n;i++) {
//				retData[i] = getComponent(i);
//			}
//        }	
//		else if (baseDataType instanceof Array) {
//			Array array = (Array)baseDataType;
//			int n = array.getNumElements();
//			retData = new Data[n];
//			for(int i=0;i<n;i++) {
//				retData[i] = getComponent(i);
//			}        
//		}
//		else if (baseDataType instanceof DynamicDataType) {
//			DynamicDataType ddt = (DynamicDataType)baseDataType;
//			int n = ddt.getNumComponents(this);
//			retData = new Data[n];
//			for(int i=0;i<n;i++) {
//				retData[i] = getComponent(i);
//			}
//		}
//		return retData;
//	}

	/**
	 * @see ghidra.program.model.listing.Data#getDataType()
	 */
	@Override
	public DataType getDataType() {
		return dataType;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getFieldName()
	 */
	@Override
	public String getFieldName() {
		return null;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getNumComponents()
	 */
	@Override
	public int getNumComponents() {
		if (length < dataType.getLength()) {
			return -1;
		}
		if (baseDataType instanceof Composite) {
			return ((Composite) baseDataType).getNumComponents();
		}
		else if (baseDataType instanceof Array) {
			return ((Array) baseDataType).getNumElements();
		}
		else if (baseDataType instanceof DynamicDataType) {
			return ((DynamicDataType) baseDataType).getNumComponents(this);
		}
		return 0;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getParent()
	 */
	@Override
	public Data getParent() {
		return null;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getParentOffset()
	 */
	@Override
	public int getParentOffset() {
		return 0;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getPathName()
	 */
	@Override
	public String getPathName() {
		if (program != null) {
			SymbolTable st = program.getSymbolTable();
			Symbol symbol = st.getPrimarySymbol(address);
			if (symbol != null) {
				return symbol.getName();
			}
		}
		return "DAT" + address.toString();
	}

	/**
	 * @see ghidra.program.model.listing.Data#getPrimitiveAt(int)
	 */
	@Override
	public Data getPrimitiveAt(int offset) {
		if (offset < 0 || offset >= length) {
			return null;
		}
		Data dc = getComponentAt(offset);
		if (dc == null || dc == this) {
			return this;
		}
		return dc.getPrimitiveAt(offset - dc.getParentOffset());
	}

	/**
	 * @see ghidra.program.model.listing.Data#getRoot()
	 */
	@Override
	public Data getRoot() {
		return this;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getRootOffset()
	 */
	@Override
	public int getRootOffset() {
		return 0;
	}

	/**
	 * @see ghidra.program.model.listing.Data#getValue()
	 */
	@Override
	public Object getValue() {
		return baseDataType.getValue(this, this, length);
	}

	@Override
	public Class<?> getValueClass() {
		DataType dt = getBaseDataType();
		if (dt != null) {
			return dt.getValueClass(this);
		}
		return null;
	}

	@Override
	public boolean hasStringValue() {
		return String.class.equals(getValueClass());
	}

	/**
	 * @see ghidra.program.model.listing.Data#getValueReferences()
	 */
	@Override
	public Reference[] getValueReferences() {
		if (refMgr == null) {
			return new Reference[0];
		}
		return refMgr.getReferencesFrom(address, OP_INDEX);
	}

	/**
	 * @see ghidra.program.model.listing.Data#isArray()
	 */
	@Override
	public boolean isArray() {
		return baseDataType instanceof Array;
	}

	/**
	 * @see ghidra.program.model.listing.Data#isDefined()
	 */
	@Override
	public boolean isDefined() {
		return !(dataType instanceof DefaultDataType);
	}

	/**
	 * @see ghidra.program.model.listing.Data#isPointer()
	 */
	@Override
	public boolean isPointer() {
		return baseDataType instanceof Pointer;
	}

	/**
	 * @see ghidra.program.model.listing.Data#isStructure()
	 */
	@Override
	public boolean isStructure() {
		return baseDataType instanceof Structure;
	}

	/**
	 * @see ghidra.program.model.listing.Data#isDynamic()
	 */
	@Override
	public boolean isDynamic() {
		return baseDataType instanceof DynamicDataType;
	}

	/**
	 * @see ghidra.program.model.listing.Data#isUnion()
	 */
	@Override
	public boolean isUnion() {
		return baseDataType instanceof Union;
	}

	/**
	 * @see ghidra.docking.settings.Settings#clearAllSettings()
	 */
	@Override
	public void clearAllSettings() {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see ghidra.docking.settings.Settings#isEmpty()
	 */
	@Override
	public boolean isEmpty() {
		if (dataMgr == null) {
			return true;
		}
		return dataMgr.isEmptySetting(address);
	}

	@Override
	public String getDefaultLabelPrefix(DataTypeDisplayOptions options) {
		return null;
	}

	@Override
	public boolean equals(Object obj) {

		if (obj == null) {
			return false;
		}
		if (obj == this) {
			return true;
		}

		if (getClass() != obj.getClass()) {
			return false;
		}

		PseudoData data = (PseudoData) obj;
		if (!address.equals(data.address)) {
			return false;
		}
		return dataType.isEquivalent(data.dataType);
	}

	@Override
	public boolean isConstant() {
		return false;
	}

	@Override
	public boolean isVolatile() {
		return false;
	}

	/**
	 * @see ghidra.docking.settings.Settings#getDefaultSettings()
	 */
	@Override
	public Settings getDefaultSettings() {
		return dataType.getDefaultSettings();
	}
}
