/*	$OpenBSD: if_rgereg.h,v 1.1 2019/11/18 03:03:37 kevlo Exp $	*/

/*
 * Copyright (c) 2019 Kevin Lo <kevlo@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define RGE_PCI_BAR0		PCI_MAPREG_START
#define RGE_PCI_BAR1		(PCI_MAPREG_START + 4)
#define RGE_PCI_BAR2		(PCI_MAPREG_START + 8)

#define RGE_MAC0		0x0000
#define RGE_MAC4		0x0004
#define RGE_MAR0		0x0008
#define RGE_MAR4		0x000c
#define RGE_TXDESC_ADDR_LO	0x0020
#define RGE_TXDESC_ADDR_HI	0x0024
#define RGE_CMD			0x0037
#define RGE_IMR			0x0038
#define RGE_ISR			0x003c
#define RGE_TXCFG		0x0040
#define RGE_RXCFG		0x0044
#define RGE_TIMERCNT		0x0048
#define RGE_EECMD		0x0050
#define RGE_CFG0		0x0051
#define RGE_CFG1		0x0052
#define RGE_CFG2		0x0053
#define RGE_CFG3		0x0054
#define RGE_CFG4		0x0055
#define RGE_CFG5		0x0056
#define RGE_TDFNR		0x0057
#define RGE_TIMERINT		0x0058
#define RGE_CSIDR		0x0064
#define RGE_CSIAR		0x0068
#define RGE_PHYSTAT		0x006c
#define RGE_PMCH		0x006f
#define RGE_EPHYAR		0x0080
#define RGE_TXSTART		0x0090
#define RGE_MACOCP		0x00b0
#define RGE_PHYOCP		0x00b8
#define RGE_TWICMD		0x00d2
#define RGE_MCUCMD		0x00d3
#define RGE_RXMAXSIZE		0x00da
#define RGE_CPLUSCMD		0x00e0
#define RGE_RXDESC_ADDR_LO	0x00e4
#define RGE_RXDESC_ADDR_HI	0x00e8
#define RGE_PPSW		0x00f2
#define RGE_IM(i)		(0x0a00 + (i) * 4)
#define RGE_PHYBASE		0x0a40
#define RGE_ADDR0		0x19e0
#define RGE_ADDR1		0x19e4

/* Flags for register RGE_CMD */
#define RGE_CMD_RXBUF_EMPTY	0x01
#define RGE_CMD_TXENB		0x04
#define RGE_CMD_RXENB		0x08
#define RGE_CMD_RESET		0x10

/* Flags for register RGE_ISR */
#define RGE_ISR_RX_OK		0x00000001
#define RGE_ISR_RX_ERR		0x00000002
#define RGE_ISR_TX_OK		0x00000004
#define RGE_ISR_TX_ERR		0x00000008
#define RGE_ISR_RX_DESC_UNAVAIL	0x00000010
#define RGE_ISR_LINKCHG		0x00000020 
#define RGE_ISR_RX_FIFO_OFLOW	0x00000040
#define RGE_ISR_TX_DESC_UNAVAIL	0x00000080
#define RGE_ISR_SWI		0x00000100
#define RGE_ISR_PCS_TIMEOUT	0x00004000
#define RGE_ISR_SYSTEM_ERR	0x00008000

#define RGE_INTRS		\
	(RGE_ISR_RX_OK | RGE_ISR_RX_ERR | RGE_ISR_TX_OK |		\
	RGE_ISR_TX_ERR | RGE_ISR_RX_DESC_UNAVAIL |			\
	RGE_ISR_RX_FIFO_OFLOW | RGE_ISR_SYSTEM_ERR)

#define RGE_INTRS_TIMER		\
	(RGE_ISR_RX_ERR | RGE_ISR_TX_ERR | RGE_ISR_PCS_TIMEOUT |	\
	RGE_ISR_SYSTEM_ERR)

/* Flags for register RGE_TXCFG */
#define RGE_TXCFG_HWREV		0x7cf00000

/* Flags for register RGE_RXCFG */
#define RGE_RXCFG_ALLPHYS	0x00000001
#define RGE_RXCFG_INDIV		0x00000002
#define RGE_RXCFG_MULTI		0x00000004
#define RGE_RXCFG_BROAD		0x00000008
#define RGE_RXCFG_RUNT		0x00000010
#define RGE_RXCFG_ERRPKT	0x00000020
#define RGE_RXCFG_VLANSTRIP	0x00c00000

/* Flags for register RGE_EECMD */
#define RGE_EECMD_WRITECFG	0xc0

/* Flags for register RGE_CFG1 */
#define RGE_CFG1_SPEED_DOWN	0x10

/* Flags for register RGE_CFG2 */
#define RGE_CFG2_CLKREQ_EN	0x80

/* Flags for register RGE_CFG3 */
#define RGE_CFG3_RDY_TO_L23	0x02

/* Flags for register RGE_CFG5 */
#define RGE_CFG5_PME_STS	0x01

/* Flags for register RGE_CSIAR */
#define RGE_CSIAR_BYTE_EN	0x0000000f
#define RGE_CSIAR_BYTE_EN_SHIFT	12
#define RGE_CSIAR_ADDR_MASK	0x00000fff
#define RGE_CSIAR_BUSY		0x80000000

/* Flags for register RGE_PHYSTAT */
#define RGE_PHYSTAT_FDX		0x0001
#define RGE_PHYSTAT_LINK	0x0002
#define RGE_PHYSTAT_10MBPS	0x0004
#define RGE_PHYSTAT_100MBPS	0x0008
#define RGE_PHYSTAT_1000MBPS	0x0010
#define RGE_PHYSTAT_RXFLOW	0x0020
#define RGE_PHYSTAT_TXFLOW	0x0040
#define RGE_PHYSTAT_2500MBPS	0x0400

/* Flags for register RGE_EPHYAR */
#define RGE_EPHYAR_DATA_MASK	0x0000ffff
#define RGE_EPHYAR_BUSY		0x80000000
#define RGE_EPHYAR_ADDR_MASK	0x0000007f
#define RGE_EPHYAR_ADDR_SHIFT	16

/* Flags for register RGE_TXSTART */
#define RGE_TXSTART_START	0x0001

/* Flags for register RGE_MACOCP */
#define RGE_MACOCP_DATA_MASK	0x0000ffff
#define RGE_MACOCP_BUSY		0x80000000
#define RGE_MACOCP_ADDR_SHIFT	16

/* Flags for register RGE_PHYOCP */
#define RGE_PHYOCP_DATA_MASK	0x0000ffff
#define RGE_PHYOCP_BUSY		0x80000000
#define RGE_PHYOCP_ADDR_SHIFT	16

/* Flags for register RGE_MCUCMD */
#define RGE_MCUCMD_RXFIFO_EMPTY	0x10
#define RGE_MCUCMD_TXFIFO_EMPTY	0x20
#define RGE_MCUCMD_IS_OOB	0x80

/* Flags for register RGE_CPLUSCMD */
#define RGE_CPLUSCMD_RXCSUM	0x0020

#define RGE_TX_NSEGS		32
#define	RGE_TX_LIST_CNT		1024
#define	RGE_RX_LIST_CNT		1024
#define RGE_ALIGN		256
#define RGE_TX_LIST_SZ		(sizeof(struct rge_tx_desc) * RGE_TX_LIST_CNT)
#define RGE_RX_LIST_SZ		(sizeof(struct rge_rx_desc) * RGE_RX_LIST_CNT)
#define RGE_NEXT_TX_DESC(x)	(((x) + 1) % RGE_TX_LIST_CNT)
#define RGE_NEXT_RX_DESC(x)	(((x) + 1) % RGE_RX_LIST_CNT)
#define RGE_ADDR_LO(y)		((uint64_t) (y) & 0xffffffff)
#define RGE_ADDR_HI(y)		((uint64_t) (y) >> 32)
#define RGE_OWN(x)		(letoh32((x)->rge_cmdsts) & RGE_RDCMDSTS_OWN)
#define RGE_RXBYTES(x)          (letoh32((x)->rge_cmdsts) & \
				RGE_RDCMDSTS_FRAGLEN)

#define RGE_ADV_2500TFDX	0x0080

/* Tx descriptor */
struct rge_tx_desc {
	uint32_t		rge_cmdsts;
	uint32_t		rge_extsts;
	uint32_t		rge_addrlo;
	uint32_t		rge_addrhi;
	uint32_t		reserved[4];
};

#define RGE_TDCMDSTS_COLL	0x000f0000
#define RGE_TDCMDSTS_EXCESSCOLL	0x00100000
#define RGE_TDCMDSTS_TXERR	0x00800000
#define RGE_TDCMDSTS_EOF	0x10000000
#define RGE_TDCMDSTS_SOF	0x20000000
#define RGE_TDCMDSTS_EOR	0x40000000
#define RGE_TDCMDSTS_OWN	0x80000000

#define RGE_TDEXTSTS_VTAG	0x00020000
#define RGE_TDEXTSTS_IPCSUM	0x20000000
#define RGE_TDEXTSTS_TCPCSUM	0x40000000
#define RGE_TDEXTSTS_UDPCSUM	0x80000000

/* Rx descriptor */
struct rge_rx_desc {
	uint32_t		rge_cmdsts;
	uint32_t		rge_extsts;
	uint32_t		rge_addrlo;
	uint32_t		rge_addrhi;
};

#define RGE_RDCMDSTS_TCPCSUMERR	0x00004000
#define RGE_RDCMDSTS_UDPCSUMERR	0x00008000
#define RGE_RDCMDSTS_IPCSUMERR	0x00010000
#define RGE_RDCMDSTS_TCPPKT	0x00020000
#define RGE_RDCMDSTS_UDPPKT	0x00040000
#define RGE_RDCMDSTS_RXERRSUM	0x00200000
#define RGE_RDCMDSTS_EOF	0x10000000
#define RGE_RDCMDSTS_SOF	0x20000000
#define RGE_RDCMDSTS_EOR	0x40000000
#define RGE_RDCMDSTS_OWN	0x80000000
#define RGE_RDCMDSTS_FRAGLEN	0x00003fff

#define RGE_RDEXTSTS_VTAG	0x00010000
#define RGE_RDEXTSTS_VLAN_MASK	0x0000ffff
#define RGE_RDEXTSTS_IPV4	0x40000000
#define RGE_RDEXTSTS_IPV6	0x80000000

struct rge_txq {
	struct mbuf		*txq_mbuf;
	bus_dmamap_t		txq_dmamap;
	int			txq_descidx;
};

struct rge_rxq {
	struct mbuf		*rxq_mbuf;
	bus_dmamap_t		rxq_dmamap;
};

struct rge_list_data {
	struct rge_txq		rge_txq[RGE_TX_LIST_CNT];
	int			rge_txq_prodidx;
	int			rge_txq_considx;
	struct rge_rxq		rge_rxq[RGE_RX_LIST_CNT];
	int			rge_rxq_prodidx;

	bus_dma_segment_t	rge_tx_listseg;
	int			rge_tx_listnseg;
	bus_dmamap_t		rge_tx_list_map;
	struct rge_tx_desc	*rge_tx_list;
	bus_dma_segment_t	rge_rx_listseg;
	int			rge_rx_listnseg;
	bus_dmamap_t		rge_rx_list_map;
	struct rge_rx_desc	*rge_rx_list;
};

/* Microcode version */
#define RGE_MAC_CFG2_MCODE_VER	0x0b11
#define RGE_MAC_CFG3_MCODE_VER	0x0b33

enum rge_mac_type {
	MAC_CFG_UNKNOWN = 1,
	MAC_CFG2,
	MAC_CFG3
};

#define RGE_TIMEOUT		100

#define RGE_JUMBO_FRAMELEN	9216
#define RGE_JUMBO_MTU							\
	(RGE_JUMBO_FRAMELEN - ETHER_HDR_LEN - ETHER_CRC_LEN - 		\
	ETHER_VLAN_ENCAP_LEN)

#define RGE_TXCFG_CONFIG	0x03000700
#define RGE_RXCFG_CONFIG	0x40c00700

struct rge_softc {
	struct device		sc_dev;
	struct arpcom		sc_arpcom;	/* Ethernet common data */
	void			*sc_ih;		/* interrupt vectoring */
	bus_space_handle_t	rge_bhandle;	/* bus space handle */
	bus_space_tag_t		rge_btag;	/* bus space tag */
	bus_size_t		rge_bsize;
	bus_dma_tag_t		sc_dmat;
	pci_chipset_tag_t	sc_pc;
	pcitag_t		sc_tag;
	bus_dma_segment_t 	sc_rx_seg;
	bus_dmamap_t		sc_rx_dmamap;
	struct ifmedia		sc_media;	/* media info */
	enum rge_mac_type	rge_type;
	struct mbuf		*rge_head;
	struct mbuf		*rge_tail;

	struct rge_list_data	rge_ldata;

	struct task		sc_task;

	struct timeout		sc_timeout;	/* tick timeout */

	uint32_t		rge_flags;
#define RGE_FLAG_MSI		0x00000001

	uint32_t		rge_intrs;
	uint32_t		rge_tx_ack;
	uint32_t		rge_rx_ack;
	int			rge_timerintr;

#define RGE_IMTYPE_NONE		0
#define RGE_IMTYPE_SIM		1
};

/*
 * Register space access macros.
 */
#define RGE_WRITE_RAW_4(sc, reg, val)	\
	bus_space_write_raw_region_4(sc->rge_btag, sc->rge_bhandle, reg, val, 4)
#define RGE_WRITE_4(sc, reg, val)	\
	bus_space_write_4(sc->rge_btag, sc->rge_bhandle, reg, val)
#define RGE_WRITE_2(sc, reg, val)	\
	bus_space_write_2(sc->rge_btag, sc->rge_bhandle, reg, val)
#define RGE_WRITE_1(sc, reg, val)	\
	bus_space_write_1(sc->rge_btag, sc->rge_bhandle, reg, val)

#define RGE_READ_4(sc, reg)		\
	bus_space_read_4(sc->rge_btag, sc->rge_bhandle, reg)
#define RGE_READ_2(sc, reg)		\
	bus_space_read_2(sc->rge_btag, sc->rge_bhandle, reg)
#define RGE_READ_1(sc, reg)		\
	bus_space_read_1(sc->rge_btag, sc->rge_bhandle, reg)

#define RGE_SETBIT_4(sc, reg, val)	\
	RGE_WRITE_4(sc, reg, RGE_READ_4(sc, reg) | (val))
#define RGE_SETBIT_2(sc, reg, val)	\
	RGE_WRITE_2(sc, reg, RGE_READ_2(sc, reg) | (val))
#define RGE_SETBIT_1(sc, reg, val)	\
	RGE_WRITE_1(sc, reg, RGE_READ_1(sc, reg) | (val))

#define RGE_CLRBIT_4(sc, reg, val)	\
	RGE_WRITE_4(sc, reg, RGE_READ_4(sc, reg) & ~(val))
#define RGE_CLRBIT_2(sc, reg, val)	\
	RGE_WRITE_2(sc, reg, RGE_READ_2(sc, reg) & ~(val))
#define RGE_CLRBIT_1(sc, reg, val)	\
	RGE_WRITE_1(sc, reg, RGE_READ_1(sc, reg) & ~(val))

#define RGE_PHY_SETBIT(sc, reg, val)	\
	rge_write_phy_ocp(sc, reg, rge_read_phy_ocp(sc, reg) | (val))

#define RGE_PHY_CLRBIT(sc, reg, val)	\
	rge_write_phy_ocp(sc, reg, rge_read_phy_ocp(sc, reg) & ~(val))

#define RGE_MAC_SETBIT(sc, reg, val)	\
	rge_write_mac_ocp(sc, reg, rge_read_mac_ocp(sc, reg) | (val))

#define RGE_MAC_CLRBIT(sc, reg, val)	\
	rge_write_mac_ocp(sc, reg, rge_read_mac_ocp(sc, reg) & ~(val))

#define RTL8125_DEF_BPS		\
	{ 0xf800, 0xe008 },	\
	{ 0xf802, 0xe01e },	\
	{ 0xf804, 0xe02e },	\
	{ 0xf806, 0xe054 },	\
	{ 0xf808, 0xe057 },	\
	{ 0xf80a, 0xe059 },	\
	{ 0xf80c, 0xe05b },	\
	{ 0xf80e, 0xe05d },	\
	{ 0xf810, 0x9996 },	\
	{ 0xf812, 0x49d1 },	\
	{ 0xf814, 0xf005 },	\
	{ 0xf816, 0x49d4 },	\
	{ 0xf818, 0xf10a },	\
	{ 0xf81a, 0x49d8 },	\
	{ 0xf81c, 0xf108 },	\
	{ 0xf81e, 0xc00f },	\
	{ 0xf820, 0x7100 },	\
	{ 0xf822, 0x209c },	\
	{ 0xf824, 0x249c },	\
	{ 0xf826, 0xc009 },	\
	{ 0xf828, 0x9900 },	\
	{ 0xf82a, 0xe004 },	\
	{ 0xf82c, 0xc006 },	\
	{ 0xf82e, 0x1900 },	\
	{ 0xf830, 0x9900 },	\
	{ 0xf832, 0xc602 },	\
	{ 0xf834, 0xbe00 },	\
	{ 0xf836, 0x5a48 },	\
	{ 0xf838, 0xe0c2 },	\
	{ 0xf83a, 0x0004 },	\
	{ 0xf83c, 0xe10a },	\
	{ 0xf83e, 0xc60f },	\
	{ 0xf840, 0x73c4 },	\
	{ 0xf842, 0x49b3 },	\
	{ 0xf844, 0xf106 },	\
	{ 0xf846, 0x73c2 },	\
	{ 0xf848, 0xc608 },	\
	{ 0xf84a, 0xb406 },	\
	{ 0xf84c, 0xc609 },	\
	{ 0xf84e, 0xff80 },	\
	{ 0xf850, 0xc605 },	\
	{ 0xf852, 0xb406 },	\
	{ 0xf854, 0xc605 },	\
	{ 0xf856, 0xff80 },	\
	{ 0xf858, 0x0544 },	\
	{ 0xf85a, 0x0568 },	\
	{ 0xf85c, 0xe906 },	\
	{ 0xf85e, 0xcde8 },	\
	{ 0xf860, 0xc724 },	\
	{ 0xf862, 0xc624 },	\
	{ 0xf864, 0x9ee2 },	\
	{ 0xf866, 0x1e01 },	\
	{ 0xf868, 0x9ee0 },	\
	{ 0xf86a, 0x76e0 },	\
	{ 0xf86c, 0x49e0 },	\
	{ 0xf86e, 0xf1fe },	\
	{ 0xf870, 0x76e6 },	\
	{ 0xf872, 0x486d },	\
	{ 0xf874, 0x4868 },	\
	{ 0xf876, 0x9ee4 },	\
	{ 0xf878, 0x1e03 },	\
	{ 0xf87a, 0x9ee0 },	\
	{ 0xf87c, 0x76e0 },	\
	{ 0xf87e, 0x49e0 },	\
	{ 0xf880, 0xf1fe },	\
	{ 0xf882, 0xc615 },	\
	{ 0xf884, 0x9ee2 },	\
	{ 0xf886, 0x1e01 },	\
	{ 0xf888, 0x9ee0 },	\
	{ 0xf88a, 0x76e0 },	\
	{ 0xf88c, 0x49e0 },	\
	{ 0xf88e, 0xf1fe },	\
	{ 0xf890, 0x76e6 },	\
	{ 0xf892, 0x486f },	\
	{ 0xf894, 0x9ee4 },	\
	{ 0xf896, 0x1e03 },	\
	{ 0xf898, 0x9ee0 },	\
	{ 0xf89a, 0x76e0 },	\
	{ 0xf89c, 0x49e0 },	\
	{ 0xf89e, 0xf1fe },	\
	{ 0xf8a0, 0x7196 },	\
	{ 0xf8a2, 0xc702 },	\
	{ 0xf8a4, 0xbf00 },	\
	{ 0xf8a6, 0x5a44 },	\
	{ 0xf8a8, 0xeb0e },	\
	{ 0xf8aa, 0x0070 },	\
	{ 0xf8ac, 0x00c3 },	\
	{ 0xf8ae, 0x1bc0 },	\
	{ 0xf8b0, 0xc602 },	\
	{ 0xf8b2, 0xbe00 },	\
	{ 0xf8b4, 0x0e26 },	\
	{ 0xf8b6, 0xc602 },	\
	{ 0xf8b8, 0xbe00 },	\
	{ 0xf8ba, 0x0eba },	\
	{ 0xf8bc, 0xc602 },	\
	{ 0xf8be, 0xbe00 },	\
	{ 0xf8c0, 0x0000 },	\
	{ 0xf8c2, 0xc602 },	\
	{ 0xf8c4, 0xbe00 },	\
	{ 0xf8c6, 0x0000 },	\
	{ 0xf8c8, 0xc602 },	\
	{ 0xf8ca, 0xbe00 },	\
	{ 0xf8cc, 0x0000 },	\
	{ 0xfc26, 0x8000 },	\
	{ 0xfc2a, 0x0540 },	\
	{ 0xfc2e, 0x0e24 },	\
	{ 0xfc30, 0x0eb8 },	\
	{ 0xfc48, 0x001a }

#define RTL8125_MAC_CFG2_EPHY	\
	{ 0x0001, 0xa812 },	\
	{ 0x0009, 0x520c },	\
	{ 0x0004, 0xd000 },	\
	{ 0x000d, 0xf702 },	\
	{ 0x000a, 0x8653 },	\
	{ 0x0006, 0x001e },	\
	{ 0x0008, 0x3595 },	\
	{ 0x0020, 0x9455 },	\
	{ 0x0021, 0x99ff },	\
	{ 0x0002, 0x6046 },	\
	{ 0x0029, 0xfe00 },	\
	{ 0x0023, 0xab62 },	\
	{ 0x0041, 0xa80c },	\
	{ 0x0049, 0x520c },	\
	{ 0x0044, 0xd000 },	\
	{ 0x004d, 0xf702 },	\
	{ 0x004a, 0x8653 },	\
	{ 0x0046, 0x001e },	\
	{ 0x0048, 0x3595 },	\
	{ 0x0060, 0x9455 },	\
	{ 0x0061, 0x99ff },	\
	{ 0x0042, 0x6046 },	\
	{ 0x0069, 0xfe00 },	\
	{ 0x0063, 0xab62 }

#define RTL8125_MAC_CFG3_EPHY	\
	{ 0x0004, 0xd000 },	\
	{ 0x000a, 0x8653 },	\
	{ 0x0023, 0xab66 },	\
	{ 0x0020, 0x9455 },	\
	{ 0x0021, 0x99ff },	\
	{ 0x0029, 0xfe04 },	\
	{ 0x0044, 0xd000 },	\
	{ 0x004a, 0x8653 },	\
	{ 0x0063, 0xab66 },	\
	{ 0x0060, 0x9455 },	\
	{ 0x0061, 0x99ff },	\
	{ 0x0069, 0xfe04 }

#define RTL8125_MAC_CFG2_MCU	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8013 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8021 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8042 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0xa088 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a50 },	\
	{ 0xa438, 0x8008 },	\
	{ 0xa438, 0xd014 },	\
	{ 0xa438, 0xd1a3 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x401a },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x40c2 },	\
	{ 0xa438, 0x60a6 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f8b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a86 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a6c },	\
	{ 0xa438, 0x8080 },	\
	{ 0xa438, 0xd019 },	\
	{ 0xa438, 0xd1a2 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x401a },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x40c4 },	\
	{ 0xa438, 0x60a6 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f8b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a86 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a84 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0x8970 },	\
	{ 0xa438, 0x0c07 },	\
	{ 0xa438, 0x0901 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xcf09 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x4000 },	\
	{ 0xa438, 0xceff },	\
	{ 0xa438, 0xaf0a },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1213 },	\
	{ 0xa438, 0x8401 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x8580 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1253 },	\
	{ 0xa438, 0xd064 },	\
	{ 0xa438, 0xd181 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4018 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xc50f },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x2c59 },	\
	{ 0xa438, 0x804d },	\
	{ 0xa438, 0xc60f },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xc605 },	\
	{ 0xa438, 0xae02 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x10fd },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x10f4 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x1252 },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x1206 },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x0a78 },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x0a60 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x0a4f },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0x3f00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8066 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x807c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8089 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80a0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80b2 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80c2 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x62db },	\
	{ 0xa438, 0x655c },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0501 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0505 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0509 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x653c },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0506 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x050a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0505 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0506 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0504 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x050c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0509 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x050a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x050c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0508 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0501 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0321 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0321 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0504 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0321 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0508 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0321 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0346 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x8208 },	\
	{ 0xa438, 0x609d },	\
	{ 0xa438, 0xa50f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x001a },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x001a },	\
	{ 0xa438, 0x607d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00ab },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00ab },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x60fd },	\
	{ 0xa438, 0xa50f },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x017b },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0a05 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x017b },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x60fd },	\
	{ 0xa438, 0xa50f },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x01e0 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0a05 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x01e0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x60fd },	\
	{ 0xa438, 0xa50f },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0231 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0a05 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0231 },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0x0221 },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0x01ce },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0x0169 },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0x00a6 },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0x000d },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x0308 },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x029f },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x007f },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8017 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8029 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8054 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x805a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8064 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80a7 },	\
	{ 0xa438, 0x9430 },	\
	{ 0xa438, 0x9480 },	\
	{ 0xa438, 0xb408 },	\
	{ 0xa438, 0xd120 },	\
	{ 0xa438, 0xd057 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x064b },	\
	{ 0xa438, 0xcb80 },	\
	{ 0xa438, 0x9906 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0567 },	\
	{ 0xa438, 0xcb94 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8406 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x8dff },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e4 },	\
	{ 0xa438, 0xa840 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0773 },	\
	{ 0xa438, 0xcb91 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd139 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd140 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xb404 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07dc },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4045 },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0742 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07ec },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0742 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb6 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07dc },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x064b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa7 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0481 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x94bc },	\
	{ 0xa438, 0x870c },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x8220 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x078e },	\
	{ 0xa438, 0xcb92 },	\
	{ 0xa438, 0xa840 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd140 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd150 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x60a0 },	\
	{ 0xa438, 0x6121 },	\
	{ 0xa438, 0x61a2 },	\
	{ 0xa438, 0x6223 },	\
	{ 0xa438, 0xf02f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d10 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf00f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d20 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d30 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d40 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e4 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xa008 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4046 },	\
	{ 0xa438, 0xa002 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0742 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07f7 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0742 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb5 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e4 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x3ad4 },	\
	{ 0xa438, 0x0537 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x8840 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x064b },	\
	{ 0xa438, 0x8301 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xa70c },	\
	{ 0xa438, 0x9402 },	\
	{ 0xa438, 0x890c },	\
	{ 0xa438, 0x8840 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x064b },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0x0642 },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0x0686 },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0x0788 },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0x047b },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x065c },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0769 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0565 },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x06f9 },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8530 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x3caf },	\
	{ 0xa438, 0x8593 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x9caf },	\
	{ 0xa438, 0x85a5 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5afb },	\
	{ 0xa438, 0xe083 },	\
	{ 0xa438, 0xfb0c },	\
	{ 0xa438, 0x020d },	\
	{ 0xa438, 0x021b },	\
	{ 0xa438, 0x10bf },	\
	{ 0xa438, 0x86d7 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86da },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbe0 },	\
	{ 0xa438, 0x83fc },	\
	{ 0xa438, 0x0c02 },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1b10 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xda02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xdd02 },	\
	{ 0xa438, 0x5afb },	\
	{ 0xa438, 0xe083 },	\
	{ 0xa438, 0xfd0c },	\
	{ 0xa438, 0x020d },	\
	{ 0xa438, 0x021b },	\
	{ 0xa438, 0x10bf },	\
	{ 0xa438, 0x86dd },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86e0 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbe0 },	\
	{ 0xa438, 0x83fe },	\
	{ 0xa438, 0x0c02 },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1b10 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xe002 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xaf2f },	\
	{ 0xa438, 0xbd02 },	\
	{ 0xa438, 0x2cac },	\
	{ 0xa438, 0x0286 },	\
	{ 0xa438, 0x65af },	\
	{ 0xa438, 0x212b },	\
	{ 0xa438, 0x022c },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x86b6 },	\
	{ 0xa438, 0xaf21 },	\
	{ 0xa438, 0x0cd1 },	\
	{ 0xa438, 0x03bf },	\
	{ 0xa438, 0x8710 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x870d },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8719 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8716 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x871f },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x871c },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8728 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8725 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8707 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbad },	\
	{ 0xa438, 0x281c },	\
	{ 0xa438, 0xd100 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x0a02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x1302 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2b02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xae1a },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x0a02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x1302 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2b02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3402 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3102 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3d02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3a02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4302 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4002 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4c02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4902 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xd100 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2e02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3702 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4602 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4f02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xaf35 },	\
	{ 0xa438, 0x7ff8 },	\
	{ 0xa438, 0xfaef },	\
	{ 0xa438, 0x69bf },	\
	{ 0xa438, 0x86e3 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x86fb },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86e6 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x86fe },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86e9 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x8701 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86ec },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x8704 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86ef },	\
	{ 0xa438, 0x0262 },	\
	{ 0xa438, 0x7cbf },	\
	{ 0xa438, 0x86f2 },	\
	{ 0xa438, 0x0262 },	\
	{ 0xa438, 0x7cbf },	\
	{ 0xa438, 0x86f5 },	\
	{ 0xa438, 0x0262 },	\
	{ 0xa438, 0x7cbf },	\
	{ 0xa438, 0x86f8 },	\
	{ 0xa438, 0x0262 },	\
	{ 0xa438, 0x7cef },	\
	{ 0xa438, 0x96fe },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8fa },	\
	{ 0xa438, 0xef69 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x6273 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xf202 },	\
	{ 0xa438, 0x6273 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xf502 },	\
	{ 0xa438, 0x6273 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xf802 },	\
	{ 0xa438, 0x6273 },	\
	{ 0xa438, 0xef96 },	\
	{ 0xa438, 0xfefc },	\
	{ 0xa438, 0x0420 },	\
	{ 0xa438, 0xb540 },	\
	{ 0xa438, 0x53b5 },	\
	{ 0xa438, 0x4086 },	\
	{ 0xa438, 0xb540 },	\
	{ 0xa438, 0xb9b5 },	\
	{ 0xa438, 0x40c8 },	\
	{ 0xa438, 0xb03a },	\
	{ 0xa438, 0xc8b0 },	\
	{ 0xa438, 0xbac8 },	\
	{ 0xa438, 0xb13a },	\
	{ 0xa438, 0xc8b1 },	\
	{ 0xa438, 0xba77 },	\
	{ 0xa438, 0xbd26 },	\
	{ 0xa438, 0xffbd },	\
	{ 0xa438, 0x2677 },	\
	{ 0xa438, 0xbd28 },	\
	{ 0xa438, 0xffbd },	\
	{ 0xa438, 0x2840 },	\
	{ 0xa438, 0xbd26 },	\
	{ 0xa438, 0xc8bd },	\
	{ 0xa438, 0x2640 },	\
	{ 0xa438, 0xbd28 },	\
	{ 0xa438, 0xc8bd },	\
	{ 0xa438, 0x28bb },	\
	{ 0xa438, 0xa430 },	\
	{ 0xa438, 0x98b0 },	\
	{ 0xa438, 0x1eba },	\
	{ 0xa438, 0xb01e },	\
	{ 0xa438, 0xdcb0 },	\
	{ 0xa438, 0x1e98 },	\
	{ 0xa438, 0xb09e },	\
	{ 0xa438, 0xbab0 },	\
	{ 0xa438, 0x9edc },	\
	{ 0xa438, 0xb09e },	\
	{ 0xa438, 0x98b1 },	\
	{ 0xa438, 0x1eba },	\
	{ 0xa438, 0xb11e },	\
	{ 0xa438, 0xdcb1 },	\
	{ 0xa438, 0x1e98 },	\
	{ 0xa438, 0xb19e },	\
	{ 0xa438, 0xbab1 },	\
	{ 0xa438, 0x9edc },	\
	{ 0xa438, 0xb19e },	\
	{ 0xa438, 0x11b0 },	\
	{ 0xa438, 0x1e22 },	\
	{ 0xa438, 0xb01e },	\
	{ 0xa438, 0x33b0 },	\
	{ 0xa438, 0x1e11 },	\
	{ 0xa438, 0xb09e },	\
	{ 0xa438, 0x22b0 },	\
	{ 0xa438, 0x9e33 },	\
	{ 0xa438, 0xb09e },	\
	{ 0xa438, 0x11b1 },	\
	{ 0xa438, 0x1e22 },	\
	{ 0xa438, 0xb11e },	\
	{ 0xa438, 0x33b1 },	\
	{ 0xa438, 0x1e11 },	\
	{ 0xa438, 0xb19e },	\
	{ 0xa438, 0x22b1 },	\
	{ 0xa438, 0x9e33 },	\
	{ 0xa438, 0xb19e },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x2f71 },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0x20d9 },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0x2109 },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0x34e7 },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0x000f }

#define RTL8125_MAC_CFG3_MCU	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8093 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8097 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x809d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80a1 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80aa },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x607b },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0x42da },	\
	{ 0xa438, 0xf01e },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x615b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf024 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf02c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf034 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac11 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4779 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf034 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac22 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa420 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4559 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf023 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac44 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa440 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4339 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf012 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac88 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa480 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4119 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0xc48f },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x141b },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x121a },	\
	{ 0xa438, 0xd0b4 },	\
	{ 0xa438, 0xd1bb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0898 },	\
	{ 0xa438, 0xd0b4 },	\
	{ 0xa438, 0xd1bb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a0e },	\
	{ 0xa438, 0xd064 },	\
	{ 0xa438, 0xd18a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0b7e },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xa804 },	\
	{ 0xa438, 0x8804 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x053b },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0648 },	\
	{ 0xa438, 0xc520 },	\
	{ 0xa438, 0xa201 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x252d },	\
	{ 0xa438, 0x1646 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x4006 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1646 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0308 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0x0307 },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0x1645 },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x0647 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x053a },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x0b7c },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x0a0c },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x0896 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x11a1 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0xff00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8015 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x02d7 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00ed },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0509 },	\
	{ 0xa438, 0xc100 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x008f },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x008d },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x00eb },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8018 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8024 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8055 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8072 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80dc },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xfffd },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xfffd },	\
	{ 0xa438, 0x8301 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xa70c },	\
	{ 0xa438, 0x9402 },	\
	{ 0xa438, 0x890c },	\
	{ 0xa438, 0x8840 },	\
	{ 0xa438, 0xa380 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xcb91 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd139 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd140 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xb404 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e0 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4085 },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07f0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb6 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xd158 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x03d4 },	\
	{ 0xa438, 0x94bc },	\
	{ 0xa438, 0x870c },	\
	{ 0xa438, 0x8380 },	\
	{ 0xa438, 0xd10d },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xa220 },	\
	{ 0xa438, 0xd130 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xbb80 },	\
	{ 0xa438, 0xd1c4 },	\
	{ 0xa438, 0xd074 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x604b },	\
	{ 0xa438, 0xa90c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0556 },	\
	{ 0xa438, 0xcb92 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd116 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd119 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x60a0 },	\
	{ 0xa438, 0x6241 },	\
	{ 0xa438, 0x63e2 },	\
	{ 0xa438, 0x6583 },	\
	{ 0xa438, 0xf054 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d10 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf02f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d50 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf02a },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d20 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf021 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d60 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d30 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf013 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d70 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d40 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d80 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e8 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5ff4 },	\
	{ 0xa438, 0xa008 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4046 },	\
	{ 0xa438, 0xa002 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07fb },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x7f6f },	\
	{ 0xa438, 0x7f4e },	\
	{ 0xa438, 0x7f2d },	\
	{ 0xa438, 0x7f0c },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e8 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb5 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x3ad4 },	\
	{ 0xa438, 0x0556 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xd1f5 },	\
	{ 0xa438, 0xd049 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x01ec },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0x01ea },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0x06a9 },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0x078a },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0x03d2 },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x067f },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0665 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x00fc },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8530 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x3caf },	\
	{ 0xa438, 0x8545 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x45af },	\
	{ 0xa438, 0x8545 },	\
	{ 0xa438, 0xee82 },	\
	{ 0xa438, 0xf900 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa438, 0xaf03 },	\
	{ 0xa438, 0xb7f8 },	\
	{ 0xa438, 0xe0a6 },	\
	{ 0xa438, 0x00e1 },	\
	{ 0xa438, 0xa601 },	\
	{ 0xa438, 0xef01 },	\
	{ 0xa438, 0x58f0 },	\
	{ 0xa438, 0xa080 },	\
	{ 0xa438, 0x37a1 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0xae16 },	\
	{ 0xa438, 0xa185 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x11a1 },	\
	{ 0xa438, 0x8702 },	\
	{ 0xa438, 0xae0c },	\
	{ 0xa438, 0xa188 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x07a1 },	\
	{ 0xa438, 0x8902 },	\
	{ 0xa438, 0xae02 },	\
	{ 0xa438, 0xae1c },	\
	{ 0xa438, 0xe0b4 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0x6901 },	\
	{ 0xa438, 0xe4b4 },	\
	{ 0xa438, 0x62e5 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0xe0b4 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0x6901 },	\
	{ 0xa438, 0xe4b4 },	\
	{ 0xa438, 0x62e5 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x03b3 },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0x0001 }
