/*
 * Decompiled with CFR 0.152.
 */
package ghidra.program.database.mem;

import db.DBBuffer;
import db.Record;
import ghidra.framework.store.LockException;
import ghidra.program.database.map.AddressMapDB;
import ghidra.program.database.mem.MemoryBlockInputStream;
import ghidra.program.database.mem.MemoryMapDB;
import ghidra.program.database.mem.MemoryMapDBAdapter;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.address.SegmentedAddress;
import ghidra.program.model.mem.MemoryAccessException;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.mem.MemoryBlockType;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.DuplicateNameException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ConcurrentModificationException;

class MemoryBlockDB
implements MemoryBlock {
    private static final MemoryAccessException UNITIALIZED_EXCEPTION = new MemoryAccessException("Cannot access uninitialized memory!");
    private MemoryMapDBAdapter adapter;
    protected Record record;
    protected MemoryMapDB memMap;
    protected AddressMapDB addrMap;
    private int id;
    protected Address startAddress;
    protected long length;
    private volatile boolean invalid;
    private DBBuffer buf;
    protected MemoryBlockType blockType;
    private boolean isInitialized;

    MemoryBlockDB(MemoryMapDBAdapter adapter, Record record, DBBuffer buf, MemoryMapDB memMap) throws IOException {
        this.adapter = adapter;
        this.memMap = memMap;
        this.addrMap = memMap.getAddressMap();
        this.buf = buf;
        this.id = (int)record.getKey();
        this.refresh(record);
    }

    void refresh(Record lRecord) throws IOException {
        this.record = lRecord;
        if ((long)this.id != lRecord.getKey()) {
            throw new AssertException("Incorrect block record");
        }
        this.startAddress = this.addrMap.decodeAddress(lRecord.getLongValue(4), false);
        if (this.startAddress instanceof SegmentedAddress) {
            SegmentedAddress imageBase = (SegmentedAddress)this.addrMap.getImageBase();
            int baseSegment = imageBase.getSegment();
            int segment = lRecord.getIntValue(9);
            this.startAddress = ((SegmentedAddress)this.startAddress).normalize(segment + baseSegment);
        }
        short dbBlockType = lRecord.getShortValue(5);
        this.blockType = this.getBlockType(dbBlockType);
        this.isInitialized = dbBlockType == 0;
        this.length = lRecord.getLongValue(7);
        int bufferID = lRecord.getIntValue(8);
        this.buf = this.adapter.getBuffer(bufferID);
    }

    private MemoryBlockType getBlockType(int dbType) {
        switch (dbType) {
            case 0: 
            case 1: {
                return this.startAddress.getAddressSpace().isOverlaySpace() ? MemoryBlockType.OVERLAY : MemoryBlockType.DEFAULT;
            }
            case 2: {
                return MemoryBlockType.BIT_MAPPED;
            }
            case 4: {
                return MemoryBlockType.BYTE_MAPPED;
            }
        }
        return MemoryBlockType.DEFAULT;
    }

    @Override
    public InputStream getData() {
        return new MemoryBlockInputStream(this);
    }

    int getID() {
        return this.id;
    }

    @Override
    public boolean contains(Address addr) {
        if (addr.hasSameAddressSpace(this.startAddress)) {
            long offset = addr.subtract(this.startAddress);
            return offset >= 0L && offset < this.length;
        }
        return false;
    }

    @Override
    public Address getStart() {
        return this.startAddress;
    }

    @Override
    public Address getEnd() {
        return this.startAddress.add(this.length - 1L);
    }

    @Override
    public long getSize() {
        return this.length;
    }

    @Override
    public String getName() {
        return this.record.getString(0);
    }

    public String toString() {
        return this.getName() + "[" + this.getStart() + ":" + this.getEnd() + "]";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void setName(String name) throws DuplicateNameException, LockException {
        this.memMap.lock.acquire();
        String oldName = this.getName();
        try {
            this.checkValid();
            try {
                if (this.getStart().getAddressSpace().isOverlaySpace()) {
                    this.memMap.overlayBlockRenamed(oldName, name);
                }
                this.record.setString(0, name);
                this.adapter.updateBlockRecord(this.record);
            }
            catch (IOException e) {
                this.memMap.dbError(e);
            }
            this.memMap.fireBlockChanged(this);
        }
        finally {
            this.memMap.lock.release();
        }
    }

    @Override
    public String getComment() {
        return this.record.getString(1);
    }

    @Override
    public void setComment(String comment) {
        this.memMap.lock.acquire();
        try {
            this.checkValid();
            try {
                this.record.setString(1, comment);
                this.adapter.updateBlockRecord(this.record);
                this.memMap.fireBlockChanged(this);
            }
            catch (IOException e) {
                this.memMap.dbError(e);
            }
        }
        finally {
            this.memMap.lock.release();
        }
    }

    @Override
    public boolean isRead() {
        return (this.record.getByteValue(3) & 4) != 0;
    }

    @Override
    public int getPermissions() {
        return this.record.getByteValue(3);
    }

    @Override
    public boolean isInitialized() {
        return this.isInitialized || this.memMap.getLiveMemoryHandler() != null;
    }

    private void setPermissionBit(int permBitMask, boolean enable) {
        byte p = this.record.getByteValue(3);
        p = enable ? (byte)(p | permBitMask) : (byte)(p & ~permBitMask);
        this.record.setByteValue(3, p);
        try {
            this.adapter.updateBlockRecord(this.record);
        }
        catch (IOException e) {
            this.memMap.dbError(e);
        }
    }

    @Override
    public void setRead(boolean r) {
        this.memMap.lock.acquire();
        try {
            this.checkValid();
            this.setPermissionBit(4, r);
            this.memMap.fireBlockChanged(this);
        }
        finally {
            this.memMap.lock.release();
        }
    }

    @Override
    public boolean isWrite() {
        return (this.record.getByteValue(3) & 2) != 0;
    }

    @Override
    public void setWrite(boolean w) {
        this.memMap.lock.acquire();
        try {
            this.checkValid();
            this.setPermissionBit(2, w);
            this.memMap.fireBlockChanged(this);
        }
        finally {
            this.memMap.lock.release();
        }
    }

    @Override
    public boolean isExecute() {
        return (this.record.getByteValue(3) & 1) != 0;
    }

    @Override
    public void setExecute(boolean x) {
        this.memMap.lock.acquire();
        try {
            this.checkValid();
            this.setPermissionBit(1, x);
            this.memMap.fireBlockChanged(this);
        }
        finally {
            this.memMap.lock.release();
        }
    }

    @Override
    public boolean isVolatile() {
        return (this.record.getByteValue(3) & 8) != 0;
    }

    @Override
    public void setVolatile(boolean v) {
        this.memMap.lock.acquire();
        try {
            this.checkValid();
            this.setPermissionBit(8, v);
            this.memMap.fireBlockChanged(this);
        }
        finally {
            this.memMap.lock.release();
        }
    }

    @Override
    public String getSourceName() {
        return this.record.getString(2);
    }

    @Override
    public void setSourceName(String sourceName) {
        this.memMap.lock.acquire();
        try {
            this.checkValid();
            try {
                this.record.setString(2, sourceName);
                this.adapter.updateBlockRecord(this.record);
            }
            catch (IOException e) {
                this.memMap.dbError(e);
            }
            this.memMap.fireBlockChanged(this);
        }
        finally {
            this.memMap.lock.release();
        }
    }

    protected long getBlockOffset(Address addr) throws MemoryAccessException {
        if (!addr.hasSameAddressSpace(this.startAddress)) {
            throw new MemoryAccessException("Address not contained in block: " + addr);
        }
        long offset = addr.subtract(this.startAddress);
        if (offset < 0L || offset >= this.length) {
            throw new MemoryAccessException("Address not contained in block: " + addr);
        }
        return offset;
    }

    @Override
    public byte getByte(Address addr) throws MemoryAccessException {
        long offset = this.getBlockOffset(addr);
        if (this.memMap.getLiveMemoryHandler() != null) {
            return this.memMap.getByte(addr);
        }
        this.checkBlockType();
        try {
            return this.getByte(offset);
        }
        catch (IOException e) {
            this.memMap.dbError(e);
            return 0;
        }
    }

    byte getByte(long offset) throws IOException {
        this.checkValid();
        try {
            return this.buf.getByte((int)offset);
        }
        catch (Exception e) {
            this.checkValid();
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            throw new RuntimeException("Unexpected Error", e);
        }
    }

    @Override
    public int getBytes(Address addr, byte[] b) throws MemoryAccessException {
        return this.getBytes(addr, b, 0, b.length);
    }

    @Override
    public int getBytes(Address addr, byte[] b, int off, int size) throws MemoryAccessException {
        long offset = this.getBlockOffset(addr);
        int len = (int)Math.min((long)size, this.length - offset);
        if (this.memMap.getLiveMemoryHandler() != null) {
            return this.memMap.getBytes(addr, b, off, len);
        }
        this.checkValid();
        this.checkBlockType();
        try {
            this.buf.get((int)offset, b, off, len);
        }
        catch (Exception e) {
            this.checkValid();
            if (e instanceof IOException) {
                this.memMap.dbError((IOException)e);
            }
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            throw new RuntimeException("Unexpected Error", e);
        }
        return len;
    }

    int getBytes(long offset, byte[] b, int off, int size) throws IOException {
        int len = (int)Math.min((long)size, this.length - offset);
        this.checkValid();
        try {
            this.buf.get((int)offset, b, off, len);
        }
        catch (Exception e) {
            this.checkValid();
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            throw new RuntimeException("Unexpected Error", e);
        }
        return len;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void putByte(Address addr, byte b) throws MemoryAccessException {
        long offset = this.getBlockOffset(addr);
        if (this.memMap.getLiveMemoryHandler() != null) {
            this.memMap.setByte(addr, b);
            return;
        }
        this.memMap.lock.acquire();
        try {
            this.checkValid();
            this.checkBlockType();
            this.memMap.checkMemoryWrite(addr);
            try {
                this.buf.putByte((int)offset, b);
            }
            catch (IOException e) {
                this.memMap.dbError(e);
            }
            this.memMap.fireBytesChanged(addr, 1);
        }
        finally {
            this.memMap.lock.release();
        }
    }

    @Override
    public int putBytes(Address addr, byte[] b) throws MemoryAccessException {
        return this.putBytes(addr, b, 0, b.length);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int putBytes(Address addr, byte[] b, int off, int size) throws MemoryAccessException {
        int len = 0;
        long offset = this.getBlockOffset(addr);
        len = (int)Math.min((long)size, this.length - offset);
        if (len > 0) {
            if (this.memMap.getLiveMemoryHandler() != null) {
                this.memMap.setBytes(addr, b, off, len);
                return len;
            }
            this.memMap.lock.acquire();
            try {
                this.checkValid();
                this.checkBlockType();
                this.memMap.checkMemoryWrite(addr, len);
                try {
                    this.buf.put((int)offset, b, off, len);
                }
                catch (IOException e) {
                    this.memMap.dbError(e);
                }
                this.memMap.fireBytesChanged(addr, len);
            }
            finally {
                this.memMap.lock.release();
            }
        }
        return len;
    }

    @Override
    public MemoryBlockType getType() {
        return this.blockType;
    }

    void checkValid() {
        if (this.invalid) {
            throw new ConcurrentModificationException();
        }
    }

    void invalidate() {
        this.invalid = true;
    }

    @Override
    public int compareTo(MemoryBlock block) {
        MemoryBlockDB blockDB = (MemoryBlockDB)block;
        return this.startAddress.compareTo(blockDB.startAddress);
    }

    void setStartAddress(Address newStartAddr) throws AddressOverflowException, IOException {
        this.startAddress = newStartAddr;
        long addr = this.addrMap.getKey(newStartAddr, true);
        Address endAddr = newStartAddr.addNoWrap(this.length - 1L);
        this.addrMap.getKey(endAddr, true);
        this.record.setLongValue(4, addr);
        if (newStartAddr instanceof SegmentedAddress) {
            SegmentedAddress imageBase = (SegmentedAddress)this.addrMap.getImageBase();
            int baseSegment = imageBase.getSegment();
            int segment = ((SegmentedAddress)this.startAddress).getSegment();
            this.record.setIntValue(9, segment - baseSegment);
        }
        this.adapter.updateBlockRecord(this.record);
    }

    void join(MemoryBlockDB memBlock2) throws IOException {
        this.length += memBlock2.length;
        this.record.setLongValue(7, this.length);
        if (this.buf != null) {
            this.buf.append(memBlock2.buf);
            memBlock2.buf = null;
        }
        this.adapter.updateBlockRecord(this.record);
        this.adapter.deleteMemoryBlock(memBlock2);
    }

    MemoryBlockDB split(Address addr) throws IOException {
        MemoryBlockDB newBlock;
        long offset = addr.subtract(this.startAddress);
        long newLength = this.length - offset;
        this.length = offset;
        this.record.setLongValue(7, this.length);
        this.adapter.updateBlockRecord(this.record);
        DBBuffer newBuf = null;
        try {
            if (this.buf != null) {
                newBuf = this.buf.split((int)offset);
                newBlock = this.adapter.createInitializedBlock(this.getName() + ".split", addr, newBuf, this.getPermissions());
            } else {
                newBlock = this.adapter.createBlock(this.getType(), this.getName() + ".split", addr, newLength, this.getOverlayAddress(offset), this.isInitialized(), this.getPermissions());
            }
        }
        catch (AddressOverflowException e) {
            throw new AssertException((Throwable)((Object)e));
        }
        return newBlock;
    }

    Address getOverlayAddress(long offset) {
        return null;
    }

    void initializeBlock(byte initialValue) throws IOException {
        if (this.length > Integer.MAX_VALUE) {
            throw new AssertException();
        }
        this.buf = this.adapter.createBuffer((int)this.length, initialValue);
        this.record.setIntValue(8, this.buf.getId());
        this.record.setShortValue(5, (short)0);
        this.isInitialized = true;
        this.adapter.updateBlockRecord(this.record);
    }

    void uninitializeBlock() throws IOException {
        this.buf.delete();
        this.buf = null;
        this.record.setIntValue(8, -1);
        this.record.setShortValue(5, (short)1);
        this.isInitialized = false;
        this.adapter.updateBlockRecord(this.record);
    }

    private void checkBlockType() throws MemoryAccessException {
        if (!this.isInitialized) {
            throw UNITIALIZED_EXCEPTION;
        }
    }

    DBBuffer getBuffer() {
        return this.buf;
    }

    void delete() throws IOException {
        if (this.buf != null) {
            this.buf.delete();
        }
        this.adapter.deleteMemoryBlock(this);
    }

    @Override
    public boolean isMapped() {
        return false;
    }

    @Override
    public boolean isLoaded() {
        return this.startAddress.getAddressSpace().isLoadedMemorySpace();
    }
}

