/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.disassembler;

import docking.ActionContext;
import docking.DialogComponentProvider;
import docking.action.DockingAction;
import docking.action.DockingActionIf;
import docking.action.MenuData;
import docking.widgets.table.GTable;
import docking.widgets.table.threaded.ThreadedTableModelListener;
import ghidra.app.cmd.disassemble.DisassembleCommand;
import ghidra.app.plugin.ProgramPlugin;
import ghidra.app.plugin.core.disassembler.AddressTable;
import ghidra.app.plugin.core.disassembler.AddressTableDialog;
import ghidra.app.plugin.core.disassembler.AutoTableDisassemblerModel;
import ghidra.app.services.GoToService;
import ghidra.app.util.PseudoDisassembler;
import ghidra.framework.cmd.BackgroundCommand;
import ghidra.framework.cmd.CompoundBackgroundCommand;
import ghidra.framework.model.DomainObjectChangedEvent;
import ghidra.framework.model.DomainObjectException;
import ghidra.framework.model.DomainObjectListener;
import ghidra.framework.model.UndoableDomainObject;
import ghidra.framework.plugintool.PluginInfo;
import ghidra.framework.plugintool.PluginTool;
import ghidra.framework.plugintool.ServiceProvider;
import ghidra.framework.plugintool.util.PluginStatus;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.lang.RegisterValue;
import ghidra.program.model.listing.Listing;
import ghidra.program.model.listing.Program;
import ghidra.program.util.ProgramSelection;
import ghidra.util.HelpLocation;
import ghidra.util.Msg;
import ghidra.util.SystemUtilities;
import ghidra.util.task.Task;
import ghidra.util.task.TaskMonitor;
import java.awt.Component;

@PluginInfo(status=PluginStatus.RELEASED, packageName="Ghidra Core", category="Search", shortDescription="Search for Address Tables", description="This plugin identifies possible 32-bit address tables and allows the user to disassemble starting at address references in the table. The search is done on the entire program or a selection.", servicesRequired={GoToService.class})
public class AutoTableDisassemblerPlugin
extends ProgramPlugin
implements DomainObjectListener {
    private AddressTableDialog addressTableDialog;
    private AutoTableDisassemblerModel model;
    private boolean automaticLabel;
    private int offsetLen;
    private DockingAction findTableAction;
    static final String SEARCH_ACTION_NAME = "Search for Address Tables";

    public AutoTableDisassemblerPlugin(PluginTool tool) {
        super(tool, true, true);
    }

    protected void init() {
        this.createActions();
    }

    Program getProgram() {
        return this.currentProgram;
    }

    private void startDialog() {
        int size = this.currentProgram.getAddressFactory().getDefaultAddressSpace().getSize();
        if (size != 32 && size != 64 && size != 24) {
            Msg.showWarn(((Object)((Object)this)).getClass(), null, (String)"Search For Address Tables", (Object)("Cannot search for Address tables on " + size + "-bit memory!"));
            return;
        }
        if (this.addressTableDialog == null) {
            this.model = new AutoTableDisassemblerModel((ServiceProvider)this.tool, this);
            this.addressTableDialog = new AddressTableDialog(this);
            this.addressTableDialog.setHasSelection(this.currentSelection != null);
        }
        if (this.addressTableDialog.isVisible()) {
            this.addressTableDialog.toFront();
        } else {
            this.tool.showDialog((DialogComponentProvider)this.addressTableDialog);
        }
    }

    void dialogDismissed() {
        this.addressTableDialog = null;
        if (this.model != null) {
            this.model.dispose();
            this.model = null;
        }
    }

    int getMinimumTableSize() {
        return this.addressTableDialog.getMinTableSize();
    }

    int getAlignment() {
        return this.addressTableDialog.getAlignment();
    }

    int getSkipLength() {
        return this.addressTableDialog.getSkipLength();
    }

    boolean isShiftAddresses() {
        return this.addressTableDialog.getShiftedAddresses();
    }

    AddressSetView getSelection() {
        if (!this.addressTableDialog.isSearchSelection()) {
            return this.currentProgram.getMemory();
        }
        return this.currentSelection;
    }

    private void createActions() {
        this.findTableAction = new DockingAction(SEARCH_ACTION_NAME, this.getName()){

            public void actionPerformed(ActionContext context) {
                AutoTableDisassemblerPlugin.this.startDialog();
            }
        };
        this.findTableAction.setHelpLocation(new HelpLocation("Search", this.findTableAction.getName()));
        this.findTableAction.setMenuBarData(new MenuData(new String[]{"&Search", "For Address Tables..."}, null, "search for"));
        this.findTableAction.setDescription(this.getPluginDescription().getDescription());
        this.enableOnLocation(this.findTableAction);
        this.tool.addAction((DockingActionIf)this.findTableAction);
    }

    void findTablesInSet(GTable table, boolean searchSelection) {
        if (searchSelection && (this.currentSelection == null || this.currentSelection.isEmpty())) {
            this.addressTableDialog.setDialogText("Please make a selection to search.");
            return;
        }
        int minimumTableSize = this.addressTableDialog.getMinTableSize();
        if (minimumTableSize < 2) {
            this.addressTableDialog.setDialogText("Please enter a valid minimum search length. Must be >= 2");
            return;
        }
        int alignment = this.addressTableDialog.getAlignment();
        if (alignment <= 0 || alignment > 8) {
            this.addressTableDialog.setDialogText("Please enter a valid alignment value. Must be > 0 and <= 8");
            return;
        }
        this.addressTableDialog.enableSearchButton(false);
        this.model.addInitialLoadListener(new ThreadedTableModelListener(){

            public void loadPending() {
            }

            public void loadingStarted() {
            }

            public void loadingFinished(boolean wasCancelled) {
                if (AutoTableDisassemblerPlugin.this.addressTableDialog != null) {
                    AutoTableDisassemblerPlugin.this.addressTableDialog.searchComplete(wasCancelled);
                }
            }
        });
        this.model.reload();
    }

    void disassembleTable(int[] selectedRows) {
        if (this.currentProgram == null || this.currentLocation == null) {
            return;
        }
        Address[] selectedAddresses = new Address[selectedRows.length];
        for (int i = 0; i < selectedRows.length; ++i) {
            selectedAddresses[i] = this.model.getAddress(selectedRows[i]);
        }
        CompoundBackgroundCommand backCmd = new CompoundBackgroundCommand("Disassemble Address Tables", false, true);
        this.offsetLen = this.addressTableDialog.getOffset();
        for (int i = 0; i < selectedRows.length; ++i) {
            Address currentAddress = selectedAddresses[i];
            if (!this.model.containsKey(currentAddress)) continue;
            this.createDisassemblyCommandsForAddress(backCmd, currentAddress);
        }
        if (backCmd.isEmpty()) {
            Msg.showError((Object)((Object)this), (Component)this.addressTableDialog.getComponent(), (String)"Disassemble Address Tables Failed", (Object)"No undefined/aligned code units were found");
            return;
        }
        this.tool.executeBackgroundCommand((BackgroundCommand)backCmd, (UndoableDomainObject)this.currentProgram);
    }

    private void createDisassemblyCommandsForAddress(CompoundBackgroundCommand backCmd, Address currentAddress) {
        Listing listing = this.currentProgram.getListing();
        int align = this.currentProgram.getLanguage().getInstructionAlignment();
        AddressTable addrTable = this.model.get(currentAddress);
        Address[] elements = addrTable.getTableElements();
        for (int i = this.offsetLen; i < elements.length; ++i) {
            Address addr = elements[i];
            Address targetAddr = PseudoDisassembler.getNormalizedDisassemblyAddress((Program)this.currentProgram, (Address)addr);
            if (targetAddr.getOffset() % (long)align != 0L || listing.getUndefinedDataAt(targetAddr) == null) continue;
            DisassembleCommand disassembleCmd = new DisassembleCommand(addr, null, true);
            RegisterValue rval = PseudoDisassembler.getTargetContextRegisterValueForDisassembly((Program)this.currentProgram, (Address)addr);
            disassembleCmd.setInitialContext(rval);
            backCmd.add((BackgroundCommand)disassembleCmd);
        }
    }

    void makeTable(int[] selectedRows) {
        if (this.currentProgram == null) {
            return;
        }
        if (!this.validateSelectionOffsets()) {
            return;
        }
        SystemUtilities.assertTrue((selectedRows.length > 0 ? 1 : 0) != 0, (String)"Cannot make address tables when the find dialog's table contains no selection");
        this.automaticLabel = this.addressTableDialog.getAutomaticLabel();
        Address[] selectedAddresses = new Address[selectedRows.length];
        for (int i = 0; i < selectedRows.length; ++i) {
            selectedAddresses[i] = this.model.getAddress(selectedRows[i]);
        }
        MakeTablesTask task = new MakeTablesTask(selectedAddresses);
        this.addressTableDialog.executeProgressTask(task, 500);
    }

    private int makeTables(Address[] addresses, TaskMonitor monitor) {
        monitor.initialize((long)addresses.length);
        monitor.setMessage("Make Address Tables...");
        int collisionCount = 0;
        for (int i = 0; i < addresses.length && !monitor.isCancelled(); ++i) {
            Address currentAddress = addresses[i];
            AddressTable addrTable = this.model.get(currentAddress);
            if (addrTable == null) continue;
            monitor.setProgress((long)i);
            boolean madeTable = addrTable.makeTable(this.currentProgram, this.offsetLen, addrTable.getNumberAddressEntries(), this.automaticLabel);
            if (madeTable) continue;
            ++collisionCount;
        }
        return collisionCount;
    }

    @Override
    protected void programDeactivated(Program program) {
        program.removeListener((DomainObjectListener)this);
        if (this.addressTableDialog != null) {
            this.addressTableDialog.close();
        }
    }

    @Override
    protected void programActivated(Program program) {
        program.addListener((DomainObjectListener)this);
    }

    public void domainObjectChanged(DomainObjectChangedEvent ev) {
        if (this.model == null) {
            return;
        }
        if (this.addressTableDialog.isRunningTask()) {
            return;
        }
        this.refreshModel();
    }

    private void refreshModel() {
        if (this.addressTableDialog == null) {
            return;
        }
        this.model.refresh();
    }

    @Override
    protected void selectionChanged(ProgramSelection sel) {
        if (this.addressTableDialog != null) {
            this.addressTableDialog.setHasSelection(sel != null && !sel.isEmpty());
        }
    }

    void updateOffsetString(int[] selectedRows) {
        if (this.addressTableDialog == null) {
            return;
        }
        if (selectedRows.length <= 0) {
            this.addressTableDialog.clearMakeTableOptions();
            return;
        }
        if (!this.validateSelectionOffsets()) {
            return;
        }
        this.addressTableDialog.setEnableMakeTableOptions(true);
        String offsetText = " ";
        int offset = this.addressTableDialog.getOffset();
        if (selectedRows.length == 1) {
            Address addr = this.model.getAddress(selectedRows[0]);
            Address addressWithOffset = addr.addWrap((long)(offset * 4));
            offsetText = addressWithOffset.toString();
        }
        this.offsetLen = offset;
        this.addressTableDialog.setDialogText("");
        this.addressTableDialog.setOffsetText(offsetText);
    }

    boolean validateSelectionOffsets() {
        int[] selectedRows = this.addressTableDialog.getSelectedRows();
        int shortestAddressTableRow = this.getIndexOfShortestAddressTable(selectedRows);
        Address addr = this.model.getAddress(selectedRows[shortestAddressTableRow]);
        int offset = this.addressTableDialog.getOffset();
        AddressTable addrTable = this.model.get(addr);
        int len = 0;
        if (addrTable != null) {
            len = addrTable.getTableElements().length;
        }
        if (offset >= 0 && offset < len) {
            return true;
        }
        this.addressTableDialog.setEnableMakeTableButtons(false);
        String dialogText = "Invalid offset length - check table length, must be >= 0 and < " + len;
        this.addressTableDialog.setDialogText(dialogText);
        this.addressTableDialog.setOffsetText("");
        this.offsetLen = 0;
        return false;
    }

    private int getIndexOfShortestAddressTable(int[] selectedRows) {
        if (selectedRows.length < 0) {
            throw new AssertionError((Object)"There must be rows selected in order to process multiple rows.");
        }
        int shortestRowIndex = selectedRows[0];
        int shortestLength = Integer.MAX_VALUE;
        for (int i = 0; i < selectedRows.length; ++i) {
            int row = selectedRows[i];
            int length = this.model.getTableLength(row);
            if (length >= shortestLength) continue;
            shortestLength = length;
            shortestRowIndex = i;
        }
        return shortestRowIndex;
    }

    AutoTableDisassemblerModel getModel() {
        return this.model;
    }

    private class MakeTablesTask
    extends Task {
        private Address[] addresses;
        private int collisionCount;
        private String collisionMessage;

        private MakeTablesTask(Address[] addresses) {
            super("Make Tables", true, true, false);
            this.collisionMessage = "Address Table(s) could not be created due to collisions with existing\ndata. Check the Address Table for those tables not created.";
            this.addresses = addresses;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run(TaskMonitor monitor) {
            Program program = AutoTableDisassemblerPlugin.this.currentProgram;
            int transactionID = program.startTransaction("Make Address Table");
            boolean commit = false;
            try {
                this.collisionCount = AutoTableDisassemblerPlugin.this.makeTables(this.addresses, monitor);
                commit = true;
            }
            catch (Exception e) {
                if (!(e instanceof DomainObjectException)) {
                    Msg.showError((Object)((Object)this), null, null, null, (Throwable)e);
                }
            }
            finally {
                program.endTransaction(transactionID, commit);
            }
            SystemUtilities.runSwingLater(() -> {
                AddressTableDialog tempDialog = AutoTableDisassemblerPlugin.this.addressTableDialog;
                if (tempDialog == null) {
                    return;
                }
                tempDialog.makeTablesCompleted();
                if (this.collisionCount > 0) {
                    Msg.showWarn(((Object)((Object)this)).getClass(), (Component)tempDialog.getComponent(), (String)"Collisions while Making Tables", (Object)this.collisionMessage);
                }
                AutoTableDisassemblerPlugin.this.refreshModel();
            });
        }
    }
}

