/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { EuiScreenReaderOnly } from '@elastic/eui';
import { i18n } from '@kbn/i18n';
import React, { useEffect, useRef } from 'react';
import { expandLiteralStrings } from '../../../../../../../es_ui_shared/public';
import { useEditorReadContext, useRequestReadContext, useServicesContext } from '../../../../contexts';
import { createReadOnlyAceEditor } from '../../../../models/legacy_core_editor';
import { subscribeResizeChecker } from '../subscribe_console_resize_checker';
import { applyCurrentSettings } from './apply_editor_settings';

function modeForContentType(contentType) {
  if (!contentType) {
    return 'ace/mode/text';
  }

  if (contentType.indexOf('application/json') >= 0) {
    return 'ace/mode/json';
  } else if (contentType.indexOf('application/yaml') >= 0) {
    return 'ace/mode/yaml';
  }

  return 'ace/mode/text';
}

function EditorOutputUI() {
  var editorRef = useRef(null);
  var editorInstanceRef = useRef(null);

  var _useServicesContext = useServicesContext(),
      services = _useServicesContext.services;

  var _useEditorReadContext = useEditorReadContext(),
      readOnlySettings = _useEditorReadContext.settings;

  var _useRequestReadContex = useRequestReadContext(),
      _useRequestReadContex2 = _useRequestReadContex.lastResult,
      data = _useRequestReadContex2.data,
      error = _useRequestReadContex2.error;

  var inputId = 'ConAppOutputTextarea';
  useEffect(function () {
    editorInstanceRef.current = createReadOnlyAceEditor(editorRef.current);
    var unsubscribe = subscribeResizeChecker(editorRef.current, editorInstanceRef.current);
    var textarea = editorRef.current.querySelector('textarea');
    textarea.setAttribute('id', inputId);
    textarea.setAttribute('readonly', 'true');
    return function () {
      unsubscribe();
      editorInstanceRef.current.destroy();
    };
  }, [services.settings]);
  useEffect(function () {
    var editor = editorInstanceRef.current;

    if (data) {
      var mode = modeForContentType(data[0].response.contentType);
      editor.session.setMode(mode);
      editor.update(data.map(function (d) {
        return d.response.value;
      }).map(readOnlySettings.tripleQuotes ? expandLiteralStrings : function (a) {
        return a;
      }).join('\n'));
    } else if (error) {
      editor.session.setMode(modeForContentType(error.response.contentType));
      editor.update(error.response.value);
    } else {
      editor.update('');
    }
  }, [readOnlySettings, data, error]);
  useEffect(function () {
    applyCurrentSettings(editorInstanceRef.current, readOnlySettings);
  }, [readOnlySettings]);
  return /*#__PURE__*/React.createElement(React.Fragment, null, /*#__PURE__*/React.createElement(EuiScreenReaderOnly, null, /*#__PURE__*/React.createElement("label", {
    htmlFor: inputId
  }, i18n.translate('console.outputTextarea', {
    defaultMessage: 'Dev Tools Console output'
  }))), /*#__PURE__*/React.createElement("div", {
    ref: editorRef,
    className: "conApp__output",
    "data-test-subj": "response-editor"
  }, /*#__PURE__*/React.createElement("div", {
    className: "conApp__outputContent",
    id: "ConAppOutput"
  })));
}

export var EditorOutput = /*#__PURE__*/React.memo(EditorOutputUI);