/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import _ from 'lodash';
import { isExistsFilter, isPhraseFilter, getPhraseFilterValue, getPhraseFilterField, isScriptedPhraseFilter, buildFilter, FilterStateStore, FILTERS } from '../../../../common';

function getExistingFilter(appFilters, fieldName, value) {
  // TODO: On array fields, negating does not negate the combination, rather all terms
  return _.find(appFilters, function (filter) {
    if (!filter) return;

    if (fieldName === '_exists_' && isExistsFilter(filter)) {
      return filter.exists.field === value;
    }

    if (isPhraseFilter(filter)) {
      return getPhraseFilterField(filter) === fieldName && getPhraseFilterValue(filter) === value;
    }

    if (isScriptedPhraseFilter(filter)) {
      return filter.meta.field === fieldName && filter.script.script.params.value === value;
    }
  });
}

function updateExistingFilter(existingFilter, negate) {
  existingFilter.meta.disabled = false;

  if (existingFilter.meta.negate !== negate) {
    existingFilter.meta.negate = !existingFilter.meta.negate;
  }
}
/**
 * Generate filter objects, as a result of triggering a filter action on a
 * specific index pattern field.
 *
 * @param {FilterManager} filterManager - The active filter manager to lookup for existing filters
 * @param {Field | string} field - The field for which filters should be generated
 * @param {any} values - One or more values to filter for.
 * @param {string} operation - "-" to create a negated filter
 * @param {string} index - Index string to generate filters for
 *
 * @returns {object} An array of filters to be added back to filterManager
 */


export function generateFilters(filterManager, field, values, operation, index) {
  values = Array.isArray(values) ? values : [values];
  var fieldObj = _.isObject(field) ? field : {
    name: field
  };
  var fieldName = fieldObj.name;
  var newFilters = [];
  var appFilters = filterManager.getAppFilters();
  var negate = operation === '-';
  var filter;

  _.each(values, function (value) {
    var existing = getExistingFilter(appFilters, fieldName, value);

    if (existing) {
      updateExistingFilter(existing, negate);
      filter = existing;
    } else {
      var tmpIndexPattern = {
        id: index
      }; // exists filter special case:  fieldname = '_exists' and value = fieldname

      var filterType = fieldName === '_exists_' ? FILTERS.EXISTS : FILTERS.PHRASE;
      var actualFieldObj = fieldName === '_exists_' ? {
        name: value
      } : fieldObj; // Fix for #7189 - if value is empty, phrase filters become exists filters.

      var isNullFilter = value === null || value === undefined;
      filter = buildFilter(tmpIndexPattern, actualFieldObj, isNullFilter ? FILTERS.EXISTS : filterType, isNullFilter ? !negate : negate, false, value, null, FilterStateStore.APP_STATE);
    }

    newFilters.push(filter);
  });

  return newFilters;
}