/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { findIndex, isEmpty } from 'lodash';
import { getSchemaByName } from '../schemas';

var isAggRemovable = function isAggRemovable(agg, group, schemas) {
  var schema = getSchemaByName(schemas, agg.schema);
  var metricCount = group.reduce(function (count, aggregation) {
    return aggregation.schema === agg.schema ? ++count : count;
  }, 0); // make sure the the number of these aggs is above the min

  return metricCount > schema.min;
};

var getEnabledMetricAggsCount = function getEnabledMetricAggsCount(group) {
  return group.reduce(function (count, aggregation) {
    return aggregation.schema === 'metric' && aggregation.enabled ? ++count : count;
  }, 0);
};

var calcAggIsTooLow = function calcAggIsTooLow(agg, aggIndex, group, schemas) {
  var schema = getSchemaByName(schemas, agg.schema);

  if (!schema.mustBeFirst) {
    return false;
  }

  var firstDifferentSchema = findIndex(group, function (aggr) {
    return aggr.schema !== agg.schema;
  });

  if (firstDifferentSchema === -1) {
    return false;
  }

  return aggIndex > firstDifferentSchema;
};

function isInvalidAggsTouched(aggsState) {
  var invalidAggs = Object.values(aggsState).filter(function (agg) {
    return !agg.valid;
  });

  if (isEmpty(invalidAggs)) {
    return false;
  }

  return invalidAggs.every(function (agg) {
    return agg.touched;
  });
}

export { isAggRemovable, calcAggIsTooLow, isInvalidAggsTouched, getEnabledMetricAggsCount };