/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import React, { useCallback, useMemo } from 'react';
import { EuiPanel, EuiTitle, EuiSpacer, EuiAccordion, EuiFlexGroup, EuiFlexItem, EuiButtonIcon, EuiToolTip } from '@elastic/eui';
import { i18n } from '@kbn/i18n';
import { FormattedMessage } from '@kbn/i18n/react';
import { ValueAxisOptions } from './value_axis_options';

function ValueAxesPanel(props) {
  var addValueAxis = props.addValueAxis,
      removeValueAxis = props.removeValueAxis,
      seriesParams = props.seriesParams,
      valueAxes = props.valueAxes;
  var getSeries = useCallback(function (axis) {
    var isFirst = valueAxes[0].id === axis.id;
    var series = seriesParams.filter(function (serie) {
      return serie.valueAxis === axis.id || isFirst && !serie.valueAxis;
    });
    return series.map(function (serie) {
      return serie.data.label;
    }).join(', ');
  }, [seriesParams, valueAxes]);
  var removeButtonTooltip = useMemo(function () {
    return i18n.translate('visTypeVislib.controls.pointSeries.valueAxes.removeButtonTooltip', {
      defaultMessage: 'Remove Y-axis'
    });
  }, []);
  var renderRemoveButton = useCallback(function (axis) {
    return /*#__PURE__*/React.createElement(EuiToolTip, {
      position: "bottom",
      content: removeButtonTooltip
    }, /*#__PURE__*/React.createElement(EuiButtonIcon, {
      color: "danger",
      iconType: "cross",
      onClick: function onClick() {
        return removeValueAxis(axis);
      },
      "aria-label": removeButtonTooltip,
      "data-test-subj": "removeValueAxisBtn"
    }));
  }, [removeValueAxis, removeButtonTooltip]);
  var addButtonTooltip = useMemo(function () {
    return i18n.translate('visTypeVislib.controls.pointSeries.valueAxes.addButtonTooltip', {
      defaultMessage: 'Add Y-axis'
    });
  }, []);
  var getButtonContent = useCallback(function (axis) {
    var description = getSeries(axis);
    return /*#__PURE__*/React.createElement(React.Fragment, null, axis.name, ' ', /*#__PURE__*/React.createElement(EuiToolTip, {
      content: description
    }, /*#__PURE__*/React.createElement(React.Fragment, null, description)));
  }, [getSeries]);
  return /*#__PURE__*/React.createElement(EuiPanel, {
    paddingSize: "s"
  }, /*#__PURE__*/React.createElement(EuiFlexGroup, {
    gutterSize: "none",
    justifyContent: "spaceBetween",
    alignItems: "baseline"
  }, /*#__PURE__*/React.createElement(EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/React.createElement(EuiTitle, {
    size: "xs"
  }, /*#__PURE__*/React.createElement("h3", null, /*#__PURE__*/React.createElement(FormattedMessage, {
    id: "visTypeVislib.controls.pointSeries.valueAxes.yAxisTitle",
    defaultMessage: "Y-axes"
  })))), /*#__PURE__*/React.createElement(EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/React.createElement(EuiToolTip, {
    position: "bottom",
    content: addButtonTooltip
  }, /*#__PURE__*/React.createElement(EuiButtonIcon, {
    iconType: "plusInCircleFilled",
    onClick: addValueAxis,
    "aria-label": addButtonTooltip,
    "data-test-subj": "visualizeAddYAxisButton"
  })))), /*#__PURE__*/React.createElement(EuiSpacer, {
    size: "s"
  }), valueAxes.map(function (axis, index) {
    return /*#__PURE__*/React.createElement(EuiAccordion, {
      id: "yAxisAccordion".concat(axis.id),
      key: axis.id,
      "data-test-subj": "toggleYAxisOptions-".concat(axis.id),
      className: "visEditorSidebar__section visEditorSidebar__collapsible",
      initialIsOpen: false,
      buttonContent: getButtonContent(axis),
      buttonClassName: "eui-textTruncate",
      buttonContentClassName: "visEditorSidebar__aggGroupAccordionButtonContent eui-textTruncate",
      "aria-label": i18n.translate('visTypeVislib.controls.pointSeries.valueAxes.toggleOptionsAriaLabel', {
        defaultMessage: 'Toggle {axisName} options',
        values: {
          axisName: axis.name
        }
      }),
      extraAction: valueAxes.length === 1 ? undefined : renderRemoveButton(axis)
    }, /*#__PURE__*/React.createElement(React.Fragment, null, /*#__PURE__*/React.createElement(EuiSpacer, {
      size: "m"
    }), /*#__PURE__*/React.createElement(ValueAxisOptions, {
      axis: axis,
      index: index,
      valueAxis: valueAxes[index],
      isCategoryAxisHorizontal: props.isCategoryAxisHorizontal,
      onValueAxisPositionChanged: props.onValueAxisPositionChanged,
      setParamByIndex: props.setParamByIndex,
      setMultipleValidity: props.setMultipleValidity,
      vis: props.vis
    })));
  }));
}

export { ValueAxesPanel };