/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.secret.cli;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.logstash.secret.SecretIdentifier;
import org.logstash.secret.cli.Terminal;
import org.logstash.secret.store.SecretStore;
import org.logstash.secret.store.SecretStoreFactory;
import org.logstash.secret.store.SecretStoreUtil;
import org.logstash.secret.store.SecureConfig;

public class SecretStoreCli {
    private final Terminal terminal;
    private final SecretStoreFactory secretStoreFactory;

    public SecretStoreCli(Terminal terminal) {
        this(terminal, SecretStoreFactory.fromEnvironment());
    }

    SecretStoreCli(Terminal terminal, SecretStoreFactory secretStoreFactory) {
        this.terminal = terminal;
        this.secretStoreFactory = secretStoreFactory;
    }

    public void command(String primaryCommand, SecureConfig config, String argument) {
        this.terminal.writeLine("");
        Command command = Command.fromString(primaryCommand).orElse(Command.HELP);
        Optional<Command> sub = Command.fromString(argument);
        boolean help = Command.HELP.equals(sub.orElse(null));
        switch (command) {
            case CREATE: {
                if (help) {
                    this.terminal.writeLine("Creates a new keystore. For example: 'bin/logstash-keystore create'");
                    return;
                }
                if (this.secretStoreFactory.exists(config.clone())) {
                    this.terminal.write("An Logstash keystore already exists. Overwrite ? [y/N] ");
                    if (!SecretStoreCli.isYes(this.terminal.readLine())) break;
                    this.create(config);
                    break;
                }
                this.create(config);
                break;
            }
            case LIST: {
                if (help) {
                    this.terminal.writeLine("List all secret identifiers from the keystore. For example: `bin/logstash-keystore list`. Note - only the identifiers will be listed, not the secrets.");
                    return;
                }
                Collection<SecretIdentifier> ids = this.secretStoreFactory.load(config).list();
                List<String> keys = ids.stream().filter(id -> !id.equals(SecretStoreFactory.LOGSTASH_MARKER)).map(id -> id.getKey()).collect(Collectors.toList());
                Collections.sort(keys);
                keys.forEach(this.terminal::writeLine);
                break;
            }
            case ADD: {
                if (help) {
                    this.terminal.writeLine("Adds a new secret to the keystore. For example: `bin/logstash-keystore add my-secret`, at the prompt enter your secret. You will use the identifier ${my-secret} in your Logstash configuration.");
                    return;
                }
                if (argument == null || argument.isEmpty()) {
                    this.terminal.writeLine("ERROR: You must supply a identifier to add. (e.g. bin/logstash-keystore add my-secret)");
                    return;
                }
                if (this.secretStoreFactory.exists(config.clone())) {
                    SecretIdentifier id2 = new SecretIdentifier(argument);
                    SecretStore secretStore = this.secretStoreFactory.load(config);
                    byte[] s = secretStore.retrieveSecret(id2);
                    if (s == null) {
                        this.terminal.write(String.format("Enter value for %s: ", argument));
                        char[] secret = this.terminal.readSecret();
                        if (secret == null || secret.length == 0) {
                            this.terminal.writeLine("ERROR: You must supply a identifier to add. (e.g. bin/logstash-keystore add my-secret)");
                            return;
                        }
                        this.add(secretStore, id2, SecretStoreUtil.asciiCharToBytes(secret));
                        break;
                    }
                    SecretStoreUtil.clearBytes(s);
                    this.terminal.write(String.format("%s already exists. Overwrite ? [y/N] ", argument));
                    if (!SecretStoreCli.isYes(this.terminal.readLine())) break;
                    this.terminal.write(String.format("Enter value for %s: ", argument));
                    char[] secret = this.terminal.readSecret();
                    this.add(secretStore, id2, SecretStoreUtil.asciiCharToBytes(secret));
                    break;
                }
                this.terminal.writeLine(String.format("ERROR: Logstash keystore not found. Use 'create' command to create one.", new Object[0]));
                break;
            }
            case REMOVE: {
                if (help) {
                    this.terminal.writeLine("Removes a secret from the keystore. For example: `bin/logstash-keystore remove my-secret`");
                    return;
                }
                if (argument == null || argument.isEmpty()) {
                    this.terminal.writeLine("ERROR: You must supply a value to remove. (e.g. bin/logstash-keystore remove my-secret)");
                    return;
                }
                SecretIdentifier id3 = new SecretIdentifier(argument);
                SecretStore secretStore = this.secretStoreFactory.load(config);
                byte[] s = secretStore.retrieveSecret(id3);
                if (s == null) {
                    this.terminal.writeLine(String.format("ERROR: '%s' does not exist in the Logstash keystore.", argument));
                    break;
                }
                SecretStoreUtil.clearBytes(s);
                secretStore.purgeSecret(id3);
                this.terminal.writeLine(String.format("Removed '%s' from the Logstash keystore.", id3.getKey()));
                break;
            }
            case HELP: {
                this.terminal.writeLine("Usage:");
                this.terminal.writeLine("--------");
                this.terminal.writeLine("bin/logstash-keystore [option] command [argument]");
                this.terminal.writeLine("");
                this.terminal.writeLine("Commands:");
                this.terminal.writeLine("--------");
                this.terminal.writeLine("create - Creates a new Logstash keystore  (e.g. bin/logstash-keystore create)");
                this.terminal.writeLine("list   - List entries in the keystore  (e.g. bin/logstash-keystore list)");
                this.terminal.writeLine("add    - Add a value to the keystore (e.g. bin/logstash-keystore add my-secret)");
                this.terminal.writeLine("remove - Remove a value from the keystore  (e.g. bin/logstash-keystore remove my-secret)");
                this.terminal.writeLine("");
                this.terminal.writeLine("Argument:");
                this.terminal.writeLine("--------");
                this.terminal.writeLine("--help - Display command specific help  (e.g. bin/logstash-keystore add --help)");
                this.terminal.writeLine("");
                this.terminal.writeLine("Options:");
                this.terminal.writeLine("--------");
                this.terminal.writeLine("--path.settings - Set the directory for the keystore. This is should be the same directory as the logstash.yml settings file. The default is the config directory under Logstash home. (e.g. bin/logstash-keystore --path.settings /tmp/foo create)");
                this.terminal.writeLine("");
            }
        }
    }

    private void add(SecretStore secretStore, SecretIdentifier id, byte[] secret) {
        secretStore.persistSecret(id, secret);
        this.terminal.writeLine(String.format("Added '%s' to the Logstash keystore.", id.getKey()));
        SecretStoreUtil.clearBytes(secret);
    }

    private void create(SecureConfig config) {
        if (System.getenv("LOGSTASH_KEYSTORE_PASS") == null) {
            this.terminal.write(String.format("WARNING: The keystore password is not set. Please set the environment variable `%s`. Failure to do so will result in reduced security. Continue without password protection on the keystore? [y/N] ", "LOGSTASH_KEYSTORE_PASS"));
            if (SecretStoreCli.isYes(this.terminal.readLine())) {
                this.deleteThenCreate(config);
            }
        } else {
            this.deleteThenCreate(config);
        }
    }

    private void deleteThenCreate(SecureConfig config) {
        this.secretStoreFactory.delete(config.clone());
        this.secretStoreFactory.create(config.clone());
        char[] fileLocation = config.getPlainText("keystore.file");
        this.terminal.writeLine("Created Logstash keystore" + (fileLocation == null ? "." : " at " + new String(fileLocation)));
    }

    private static boolean isYes(String response) {
        return "y".equalsIgnoreCase(response) || "yes".equalsIgnoreCase(response);
    }

    static enum Command {
        CREATE("create"),
        LIST("list"),
        ADD("add"),
        REMOVE("remove"),
        HELP("--help");

        private final String option;

        private Command(String option) {
            this.option = option;
        }

        static Optional<Command> fromString(String input) {
            Optional<Command> command = EnumSet.allOf(Command.class).stream().filter(c -> c.option.equals(input)).findFirst();
            return command;
        }
    }
}

