/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.launcher;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.ref.WeakReference;
import java.nio.channels.FileChannel;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.Formatter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import org.graalvm.launcher.AbstractLanguageLauncher;
import org.graalvm.nativeimage.ProcessProperties;
import org.graalvm.nativeimage.RuntimeOptions;
import org.graalvm.nativeimage.VMRuntime;
import org.graalvm.options.OptionCategory;
import org.graalvm.options.OptionDescriptor;
import org.graalvm.options.OptionDescriptors;
import org.graalvm.options.OptionStability;
import org.graalvm.options.OptionType;
import org.graalvm.polyglot.Context;
import org.graalvm.polyglot.Engine;
import org.graalvm.polyglot.Instrument;
import org.graalvm.polyglot.Language;
import org.graalvm.polyglot.PolyglotException;

public abstract class Launcher {
    private static final boolean STATIC_VERBOSE = Boolean.getBoolean("org.graalvm.launcher.verbose");
    private static final boolean SHELL_SCRIPT_LAUNCHER = Boolean.getBoolean("org.graalvm.launcher.shell");
    static final boolean IS_AOT = Boolean.getBoolean("com.oracle.graalvm.isaot");
    private static Engine tempEngine;
    final Native nativeAccess;
    private final boolean verbose;
    private boolean help;
    private boolean helpInternal;
    private boolean helpExpert;
    private boolean helpTools;
    private boolean helpLanguages;
    private boolean helpVM;
    private boolean seenPolyglot;
    private Path logFile;
    private VersionAction versionAction = VersionAction.None;
    private Path home;
    private static final String CLASSPATH;

    Launcher() {
        this.verbose = STATIC_VERBOSE || Boolean.valueOf(System.getenv("VERBOSE_GRAALVM_LAUNCHERS")) != false;
        this.nativeAccess = IS_AOT ? new Native() : null;
    }

    final boolean isPolyglot() {
        return this.seenPolyglot;
    }

    final void setPolyglot(boolean polyglot) {
        this.seenPolyglot = polyglot;
    }

    final void setupLogHandler(Context.Builder builder) {
        if (this.logFile != null) {
            try {
                builder.logHandler(Launcher.newLogStream(this.logFile));
            }
            catch (IOException ioe) {
                throw this.abort(ioe);
            }
        }
    }

    static Engine getTempEngine() {
        if (tempEngine == null) {
            tempEngine = Engine.create();
        }
        return tempEngine;
    }

    protected void argumentsProcessingDone() {
        if (tempEngine != null) {
            tempEngine.close();
            tempEngine = null;
        }
    }

    static void handleAbortException(AbortException e) {
        if (e.getMessage() != null) {
            System.err.println("ERROR: " + e.getMessage());
        }
        if (e.getCause() != null) {
            e.printStackTrace();
        }
        System.exit(e.getExitCode());
    }

    static void handlePolyglotException(PolyglotException e) {
        if (e.getMessage() != null) {
            System.err.println("ERROR: " + e.getMessage());
        }
        if (e.isInternalError()) {
            e.printStackTrace();
        }
        if (e.isExit()) {
            System.exit(e.getExitStatus());
        } else {
            System.exit(1);
        }
    }

    protected final AbortException exit() {
        return this.exit(0);
    }

    protected final AbortException exit(int exitCode) {
        return this.abort((String)null, exitCode);
    }

    protected final AbortException abort(String message) {
        return this.abort(message, 1);
    }

    protected final AbortException abort(String message, int exitCode) {
        throw new AbortException(message, exitCode);
    }

    protected final AbortException abort(Throwable t) {
        return this.abort(t, 255);
    }

    protected final AbortException abort(Throwable t, int exitCode) {
        if (t.getCause() instanceof IOException && t.getClass() == RuntimeException.class) {
            String message = t.getMessage();
            if (message != null && !message.startsWith(t.getCause().getClass().getName() + ": ")) {
                System.err.println(message);
            }
            throw this.abort((IOException)t.getCause(), exitCode);
        }
        throw new AbortException(t, exitCode);
    }

    protected final AbortException abort(IOException e) {
        return this.abort(e, 74);
    }

    protected final AbortException abort(IOException e, int exitCode) {
        String message = e.getMessage();
        if (message != null) {
            if (e instanceof NoSuchFileException) {
                throw this.abort("Not such file: " + message, exitCode);
            }
            if (e instanceof AccessDeniedException) {
                throw this.abort("Access denied: " + message, exitCode);
            }
            throw this.abort(message + " (" + e.getClass().getSimpleName() + ")", exitCode);
        }
        throw this.abort((Throwable)e, exitCode);
    }

    protected AbortException abortUnrecognizedArgument(String argument) {
        throw this.abortInvalidArgument(argument, "Unrecognized argument: '" + argument + "'. Use --help for usage instructions.");
    }

    protected final AbortException abortInvalidArgument(String argument, String message) {
        return this.abortInvalidArgument(argument, message, 2);
    }

    protected final AbortException abortInvalidArgument(String argument, String message, int exitCode) {
        List<String> matches;
        Set<String> allArguments = this.collectAllArguments();
        String testString = argument;
        int equalIndex = argument.indexOf(61);
        if (equalIndex != -1) {
            testString = argument.substring(0, equalIndex);
        }
        if ((matches = Launcher.fuzzyMatch(allArguments, testString, 0.7f)).isEmpty()) {
            matches = Launcher.fuzzyMatch(allArguments, testString, 0.5f);
        }
        StringBuilder sb = new StringBuilder();
        if (message != null) {
            sb.append(message);
        }
        if (!matches.isEmpty()) {
            if (sb.length() > 0) {
                sb.append(System.lineSeparator());
            }
            sb.append("Did you mean one of the following arguments?").append(System.lineSeparator());
            Iterator<String> iterator = matches.iterator();
            while (true) {
                String match = iterator.next();
                sb.append("  ").append(match);
                if (!iterator.hasNext()) break;
                sb.append(System.lineSeparator());
            }
        }
        if (sb.length() > 0) {
            throw this.abort(sb.toString(), exitCode);
        }
        throw this.exit(exitCode);
    }

    protected void warn(String message) {
        System.err.println("Warning: " + message);
    }

    protected void warn(String message, Object ... args) {
        StringBuilder sb = new StringBuilder("Warning: ");
        new Formatter(sb).format(message, args);
        sb.append(System.lineSeparator());
        System.err.print(sb.toString());
    }

    protected abstract void printHelp(OptionCategory var1);

    protected abstract void printVersion();

    protected abstract void collectArguments(Set<String> var1);

    private String executableName(String basename) {
        switch (OS.current) {
            case Linux: 
            case Darwin: 
            case Solaris: {
                return basename;
            }
        }
        throw this.abort("executableName: OS not supported: " + (Object)((Object)OS.current));
    }

    protected static void printPolyglotVersions() {
        Engine engine = Launcher.getTempEngine();
        System.out.println("GraalVM Polyglot Engine Version " + engine.getVersion());
        Path graalVMHome = Engine.findHome();
        if (graalVMHome != null) {
            System.out.println("GraalVM Home " + graalVMHome);
        }
        Launcher.printLanguages(engine, true);
        Launcher.printInstruments(engine, true);
    }

    protected String getMainClass() {
        return this.getClass().getName();
    }

    protected VMType getDefaultVMType() {
        return VMType.Native;
    }

    public static boolean isAOT() {
        return IS_AOT;
    }

    private boolean isVerbose() {
        return this.verbose;
    }

    protected boolean isGraalVMAvailable() {
        return this.getGraalVMHome() != null;
    }

    protected boolean isStandalone() {
        return !this.isGraalVMAvailable();
    }

    protected Path getGraalVMHome() {
        if (this.home == null) {
            this.home = Engine.findHome();
        }
        return this.home;
    }

    final boolean runPolyglotAction() {
        boolean printDefaultHelp;
        OptionCategory helpCategory = this.helpInternal ? OptionCategory.INTERNAL : (this.helpExpert ? OptionCategory.EXPERT : OptionCategory.USER);
        switch (this.versionAction) {
            case PrintAndExit: {
                Launcher.printPolyglotVersions();
                return true;
            }
            case PrintAndContinue: {
                Launcher.printPolyglotVersions();
                break;
            }
        }
        boolean bl = printDefaultHelp = this.help || (this.helpExpert || this.helpInternal) && !this.helpTools && !this.helpLanguages && !this.helpVM;
        if (printDefaultHelp) {
            VMType defaultVMType = SHELL_SCRIPT_LAUNCHER ? VMType.JVM : this.getDefaultVMType();
            this.printHelp(helpCategory);
            System.out.println();
            System.out.println("Runtime options:");
            if (!this.isStandalone()) {
                Launcher.printOption("--polyglot", "Run with all other guest languages accessible.");
            }
            if (!SHELL_SCRIPT_LAUNCHER) {
                Launcher.printOption("--native", "Run using the native launcher with limited Java access" + (defaultVMType == VMType.Native ? " (default)" : "") + ".");
            }
            if (!this.isStandalone()) {
                Launcher.printOption("--jvm", "Run on the Java Virtual Machine with Java access" + (defaultVMType == VMType.JVM ? " (default)" : "") + ".");
            }
            Launcher.printOption("--vm.[option]", "Pass options to the host VM. To see available options, use '--help:vm'.");
            Launcher.printOption("--help", "Print this help message.");
            Launcher.printOption("--help:languages", "Print options for all installed languages.");
            Launcher.printOption("--help:tools", "Print options for all installed tools.");
            Launcher.printOption("--help:vm", "Print options for the host VM.");
            Launcher.printOption("--help:expert", "Print additional options for experts.");
            Launcher.printOption("--help:internal", "Print internal options for debugging language implementations and tools.");
            Launcher.printOption("--version:graalvm", "Print GraalVM version information and exit.");
            Launcher.printOption("--show-version:graalvm", "Print GraalVM version information and continue execution.");
            Launcher.printOption("--log.file=<String>", "Redirect guest languages logging into a given file.");
            Launcher.printOption("--log.[logger].level=<String>", "Set language log level to OFF, SEVERE, WARNING, INFO, CONFIG, FINE, FINER, FINEST or ALL.");
            Launcher.printEngineOptions(Launcher.getTempEngine(), helpCategory);
        }
        if (this.helpLanguages) {
            Launcher.printLanguageOptions(Launcher.getTempEngine(), helpCategory);
        }
        if (this.helpTools) {
            Launcher.printInstrumentOptions(Launcher.getTempEngine(), helpCategory);
        }
        if (this.helpVM) {
            if (this.nativeAccess == null) {
                Launcher.printJvmHelp();
            } else {
                this.nativeAccess.printNativeHelp();
            }
        }
        if (printDefaultHelp || this.helpLanguages || this.helpTools || this.helpVM) {
            System.out.println();
            if (this.helpLanguages) {
                this.printOtherHelpCategories("language", "--help:languages");
            }
            if (this.helpTools) {
                this.printOtherHelpCategories("tool", "--help:tools");
            }
            System.out.println("See http://www.graalvm.org for more information.");
            return true;
        }
        return false;
    }

    private void printOtherHelpCategories(String kind, String option) {
        if (this.helpExpert || this.helpInternal) {
            System.out.println("Use '" + option + "' to list user " + kind + " options.");
        }
        if (!this.helpExpert) {
            System.out.println("Use '" + option + " --help:expert' to list expert " + kind + " options.");
        }
        if (!this.helpInternal) {
            System.out.println("Use '" + option + " --help:internal' to list internal " + kind + " options.");
        }
    }

    private static void printEngineOptions(Engine engine, OptionCategory optionCategory) {
        List<PrintableOption> engineOptions = Launcher.filterOptions(engine.getOptions(), optionCategory);
        if (!engineOptions.isEmpty()) {
            System.out.println();
            Launcher.printOptions(engineOptions, Launcher.optionsTitle("engine", optionCategory), 2);
        }
    }

    private static void printInstrumentOptions(Engine engine, OptionCategory optionCategory) {
        List options;
        HashMap<Instrument, List> instrumentsOptions = new HashMap<Instrument, List>();
        List<Instrument> instruments = Launcher.sortedInstruments(engine);
        for (Instrument instrument : instruments) {
            options = Launcher.filterOptions(instrument.getOptions(), optionCategory);
            if (options.isEmpty()) continue;
            instrumentsOptions.put(instrument, options);
        }
        if (!instrumentsOptions.isEmpty()) {
            System.out.println();
            System.out.println(Launcher.optionsTitle("tool", optionCategory));
            for (Instrument instrument : instruments) {
                options = (List)instrumentsOptions.get(instrument);
                if (options == null) continue;
                Launcher.printOptions(options, "  " + instrument.getName() + ":", 4);
            }
        }
    }

    private static void printLanguageOptions(Engine engine, OptionCategory optionCategory) {
        List options;
        HashMap<Language, List> languagesOptions = new HashMap<Language, List>();
        List<Language> languages = Launcher.sortedLanguages(engine);
        for (Language language : languages) {
            options = Launcher.filterOptions(language.getOptions(), optionCategory);
            if (options.isEmpty()) continue;
            languagesOptions.put(language, options);
        }
        if (!languagesOptions.isEmpty()) {
            System.out.println();
            System.out.println(Launcher.optionsTitle("language", optionCategory));
            for (Language language : languages) {
                options = (List)languagesOptions.get(language);
                if (options == null) continue;
                Launcher.printOptions(options, "  " + language.getName() + ":", 4);
            }
        }
    }

    private static String optionsTitle(String kind, OptionCategory optionCategory) {
        String category;
        switch (optionCategory) {
            case USER: {
                category = "User ";
                break;
            }
            case EXPERT: {
                category = "Expert ";
                break;
            }
            case INTERNAL: {
                category = "Internal ";
                break;
            }
            default: {
                category = "";
            }
        }
        return category + kind + " options:";
    }

    private static List<PrintableOption> filterOptions(OptionDescriptors descriptors, OptionCategory optionCategory) {
        ArrayList<PrintableOption> options = new ArrayList<PrintableOption>();
        for (OptionDescriptor descriptor : descriptors) {
            if (descriptor.isDeprecated() || !Launcher.sameCategory(descriptor, optionCategory)) continue;
            options.add(Launcher.asPrintableOption(descriptor));
        }
        return options;
    }

    private static boolean sameCategory(OptionDescriptor descriptor, OptionCategory optionCategory) {
        return descriptor.getCategory().ordinal() == optionCategory.ordinal();
    }

    void parsePolyglotOptions(String defaultOptionPrefix, Map<String, String> polyglotOptions, List<String> unrecognizedArgs) {
        boolean experimentalOptions = false;
        Iterator<String> iterator = unrecognizedArgs.iterator();
        while (iterator.hasNext()) {
            String arg;
            switch (arg = iterator.next()) {
                case "--experimental-options": 
                case "--experimental-options=true": {
                    experimentalOptions = true;
                    break;
                }
                case "--experimental-options=false": {
                    experimentalOptions = false;
                }
            }
        }
        for (String arg : unrecognizedArgs) {
            this.parsePolyglotOption(defaultOptionPrefix, polyglotOptions, experimentalOptions, arg);
        }
    }

    private void parsePolyglotOption(String defaultOptionPrefix, Map<String, String> polyglotOptions, boolean experimentalOptions, String arg) {
        switch (arg) {
            case "--help": {
                this.help = true;
                break;
            }
            case "--help:debug": {
                this.warn("--help:debug is deprecated, use --help:internal instead.");
                this.helpInternal = true;
                break;
            }
            case "--help:internal": {
                this.helpInternal = true;
                break;
            }
            case "--help:expert": {
                this.helpExpert = true;
                break;
            }
            case "--help:tools": {
                this.helpTools = true;
                break;
            }
            case "--help:languages": {
                this.helpLanguages = true;
                break;
            }
            case "--help:vm": {
                this.helpVM = true;
                break;
            }
            case "--version:graalvm": {
                this.versionAction = VersionAction.PrintAndExit;
                break;
            }
            case "--show-version:graalvm": {
                this.versionAction = VersionAction.PrintAndContinue;
                break;
            }
            case "--polyglot": {
                this.seenPolyglot = true;
                break;
            }
            case "--experimental-options": 
            case "--experimental-options=true": 
            case "--experimental-options=false": {
                break;
            }
            default: {
                OptionDescriptor descriptor;
                String value;
                String key;
                if (arg.startsWith("--jvm.") && arg.length() > "--jvm.".length() || arg.equals("--jvm")) {
                    if (Launcher.isAOT()) {
                        throw this.abort("should not reach here: jvm option failed to switch to JVM");
                    }
                    return;
                }
                if (arg.startsWith("--native.") && arg.length() > "--native.".length() || arg.equals("--native")) {
                    if (!Launcher.isAOT()) {
                        throw this.abort("native options are not supported on the JVM");
                    }
                    return;
                }
                if (arg.startsWith("--vm.") && arg.length() > "--vm.".length()) {
                    return;
                }
                if (arg.length() <= 2 || !arg.startsWith("--")) {
                    throw this.abortUnrecognizedArgument(arg);
                }
                int eqIdx = arg.indexOf(61);
                if (eqIdx < 0) {
                    key = arg.substring(2);
                    value = null;
                } else {
                    key = arg.substring(2, eqIdx);
                    value = arg.substring(eqIdx + 1);
                }
                if (value == null) {
                    value = "true";
                }
                int index = key.indexOf(46);
                String group = key;
                if (index >= 0) {
                    group = group.substring(0, index);
                }
                if ("log".equals(group)) {
                    if (key.endsWith(".level")) {
                        try {
                            Level.parse(value);
                            polyglotOptions.put(key, value);
                        }
                        catch (IllegalArgumentException e) {
                            throw this.abort(String.format("Invalid log level %s specified. %s'", arg, e.getMessage()));
                        }
                        return;
                    }
                    if (key.equals("log.file")) {
                        this.logFile = Paths.get(value, new String[0]);
                        return;
                    }
                }
                if ((descriptor = Launcher.findPolyglotOptionDescriptor(group, key)) == null) {
                    if (defaultOptionPrefix != null) {
                        descriptor = Launcher.findPolyglotOptionDescriptor(defaultOptionPrefix, defaultOptionPrefix + "." + key);
                    }
                    if (descriptor == null) {
                        throw this.abortUnrecognizedArgument(arg);
                    }
                }
                try {
                    descriptor.getKey().getType().convert(value);
                }
                catch (IllegalArgumentException e) {
                    throw this.abort(String.format("Invalid argument %s specified. %s'", arg, e.getMessage()));
                }
                if (descriptor.isDeprecated()) {
                    this.warn("Option '" + descriptor.getName() + "' is deprecated and might be removed from future versions.");
                }
                if (!experimentalOptions && descriptor.getStability() == OptionStability.EXPERIMENTAL) {
                    throw this.abort(String.format("Option '%s' is experimental and must be enabled via '--experimental-options'%nDo not use experimental options in production environments.", arg));
                }
                polyglotOptions.put(descriptor.getName(), value);
            }
        }
    }

    private static OptionDescriptor findPolyglotOptionDescriptor(String group, String key) {
        OptionDescriptors descriptors = null;
        switch (group) {
            case "engine": {
                descriptors = Launcher.getTempEngine().getOptions();
                break;
            }
            default: {
                Engine engine = Launcher.getTempEngine();
                if (engine.getLanguages().containsKey(group)) {
                    descriptors = ((Language)engine.getLanguages().get(group)).getOptions();
                    break;
                }
                if (!engine.getInstruments().containsKey(group)) break;
                descriptors = ((Instrument)engine.getInstruments().get(group)).getOptions();
            }
        }
        if (descriptors == null) {
            return null;
        }
        return descriptors.get(key);
    }

    private Set<String> collectAllArguments() {
        Engine engine = Launcher.getTempEngine();
        LinkedHashSet<String> options = new LinkedHashSet<String>();
        this.collectArguments(options);
        if (!this.isStandalone()) {
            options.add("--polyglot");
            options.add("--jvm");
        }
        options.add("--native");
        options.add("--help");
        options.add("--help:languages");
        options.add("--help:tools");
        options.add("--help:expert");
        options.add("--help:internal");
        options.add("--help:vm");
        options.add("--version:graalvm");
        options.add("--show-version:graalvm");
        Launcher.addOptions(engine.getOptions(), options);
        for (Instrument instrument : engine.getInstruments().values()) {
            Launcher.addOptions(instrument.getOptions(), options);
        }
        String languageId = null;
        if (this instanceof AbstractLanguageLauncher) {
            languageId = ((AbstractLanguageLauncher)this).getLanguageId();
        }
        for (Language language : engine.getLanguages().values()) {
            if (language.getId().equals(languageId)) {
                for (OptionDescriptor descriptor : language.getOptions()) {
                    options.add("--" + descriptor.getName().substring(languageId.length() + 1));
                }
            }
            Launcher.addOptions(language.getOptions(), options);
        }
        return options;
    }

    private static void addOptions(OptionDescriptors descriptors, Set<String> target) {
        for (OptionDescriptor descriptor : descriptors) {
            target.add("--" + descriptor.getName());
        }
    }

    private static List<String> fuzzyMatch(Set<String> arguments, String argument, float threshold) {
        ArrayList<String> matches = new ArrayList<String>();
        for (String arg : arguments) {
            float score = Launcher.stringSimilarity(arg, argument);
            if (!(score >= threshold)) continue;
            matches.add(arg);
        }
        return matches;
    }

    private static float stringSimilarity(String str1, String str2) {
        int hit = 0;
        block0: for (int i = 0; i < str1.length() - 1; ++i) {
            for (int j = 0; j < str2.length() - 1; ++j) {
                if (str1.charAt(i) != str2.charAt(j) || str1.charAt(i + 1) != str2.charAt(j + 1)) continue;
                ++hit;
                continue block0;
            }
        }
        return 2.0f * (float)hit / (float)(str1.length() + str2.length());
    }

    static List<Language> sortedLanguages(Engine engine) {
        ArrayList<Language> languages = new ArrayList<Language>(engine.getLanguages().values());
        languages.sort(Comparator.comparing(Language::getId));
        return languages;
    }

    static List<Instrument> sortedInstruments(Engine engine) {
        ArrayList<Instrument> instruments = new ArrayList<Instrument>();
        for (Instrument instrument : engine.getInstruments().values()) {
            if (!instrument.getOptions().iterator().hasNext()) continue;
            instruments.add(instrument);
        }
        instruments.sort(Comparator.comparing(Instrument::getId));
        return instruments;
    }

    static void printOption(OptionCategory optionCategory, OptionDescriptor descriptor) {
        if (!descriptor.isDeprecated() && Launcher.sameCategory(descriptor, optionCategory)) {
            Launcher.printOption(Launcher.asPrintableOption(descriptor));
        }
    }

    private static PrintableOption asPrintableOption(OptionDescriptor descriptor) {
        StringBuilder key = new StringBuilder("--");
        key.append(descriptor.getName());
        Object defaultValue = descriptor.getKey().getDefaultValue();
        if (!(defaultValue instanceof Boolean) || defaultValue != Boolean.FALSE) {
            key.append("=<");
            key.append(descriptor.getKey().getType().getName());
            key.append(">");
        }
        return new PrintableOption(key.toString(), descriptor.getHelp());
    }

    static void printOption(String option, String description) {
        Launcher.printOption(option, description, 2);
    }

    private static String spaces(int length) {
        return new String(new char[length]).replace('\u0000', ' ');
    }

    private static String wrap(String s) {
        int width = 120;
        StringBuilder sb = new StringBuilder(s);
        int cursor = 0;
        while (cursor + 120 < sb.length()) {
            int i = sb.lastIndexOf(" ", cursor + 120);
            if (i == -1 || i < cursor) {
                i = sb.indexOf(" ", cursor + 120);
            }
            if (i == -1) break;
            sb.replace(i, i + 1, System.lineSeparator());
            cursor = i;
        }
        return sb.toString();
    }

    private static void printOption(String option, String description, int indentation) {
        String indent = Launcher.spaces(indentation);
        String desc = Launcher.wrap(description != null ? description : "");
        String nl = System.lineSeparator();
        String[] descLines = desc.split(nl);
        int optionWidth = 45;
        if (option.length() >= optionWidth && description != null) {
            System.out.println(indent + option + nl + indent + Launcher.spaces(optionWidth) + descLines[0]);
        } else {
            System.out.println(indent + option + Launcher.spaces(optionWidth - option.length()) + descLines[0]);
        }
        for (int i = 1; i < descLines.length; ++i) {
            System.out.println(indent + Launcher.spaces(optionWidth) + descLines[i]);
        }
    }

    private static void printOption(PrintableOption option) {
        Launcher.printOption(option, 2);
    }

    private static void printOption(PrintableOption option, int indentation) {
        Launcher.printOption(option.option, option.description, indentation);
    }

    private static void printOptions(List<PrintableOption> options, String title, int indentation) {
        Collections.sort(options);
        System.out.println(title);
        for (PrintableOption option : options) {
            Launcher.printOption(option, indentation);
        }
    }

    private static void serializePolyglotOptions(Map<String, String> polyglotOptions, List<String> args) {
        if (polyglotOptions == null) {
            return;
        }
        for (Map.Entry<String, String> entry : polyglotOptions.entrySet()) {
            args.add("--" + entry.getKey() + '=' + entry.getValue());
        }
    }

    private static void printLanguages(Engine engine, boolean printWhenEmpty) {
        if (engine.getLanguages().isEmpty()) {
            if (printWhenEmpty) {
                System.out.println("  Installed Languages: none");
            }
        } else {
            System.out.println("  Installed Languages:");
            ArrayList<Language> languages = new ArrayList<Language>(engine.getLanguages().size());
            int nameLength = 0;
            for (Language language : engine.getLanguages().values()) {
                languages.add(language);
                nameLength = Integer.max(nameLength, language.getName().length());
            }
            languages.sort(Comparator.comparing(Language::getId));
            String langFormat = "    %-" + nameLength + "s%s version %s%n";
            for (Language language : languages) {
                String host = "";
                String version = language.getVersion();
                if (version == null || version.length() == 0) {
                    version = "";
                }
                System.out.printf(langFormat, language.getName().isEmpty() ? "Unnamed" : language.getName(), host, version);
            }
        }
    }

    private static void printInstruments(Engine engine, boolean printWhenEmpty) {
        if (engine.getInstruments().isEmpty()) {
            if (printWhenEmpty) {
                System.out.println("  Installed Tools: none");
            }
        } else {
            System.out.println("  Installed Tools:");
            List<Instrument> instruments = Launcher.sortedInstruments(engine);
            int nameLength = 0;
            for (Instrument instrument : instruments) {
                nameLength = Integer.max(nameLength, instrument.getName().length());
            }
            String instrumentFormat = "    %-" + nameLength + "s version %s%n";
            for (Instrument instrument : instruments) {
                String version = instrument.getVersion();
                if (version == null || version.length() == 0) {
                    version = "";
                }
                System.out.printf(instrumentFormat, instrument.getName().isEmpty() ? instrument.getId() : instrument.getName(), version);
            }
        }
    }

    private static void printJvmHelp() {
        System.out.println("JVM options:");
        Launcher.printOption("--vm.classpath <...>", "A " + File.pathSeparator + " separated list of classpath entries that will be added to the JVM's classpath");
        Launcher.printOption("--vm.D<name>=<value>", "Set a system property");
        Launcher.printOption("--vm.esa", "Enable system assertions");
        Launcher.printOption("--vm.ea[:<packagename>...|:<classname>]", "Enable assertions with specified granularity");
        Launcher.printOption("--vm.agentlib:<libname>[=<options>]", "Load native agent library <libname>");
        Launcher.printOption("--vm.agentpath:<pathname>[=<options>]", "Load native agent library by full pathname");
        Launcher.printOption("--vm.javaagent:<jarpath>[=<options>]", "Load Java programming language agent");
        Launcher.printOption("--vm.Xbootclasspath/a:<...>", "A " + File.pathSeparator + " separated list of classpath entries that will be added to the JVM's boot classpath");
        Launcher.printOption("--vm.Xmx<size>", "Set maximum Java heap size");
        Launcher.printOption("--vm.Xms<size>", "Set initial Java heap size");
        Launcher.printOption("--vm.Xss<size>", "Set java thread stack size");
    }

    private static void printBasicNativeHelp() {
        Launcher.printOption("--vm.D<property>=<value>", "Sets a system property");
        Launcher.printOption("--vm.Xmn<value>", "Sets the maximum size of the young generation, in bytes. Default: 256MB.");
        Launcher.printOption("--vm.Xmx<value>", "Sets the maximum size of the heap, in bytes. Default: MaximumHeapSizePercent * physical memory.");
        Launcher.printOption("--vm.Xms<value>", "Sets the minimum size of the heap, in bytes. Default: 2 * maximum young generation size.");
        Launcher.printOption("--vm.Xss<value>", "Sets the size of each thread stack, in bytes. Default: OS-dependent.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static OutputStream newLogStream(Path path) throws IOException {
        Path usedPath = path;
        Path lockFile = null;
        FileChannel lockFileChannel = null;
        int unique = 0;
        while (true) {
            StringBuilder lockFileNameBuilder = new StringBuilder();
            lockFileNameBuilder.append(path.toString());
            if (unique > 0) {
                lockFileNameBuilder.append(unique);
                usedPath = Paths.get(lockFileNameBuilder.toString(), new String[0]);
            }
            lockFileNameBuilder.append(".lck");
            lockFile = Paths.get(lockFileNameBuilder.toString(), new String[0]);
            Map.Entry<FileChannel, Boolean> openResult = Launcher.openChannel(lockFile);
            if (openResult != null) {
                lockFileChannel = openResult.getKey();
                if (Launcher.lock(lockFileChannel, openResult.getValue())) break;
                lockFileChannel.close();
            }
            ++unique;
        }
        assert (lockFile != null && lockFileChannel != null);
        boolean success = false;
        try {
            LockableOutputStream stream = new LockableOutputStream(new BufferedOutputStream(Files.newOutputStream(usedPath, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.APPEND)), lockFile, lockFileChannel);
            success = true;
            LockableOutputStream lockableOutputStream = stream;
            return lockableOutputStream;
        }
        finally {
            if (!success) {
                LockableOutputStream.unlock(lockFile, lockFileChannel);
            }
        }
    }

    private static Map.Entry<FileChannel, Boolean> openChannel(Path path) throws IOException {
        FileChannel channel = null;
        for (int retries = 0; channel == null && retries < 2; ++retries) {
            try {
                channel = FileChannel.open(path, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
                return new AbstractMap.SimpleImmutableEntry<FileChannel, Boolean>(channel, true);
            }
            catch (FileAlreadyExistsException faee) {
                if (Files.isRegularFile(path, LinkOption.NOFOLLOW_LINKS) && Launcher.isParentWritable(path)) {
                    try {
                        channel = FileChannel.open(path, StandardOpenOption.WRITE, StandardOpenOption.APPEND);
                        return new AbstractMap.SimpleImmutableEntry<FileChannel, Boolean>(channel, false);
                    }
                    catch (NoSuchFileException noSuchFileException) {
                        continue;
                    }
                    catch (IOException x) {
                        return null;
                    }
                }
                return null;
            }
        }
        return null;
    }

    private static boolean isParentWritable(Path path) {
        Path parentPath = path.getParent();
        if (parentPath == null && !path.isAbsolute()) {
            parentPath = path.toAbsolutePath().getParent();
        }
        return parentPath != null && Files.isWritable(parentPath);
    }

    private static boolean lock(FileChannel lockFileChannel, boolean newFile) {
        boolean available = false;
        try {
            available = lockFileChannel.tryLock() != null;
        }
        catch (OverlappingFileLockException overlappingFileLockException) {
        }
        catch (IOException ioe) {
            available = newFile;
        }
        return available;
    }

    static {
        CLASSPATH = System.getProperty("org.graalvm.launcher.classpath");
    }

    private static final class LockableOutputStream
    extends OutputStream {
        private final OutputStream delegate;
        private final Path lockFile;
        private final FileChannel lockFileChannel;

        LockableOutputStream(OutputStream delegate, Path lockFile, FileChannel lockFileChannel) {
            this.delegate = delegate;
            this.lockFile = lockFile;
            this.lockFileChannel = lockFileChannel;
        }

        @Override
        public void write(int b) throws IOException {
            this.delegate.write(b);
        }

        @Override
        public void write(byte[] b) throws IOException {
            this.delegate.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.delegate.write(b, off, len);
        }

        @Override
        public void flush() throws IOException {
            this.delegate.flush();
        }

        @Override
        public void close() throws IOException {
            try {
                this.delegate.close();
            }
            finally {
                LockableOutputStream.unlock(this.lockFile, this.lockFileChannel);
            }
        }

        private static void unlock(Path lockFile, FileChannel lockFileChannel) {
            try {
                lockFileChannel.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            try {
                Files.delete(lockFile);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    private static final class ShellQuotes {
        private static final BitSet safeChars = new BitSet();

        private ShellQuotes() {
        }

        private static String quote(String str) {
            if (str.isEmpty()) {
                return "''";
            }
            for (int i = 0; i < str.length(); ++i) {
                if (safeChars.get(str.charAt(i))) continue;
                return "'" + str.replace("'", "'\"'\"'") + "'";
            }
            return str;
        }

        static {
            safeChars.set(97, 123);
            safeChars.set(65, 91);
            safeChars.set(43, 59);
            safeChars.set(64);
            safeChars.set(37);
            safeChars.set(95);
            safeChars.set(61);
        }
    }

    class Native {
        private WeakReference<OptionDescriptors> compilerOptionDescriptors;
        private WeakReference<OptionDescriptors> vmOptionDescriptors;

        Native() {
        }

        void maybeExec(List<String> args, boolean isPolyglot, Map<String, String> polyglotOptions, VMType defaultVmType) {
            assert (Launcher.isAOT());
            VMType vmType = null;
            boolean polyglot = false;
            ArrayList<String> jvmArgs = new ArrayList<String>();
            ArrayList<String> remainingArgs = new ArrayList<String>(args.size());
            Iterator<Map.Entry<String, String>> polyglotOptionsIterator = polyglotOptions.entrySet().iterator();
            while (polyglotOptionsIterator.hasNext()) {
                Map.Entry<String, String> entry = polyglotOptionsIterator.next();
                if (!entry.getKey().startsWith("jvm.")) continue;
                jvmArgs.add('-' + entry.getKey().substring(4));
                if (entry.getValue() != null && !entry.getValue().isEmpty()) {
                    jvmArgs.add(entry.getValue());
                }
                vmType = VMType.JVM;
                polyglotOptionsIterator.remove();
            }
            boolean jvmDotWarned = false;
            Iterator<String> iterator = args.iterator();
            ArrayList<String> vmOptions = new ArrayList<String>();
            while (iterator.hasNext()) {
                int eqIndex;
                String arg = iterator.next();
                if (arg.startsWith("--jvm.") && arg.length() > "--jvm.".length() || arg.equals("--jvm")) {
                    if (vmType == VMType.Native) {
                        throw Launcher.this.abort("'--jvm' and '--native' options can not be used together.");
                    }
                    if (Launcher.this.isStandalone()) {
                        if (arg.equals("--jvm")) {
                            throw Launcher.this.abort("'--jvm' is only supported when this launcher is part of a GraalVM.");
                        }
                        throw Launcher.this.abort("'--jvm.*' options are deprecated and only supported when this launcher is part of a GraalVM.");
                    }
                    vmType = VMType.JVM;
                    if (arg.equals("--jvm.help")) {
                        if (defaultVmType == VMType.JVM) {
                            Launcher.this.warn("'--jvm.help' is deprecated, use '--help:vm' instead.");
                        } else {
                            Launcher.this.warn("'--jvm.help' is deprecated, use '--jvm --help:vm' instead.");
                        }
                        remainingArgs.add("--help:vm");
                    } else if (arg.startsWith("--jvm.")) {
                        String jvmArg;
                        if (!jvmDotWarned) {
                            Launcher.this.warn("'--jvm.*' options are deprecated, use '--vm.*' instead.");
                            jvmDotWarned = true;
                        }
                        if ((jvmArg = arg.substring("--jvm.".length())).equals("classpath")) {
                            throw Launcher.this.abort("'--jvm.classpath' argument must be of the form '--jvm.classpath=<classpath>', not two separate arguments");
                        }
                        if (jvmArg.equals("cp")) {
                            throw Launcher.this.abort("'--jvm.cp' argument must be of the form '--jvm.cp=<classpath>', not two separate arguments");
                        }
                        if (jvmArg.startsWith("classpath=") || jvmArg.startsWith("cp=")) {
                            eqIndex = jvmArg.indexOf(61);
                            jvmArgs.add('-' + jvmArg.substring(0, eqIndex));
                            jvmArgs.add(jvmArg.substring(eqIndex + 1));
                        } else {
                            jvmArgs.add('-' + jvmArg);
                        }
                    }
                    iterator.remove();
                    continue;
                }
                if (arg.startsWith("--native.") && arg.length() > "--native.".length() || arg.equals("--native")) {
                    if (vmType == VMType.JVM) {
                        throw Launcher.this.abort("'--jvm' and '--native' options can not be used together.");
                    }
                    vmType = VMType.Native;
                    if (arg.equals("--native.help")) {
                        if (defaultVmType == VMType.Native) {
                            Launcher.this.warn("'--native.help' is deprecated, use '--help:vm' instead.");
                        } else {
                            Launcher.this.warn("'--native.help' is deprecated, use '--native --help:vm' instead.");
                        }
                        remainingArgs.add("--help:vm");
                    } else if (arg.startsWith("--native.")) {
                        if (!jvmDotWarned) {
                            Launcher.this.warn("'--native.*' options are deprecated, use '--vm.*' instead.");
                            jvmDotWarned = true;
                        }
                        this.setNativeOption(arg.substring("--native.".length()));
                    }
                    iterator.remove();
                    continue;
                }
                if (arg.startsWith("--vm.") && arg.length() > "--vm.".length()) {
                    String vmArg;
                    if (arg.equals("--vm.help")) {
                        Launcher.this.warn("'--vm.help' is deprecated, use '--help:vm' instead.");
                        remainingArgs.add("--help:vm");
                    }
                    if ((vmArg = arg.substring("--vm.".length())).equals("classpath")) {
                        throw Launcher.this.abort("'--vm.classpath' argument must be of the form '--vm.classpath=<classpath>', not two separate arguments");
                    }
                    if (vmArg.equals("cp")) {
                        throw Launcher.this.abort("'--vm.cp' argument must be of the form '--vm.cp=<classpath>', not two separate arguments");
                    }
                    if (vmArg.startsWith("classpath=") || vmArg.startsWith("cp=")) {
                        eqIndex = vmArg.indexOf(61);
                        jvmArgs.add('-' + vmArg.substring(0, eqIndex));
                        jvmArgs.add(vmArg.substring(eqIndex + 1));
                    } else {
                        vmOptions.add(vmArg);
                    }
                    iterator.remove();
                    continue;
                }
                if (arg.equals("--polyglot")) {
                    polyglot = true;
                    continue;
                }
                remainingArgs.add(arg);
            }
            if (vmType == null) {
                vmType = defaultVmType;
            }
            for (String vmOption : vmOptions) {
                if (vmType == VMType.JVM) {
                    jvmArgs.add('-' + vmOption);
                    continue;
                }
                assert (vmType == VMType.Native);
                this.setNativeOption(vmOption);
            }
            VMRuntime.initialize();
            if (vmType == VMType.JVM) {
                if (!isPolyglot && polyglot) {
                    remainingArgs.add(0, "--polyglot");
                }
                assert (!Launcher.this.isStandalone());
                this.execJVM(jvmArgs, remainingArgs, polyglotOptions);
            } else if (!isPolyglot && polyglot) {
                assert (jvmArgs.isEmpty());
                if (Launcher.this.isStandalone()) {
                    throw Launcher.this.abort("--polyglot option is only supported when this launcher is part of a GraalVM.");
                }
                this.execNativePolyglot(remainingArgs, polyglotOptions);
            }
        }

        private OptionDescriptors getCompilerOptions() {
            OptionDescriptors descriptors = null;
            if (this.compilerOptionDescriptors != null) {
                descriptors = (OptionDescriptors)this.compilerOptionDescriptors.get();
            }
            if (descriptors == null) {
                descriptors = RuntimeOptions.getOptions(EnumSet.of(RuntimeOptions.OptionClass.Compiler));
                this.compilerOptionDescriptors = new WeakReference<OptionDescriptors>(descriptors);
            }
            return descriptors;
        }

        private OptionDescriptors getVMOptions() {
            OptionDescriptors descriptors = null;
            if (this.vmOptionDescriptors != null) {
                descriptors = (OptionDescriptors)this.vmOptionDescriptors.get();
            }
            if (descriptors == null) {
                descriptors = RuntimeOptions.getOptions(EnumSet.of(RuntimeOptions.OptionClass.VM));
                this.vmOptionDescriptors = new WeakReference<OptionDescriptors>(descriptors);
            }
            return descriptors;
        }

        private void setNativeOption(String arg) {
            if (arg.startsWith("Dgraal.")) {
                this.setGraalStyleRuntimeOption(arg.substring("Dgraal.".length()));
            } else if (arg.startsWith("D")) {
                this.setSystemProperty(arg.substring("D".length()));
            } else if (arg.startsWith("XX:")) {
                this.setRuntimeOption(arg.substring("XX:".length()));
            } else if (arg.startsWith("X") && this.isXOption(arg)) {
                this.setXOption(arg.substring("X".length()));
            } else {
                throw Launcher.this.abort("Unrecognized vm option: '--vm." + arg + "'. Such arguments should start with '--vm.D', '--vm.XX:', or '--vm.X'");
            }
        }

        private void setGraalStyleRuntimeOption(String arg) {
            String value;
            String key;
            if (arg.startsWith("+") || arg.startsWith("-")) {
                throw Launcher.this.abort("Dgraal option must use <name>=<value> format, not +/- prefix");
            }
            int eqIdx = arg.indexOf(61);
            if (eqIdx < 0) {
                key = arg;
                value = "";
            } else {
                key = arg.substring(0, eqIdx);
                value = arg.substring(eqIdx + 1);
            }
            OptionDescriptor descriptor = this.getCompilerOptions().get(key);
            if (descriptor == null && (descriptor = this.getVMOptions().get(key)) != null) {
                if (this.isBooleanOption(descriptor)) {
                    Launcher.this.warn("VM options such as '%s' should be set with '--vm.XX:\u00b1%<s'.%nSupport for setting them with '--vm.Dgraal.%<s=<value>' is deprecated and will be removed.%n", key);
                } else {
                    Launcher.this.warn("VM options such as '%s' should be set with '--vm.XX:%<s=<value>'.%nSupport for setting them with '--vm.Dgraal.%<s=<value>' is deprecated and will be removed.%n", key);
                }
            }
            if (descriptor == null) {
                throw this.unknownOption(key);
            }
            try {
                RuntimeOptions.set((String)key, (Object)descriptor.getKey().getType().convert(value));
            }
            catch (IllegalArgumentException iae) {
                throw Launcher.this.abort("Invalid argument: '--vm." + arg + "': " + iae.getMessage());
            }
        }

        public void setSystemProperty(String arg) {
            String value;
            String key;
            int eqIdx = arg.indexOf(61);
            if (eqIdx < 0) {
                key = arg;
                value = "";
            } else {
                key = arg.substring(0, eqIdx);
                value = arg.substring(eqIdx + 1);
            }
            System.setProperty(key, value);
        }

        public void setRuntimeOption(String arg) {
            Object value;
            String key;
            int eqIdx = arg.indexOf(61);
            if (arg.startsWith("+") || arg.startsWith("-")) {
                key = arg.substring(1);
                if (eqIdx >= 0) {
                    throw Launcher.this.abort("Invalid argument: '--vm." + arg + "': Use either +/- or =, but not both");
                }
                OptionDescriptor descriptor = this.getVMOptionDescriptor(key);
                if (!this.isBooleanOption(descriptor)) {
                    throw Launcher.this.abort("Invalid argument: " + key + " is not a boolean option, set it with --vm.XX:" + key + "=<value>.");
                }
                value = arg.startsWith("+");
            } else if (eqIdx > 0) {
                key = arg.substring(0, eqIdx);
                OptionDescriptor descriptor = this.getVMOptionDescriptor(key);
                if (this.isBooleanOption(descriptor)) {
                    throw Launcher.this.abort("Boolean option '" + key + "' must be set with +/- prefix, not <name>=<value> format.");
                }
                try {
                    value = descriptor.getKey().getType().convert(arg.substring(eqIdx + 1));
                }
                catch (IllegalArgumentException iae) {
                    throw Launcher.this.abort("Invalid argument: '--vm." + arg + "': " + iae.getMessage());
                }
            } else {
                throw Launcher.this.abort("Invalid argument: '--vm." + arg + "'. Prefix boolean options with + or -, suffix other options with <name>=<value>");
            }
            RuntimeOptions.set((String)key, (Object)value);
        }

        private OptionDescriptor getVMOptionDescriptor(String key) {
            OptionDescriptor descriptor = this.getVMOptions().get(key);
            if (descriptor == null && (descriptor = this.getCompilerOptions().get(key)) != null) {
                Launcher.this.warn("compiler options such as '%s' should be set with '--vm.Dgraal.%<s=<value>'.%nSupport for setting them with '--vm.XX:...' is deprecated and will be removed.%n", key);
            }
            if (descriptor == null) {
                throw this.unknownOption(key);
            }
            return descriptor;
        }

        private boolean isXOption(String arg) {
            return arg.startsWith("Xmn") || arg.startsWith("Xms") || arg.startsWith("Xmx") || arg.startsWith("Xss");
        }

        private void setXOption(String arg) {
            try {
                RuntimeOptions.set((String)arg, null);
            }
            catch (RuntimeException re) {
                throw Launcher.this.abort("Invalid argument: '--vm.X" + arg + "' does not specify a valid number.");
            }
        }

        private boolean isBooleanOption(OptionDescriptor descriptor) {
            return descriptor.getKey().getType().equals(OptionType.defaultType(Boolean.class));
        }

        private AbortException unknownOption(String key) {
            throw Launcher.this.abort("Unknown native option: " + key + ". Use --help:vm to list available options.");
        }

        private void printNativeHelp() {
            System.out.println("Native VM options:");
            TreeMap<String, OptionDescriptor> sortedOptions = new TreeMap<String, OptionDescriptor>();
            for (OptionDescriptor descriptor : this.getVMOptions()) {
                sortedOptions.put(descriptor.getName(), descriptor);
            }
            for (Map.Entry entry : sortedOptions.entrySet()) {
                OptionDescriptor descriptor = (OptionDescriptor)entry.getValue();
                String helpMsg = descriptor.getHelp();
                if (this.isBooleanOption(descriptor)) {
                    Boolean val = (Boolean)descriptor.getKey().getDefaultValue();
                    if (helpMsg.length() != 0) {
                        helpMsg = helpMsg + ' ';
                    }
                    helpMsg = val == null || val == false ? helpMsg + "Default: - (disabled)." : helpMsg + "Default: + (enabled).";
                    Launcher.printOption("--vm.XX:\u00b1" + (String)entry.getKey(), helpMsg);
                    continue;
                }
                Object def = descriptor.getKey().getDefaultValue();
                if (def instanceof String) {
                    def = "\"" + def + "\"";
                }
                Launcher.printOption("--vm.XX:" + (String)entry.getKey() + "=" + def, helpMsg);
            }
            this.printCompilerOptions();
            Launcher.printBasicNativeHelp();
        }

        private void printCompilerOptions() {
            System.out.println("Compiler options:");
            TreeMap<String, OptionDescriptor> sortedOptions = new TreeMap<String, OptionDescriptor>();
            for (OptionDescriptor descriptor : this.getCompilerOptions()) {
                sortedOptions.put(descriptor.getName(), descriptor);
            }
            for (Map.Entry entry : sortedOptions.entrySet()) {
                OptionDescriptor descriptor = (OptionDescriptor)entry.getValue();
                String helpMsg = descriptor.getHelp();
                Object def = descriptor.getKey().getDefaultValue();
                if (def instanceof String) {
                    def = '\"' + (String)def + '\"';
                }
                Launcher.printOption("--vm.Dgraal." + (String)entry.getKey() + "=" + def, helpMsg);
            }
        }

        private void execNativePolyglot(List<String> args, Map<String, String> polyglotOptions) {
            ArrayList<String> command = new ArrayList<String>(args.size() + (polyglotOptions == null ? 0 : polyglotOptions.size()) + 3);
            Path executable = this.getGraalVMBinaryPath("polyglot");
            command.add("--native");
            command.add("--use-launcher");
            command.add(Launcher.this.getMainClass());
            Launcher.serializePolyglotOptions(polyglotOptions, command);
            command.addAll(args);
            this.exec(executable, command);
        }

        private void execJVM(List<String> jvmArgs, List<String> args, Map<String, String> polyglotOptions) {
            ArrayList<String> command = new ArrayList<String>(jvmArgs.size() + args.size() + (polyglotOptions == null ? 0 : polyglotOptions.size()) + 4);
            Path executable = this.getGraalVMBinaryPath("java");
            String classpath = this.getClasspath(jvmArgs);
            if (classpath != null) {
                command.add("-classpath");
                command.add(classpath);
            }
            command.addAll(jvmArgs);
            command.add(Launcher.this.getMainClass());
            Launcher.serializePolyglotOptions(polyglotOptions, command);
            command.addAll(args);
            this.exec(executable, command);
        }

        private String getClasspath(List<String> jvmArgs) {
            assert (Launcher.isAOT());
            assert (CLASSPATH != null);
            StringBuilder sb = new StringBuilder();
            if (!CLASSPATH.isEmpty()) {
                Path graalVMHome = Launcher.this.getGraalVMHome();
                if (graalVMHome == null) {
                    throw Launcher.this.abort("Can not resolve classpath: could not get GraalVM home");
                }
                for (String entry : CLASSPATH.split(File.pathSeparator)) {
                    Path resolved = graalVMHome.resolve(entry);
                    if (Launcher.this.isVerbose() && !Files.exists(resolved, new LinkOption[0])) {
                        Launcher.this.warn("%s does not exist", resolved);
                    }
                    sb.append(resolved);
                    sb.append(File.pathSeparatorChar);
                }
            }
            String classpathFromArgs = null;
            Iterator<String> iterator = jvmArgs.iterator();
            while (iterator.hasNext()) {
                String jvmArg = iterator.next();
                if ((jvmArg.equals("-cp") || jvmArg.equals("-classpath")) && iterator.hasNext()) {
                    iterator.remove();
                    classpathFromArgs = iterator.next();
                    iterator.remove();
                }
                if (!jvmArg.startsWith("-Djava.class.path=")) continue;
                iterator.remove();
                classpathFromArgs = jvmArg.substring("-Djava.class.path=".length());
            }
            if (classpathFromArgs != null) {
                sb.append(classpathFromArgs);
                sb.append(File.pathSeparatorChar);
            }
            if (sb.length() == 0) {
                return null;
            }
            return sb.substring(0, sb.length() - 1);
        }

        private Path getGraalVMBinaryPath(String binaryName) {
            String executableName = Launcher.this.executableName(binaryName);
            Path graalVMHome = Launcher.this.getGraalVMHome();
            if (graalVMHome == null) {
                throw Launcher.this.abort("Can not exec to GraalVM binary: could not find GraalVM home");
            }
            Path jdkBin = graalVMHome.resolve("bin").resolve(executableName);
            if (Files.exists(jdkBin, new LinkOption[0])) {
                return jdkBin;
            }
            return graalVMHome.resolve("jre").resolve("bin").resolve(executableName);
        }

        private void exec(Path executable, List<String> command) {
            assert (Launcher.isAOT());
            if (Launcher.this.isVerbose()) {
                StringBuilder sb = this.formatExec(executable, command);
                System.err.print(sb.toString());
            }
            String[] argv = new String[command.size() + 1];
            int i = 0;
            Path filename = executable.getFileName();
            if (filename == null) {
                throw Launcher.this.abort(String.format("Cannot determine execute filename from path %s", filename));
            }
            argv[i++] = filename.toString();
            for (String arg : command) {
                argv[i++] = arg;
            }
            ProcessProperties.exec((Path)executable, (String[])argv);
        }

        private StringBuilder formatExec(Path executable, List<String> command) {
            StringBuilder sb = new StringBuilder("exec: ");
            sb.append(executable);
            for (String arg : command) {
                sb.append(' ');
                sb.append(ShellQuotes.quote(arg));
            }
            sb.append(System.lineSeparator());
            return sb;
        }
    }

    static enum OS {
        Darwin,
        Linux,
        Solaris,
        Windows;

        private static final OS current;

        private static OS findCurrent() {
            String name = System.getProperty("os.name");
            if (name.equals("Linux")) {
                return Linux;
            }
            if (name.equals("SunOS")) {
                return Solaris;
            }
            if (name.equals("Mac OS X") || name.equals("Darwin")) {
                return Darwin;
            }
            if (name.startsWith("Windows")) {
                return Windows;
            }
            throw new IllegalArgumentException("unknown OS: " + name);
        }

        public static OS getCurrent() {
            return current;
        }

        static {
            current = OS.findCurrent();
        }
    }

    private static final class PrintableOption
    implements Comparable<PrintableOption> {
        final String option;
        final String description;

        private PrintableOption(String option, String description) {
            this.option = option;
            this.description = description;
        }

        @Override
        public int compareTo(PrintableOption o) {
            return this.option.compareTo(o.option);
        }
    }

    protected static class AbortException
    extends RuntimeException {
        static final long serialVersionUID = 4681646279864737876L;
        private final int exitCode;

        AbortException(String message, int exitCode) {
            super(message);
            this.exitCode = exitCode;
        }

        AbortException(Throwable cause, int exitCode) {
            super(null, cause);
            this.exitCode = exitCode;
        }

        int getExitCode() {
            return this.exitCode;
        }

        @Override
        public final Throwable fillInStackTrace() {
            return this;
        }
    }

    protected static enum VersionAction {
        None,
        PrintAndExit,
        PrintAndContinue;

    }

    public static enum VMType {
        Native,
        JVM;

    }
}

