/*
 * Copyright (c) 2019 Valve Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *
 * This file was generated by aco_builder_h.py
 */

#ifndef _ACO_BUILDER_
#define _ACO_BUILDER_

#include "aco_ir.h"
#include "util/u_math.h"
#include "util/bitscan.h"

namespace aco {
enum dpp_ctrl {
    _dpp_quad_perm = 0x000,
    _dpp_row_sl = 0x100,
    _dpp_row_sr = 0x110,
    _dpp_row_rr = 0x120,
    dpp_wf_sl1 = 0x130,
    dpp_wf_rl1 = 0x134,
    dpp_wf_sr1 = 0x138,
    dpp_wf_rr1 = 0x13C,
    dpp_row_mirror = 0x140,
    dpp_row_half_mirror = 0x141,
    dpp_row_bcast15 = 0x142,
    dpp_row_bcast31 = 0x143
};

inline dpp_ctrl
dpp_quad_perm(unsigned lane0, unsigned lane1, unsigned lane2, unsigned lane3)
{
    assert(lane0 < 4 && lane1 < 4 && lane2 < 4 && lane3 < 4);
    return (dpp_ctrl)(lane0 | (lane1 << 2) | (lane2 << 4) | (lane3 << 6));
}

inline dpp_ctrl
dpp_row_sl(unsigned amount)
{
    assert(amount > 0 && amount < 16);
    return (dpp_ctrl)(((unsigned) _dpp_row_sl) | amount);
}

inline dpp_ctrl
dpp_row_sr(unsigned amount)
{
    assert(amount > 0 && amount < 16);
    return (dpp_ctrl)(((unsigned) _dpp_row_sr) | amount);
}

inline dpp_ctrl
dpp_row_rr(unsigned amount)
{
    assert(amount > 0 && amount < 16);
    return (dpp_ctrl)(((unsigned) _dpp_row_rr) | amount);
}

inline unsigned
ds_pattern_bitmode(unsigned and_mask, unsigned or_mask, unsigned xor_mask)
{
    assert(and_mask < 32 && or_mask < 32 && xor_mask < 32);
    return and_mask | (or_mask << 5) | (xor_mask << 10);
}

aco_ptr<Instruction> create_s_mov(Definition dst, Operand src);

enum sendmsg {
   sendmsg_none = 0,
   _sendmsg_gs = 2,
   _sendmsg_gs_done = 3,
   sendmsg_save_wave = 4,
   sendmsg_stall_wave_gen = 5,
   sendmsg_halt_waves = 6,
   sendmsg_ordered_ps_done = 7,
   sendmsg_early_prim_dealloc = 8,
   sendmsg_gs_alloc_req = 9,
   sendmsg_id_mask = 0xf,
};

inline sendmsg
sendmsg_gs(bool cut, bool emit, unsigned stream)
{
    assert(stream < 4);
    return (sendmsg)((unsigned)_sendmsg_gs | (cut << 4) | (emit << 5) | (stream << 8));
}

inline sendmsg
sendmsg_gs_done(bool cut, bool emit, unsigned stream)
{
    assert(stream < 4);
    return (sendmsg)((unsigned)_sendmsg_gs_done | (cut << 4) | (emit << 5) | (stream << 8));
}

class Builder {
public:
   struct Result {
      Instruction *instr;

      Result(Instruction *instr_) : instr(instr_) {}

      operator Instruction *() const {
         return instr;
      }

      operator Temp() const {
         return instr->definitions[0].getTemp();
      }

      operator Operand() const {
         return Operand((Temp)*this);
      }

      Definition& def(unsigned index) const {
         return instr->definitions[index];
      }

      aco_ptr<Instruction> get_ptr() const {
        return aco_ptr<Instruction>(instr);
      }
   };

   struct Op {
      Operand op;
      Op(Temp tmp) : op(tmp) {}
      Op(Operand op_) : op(op_) {}
      Op(Result res) : op((Temp)res) {}
   };

   enum WaveSpecificOpcode {
      s_cselect = (unsigned) aco_opcode::s_cselect_b64,
      s_cmp_lg = (unsigned) aco_opcode::s_cmp_lg_u64,
      s_and = (unsigned) aco_opcode::s_and_b64,
      s_andn2 = (unsigned) aco_opcode::s_andn2_b64,
      s_or = (unsigned) aco_opcode::s_or_b64,
      s_orn2 = (unsigned) aco_opcode::s_orn2_b64,
      s_not = (unsigned) aco_opcode::s_not_b64,
      s_mov = (unsigned) aco_opcode::s_mov_b64,
      s_wqm = (unsigned) aco_opcode::s_wqm_b64,
      s_and_saveexec = (unsigned) aco_opcode::s_and_saveexec_b64,
      s_or_saveexec = (unsigned) aco_opcode::s_or_saveexec_b64,
      s_xnor = (unsigned) aco_opcode::s_xnor_b64,
      s_xor = (unsigned) aco_opcode::s_xor_b64,
      s_bcnt1_i32 = (unsigned) aco_opcode::s_bcnt1_i32_b64,
      s_bitcmp1 = (unsigned) aco_opcode::s_bitcmp1_b64,
      s_ff1_i32 = (unsigned) aco_opcode::s_ff1_i32_b64,
      s_flbit_i32 = (unsigned) aco_opcode::s_flbit_i32_b64,
      s_lshl = (unsigned) aco_opcode::s_lshl_b64,
   };

   Program *program;
   bool use_iterator;
   bool start; // only when use_iterator == false
   RegClass lm;

   std::vector<aco_ptr<Instruction>> *instructions;
   std::vector<aco_ptr<Instruction>>::iterator it;
   bool is_precise = false;
   bool is_nuw = false;

   Builder(Program *pgm) : program(pgm), use_iterator(false), start(false), lm(pgm ? pgm->lane_mask : s2), instructions(NULL) {}
   Builder(Program *pgm, Block *block) : program(pgm), use_iterator(false), start(false), lm(pgm ? pgm->lane_mask : s2), instructions(&block->instructions) {}
   Builder(Program *pgm, std::vector<aco_ptr<Instruction>> *instrs) : program(pgm), use_iterator(false), start(false), lm(pgm ? pgm->lane_mask : s2), instructions(instrs) {}

   Builder precise() const {
      Builder res = *this;
      res.is_precise = true;
      return res;
   };

   Builder nuw() const {
      Builder res = *this;
      res.is_nuw = true;
      return res;
   }

   void moveEnd(Block *block) {
      instructions = &block->instructions;
   }

   void reset() {
      use_iterator = false;
      start = false;
      instructions = NULL;
   }

   void reset(Block *block) {
      use_iterator = false;
      start = false;
      instructions = &block->instructions;
   }

   void reset(std::vector<aco_ptr<Instruction>> *instrs) {
      use_iterator = false;
      start = false;
      instructions = instrs;
   }

   void reset(std::vector<aco_ptr<Instruction>> *instrs, std::vector<aco_ptr<Instruction>>::iterator instr_it) {
      use_iterator = true;
      start = false;
      instructions = instrs;
      it = instr_it;
   }

   Result insert(aco_ptr<Instruction> instr) {
      Instruction *instr_ptr = instr.get();
      if (instructions) {
         if (use_iterator) {
            it = instructions->emplace(it, std::move(instr));
            it = std::next(it);
         } else if (!start) {
            instructions->emplace_back(std::move(instr));
         } else {
            instructions->emplace(instructions->begin(), std::move(instr));
         }
      }
      return Result(instr_ptr);
   }

   Result insert(Instruction* instr) {
      if (instructions) {
         if (use_iterator) {
            it = instructions->emplace(it, aco_ptr<Instruction>(instr));
            it = std::next(it);
         } else if (!start) {
            instructions->emplace_back(aco_ptr<Instruction>(instr));
         } else {
            instructions->emplace(instructions->begin(), aco_ptr<Instruction>(instr));
         }
      }
      return Result(instr);
   }

   Temp tmp(RegClass rc) {
      return program->allocateTmp(rc);
   }

   Temp tmp(RegType type, unsigned size) {
      return tmp(RegClass(type, size));
   }

   Definition def(RegClass rc) {
      return Definition(program->allocateTmp(rc));
   }

   Definition def(RegType type, unsigned size) {
      return def(RegClass(type, size));
   }

   Definition def(RegClass rc, PhysReg reg) {
      return Definition(program->allocateId(rc), reg, rc);
   }

   inline aco_opcode w64or32(WaveSpecificOpcode opcode) const {
      if (program->wave_size == 64)
         return (aco_opcode) opcode;

      switch (opcode) {
      case s_cselect:
         return aco_opcode::s_cselect_b32;
      case s_cmp_lg:
         return aco_opcode::s_cmp_lg_u32;
      case s_and:
         return aco_opcode::s_and_b32;
      case s_andn2:
         return aco_opcode::s_andn2_b32;
      case s_or:
         return aco_opcode::s_or_b32;
      case s_orn2:
         return aco_opcode::s_orn2_b32;
      case s_not:
         return aco_opcode::s_not_b32;
      case s_mov:
         return aco_opcode::s_mov_b32;
      case s_wqm:
         return aco_opcode::s_wqm_b32;
      case s_and_saveexec:
         return aco_opcode::s_and_saveexec_b32;
      case s_or_saveexec:
         return aco_opcode::s_or_saveexec_b32;
      case s_xnor:
         return aco_opcode::s_xnor_b32;
      case s_xor:
         return aco_opcode::s_xor_b32;
      case s_bcnt1_i32:
         return aco_opcode::s_bcnt1_i32_b32;
      case s_bitcmp1:
         return aco_opcode::s_bitcmp1_b32;
      case s_ff1_i32:
         return aco_opcode::s_ff1_i32_b32;
      case s_flbit_i32:
         return aco_opcode::s_flbit_i32_b32;
      case s_lshl:
         return aco_opcode::s_lshl_b32;
      default:
         unreachable("Unsupported wave specific opcode.");
      }
   }

   Operand m0(Temp tmp) {
       Operand op(tmp);
       op.setFixed(aco::m0);
       return op;
   }

   Definition m0(Definition def) {
       def.setFixed(aco::m0);
       return def;
   }

   Definition hint_m0(Definition def) {
       def.setHint(aco::m0);
       return def;
   }

   Definition hint_m0(RegClass rc) {
       return hint_m0(def(rc));
   }

   Operand vcc(Temp tmp) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(tmp.type() == RegType::sgpr && tmp.bytes() <= 8);
       Operand op(tmp);
       op.setFixed(aco::vcc);
       return op;
   }

   Definition vcc(Definition def) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(def.regClass().type() == RegType::sgpr && def.bytes() <= 8);
       def.setFixed(aco::vcc);
       return def;
   }

   Definition hint_vcc(Definition def) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(def.regClass().type() == RegType::sgpr && def.bytes() <= 8);
       def.setHint(aco::vcc);
       return def;
   }

   Definition hint_vcc(RegClass rc) {
       return hint_vcc(def(rc));
   }

   Operand exec(Temp tmp) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(tmp.type() == RegType::sgpr && tmp.bytes() <= 8);
       Operand op(tmp);
       op.setFixed(aco::exec);
       return op;
   }

   Definition exec(Definition def) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(def.regClass().type() == RegType::sgpr && def.bytes() <= 8);
       def.setFixed(aco::exec);
       return def;
   }

   Definition hint_exec(Definition def) {
          //vcc_hi and exec_hi can still be used in wave32
          assert(def.regClass().type() == RegType::sgpr && def.bytes() <= 8);
       def.setHint(aco::exec);
       return def;
   }

   Definition hint_exec(RegClass rc) {
       return hint_exec(def(rc));
   }

   Operand scc(Temp tmp) {
       Operand op(tmp);
       op.setFixed(aco::scc);
       return op;
   }

   Definition scc(Definition def) {
       def.setFixed(aco::scc);
       return def;
   }

   Definition hint_scc(Definition def) {
       def.setHint(aco::scc);
       return def;
   }

   Definition hint_scc(RegClass rc) {
       return hint_scc(def(rc));
   }


   Operand set16bit(Operand op) {
       op.set16bit(true);
       return op;
   }

   Operand set24bit(Operand op) {
       op.set24bit(true);
       return op;
   }

   /* hand-written helpers */
   Temp as_uniform(Op op)
   {
      assert(op.op.isTemp());
      if (op.op.getTemp().type() == RegType::vgpr)
         return pseudo(aco_opcode::p_as_uniform, def(RegType::sgpr, op.op.size()), op);
      else
         return op.op.getTemp();
   }

   Result v_mul_imm(Definition dst, Temp tmp, uint32_t imm, bool bits24=false)
   {
      assert(tmp.type() == RegType::vgpr);
      bool has_lshl_add = program->chip_class >= GFX9;
      /* v_mul_lo_u32 has 1.6x the latency of most VALU on GFX10 (8 vs 5 cycles),
       * compared to 4x the latency on <GFX10. */
      unsigned mul_cost = program->chip_class >= GFX10 ? 1 : (4 + Operand(imm).isLiteral());
      if (imm == 0) {
         return copy(dst, Operand(0u));
      } else if (imm == 1) {
         return copy(dst, Operand(tmp));
      } else if (util_is_power_of_two_or_zero(imm)) {
         return vop2(aco_opcode::v_lshlrev_b32, dst, Operand((uint32_t)ffs(imm) - 1u), tmp);
      } else if (bits24) {
        return vop2(aco_opcode::v_mul_u32_u24, dst, Operand(imm), tmp);
      } else if (util_is_power_of_two_nonzero(imm - 1u)) {
         return vadd32(dst, vop2(aco_opcode::v_lshlrev_b32, def(v1), Operand((uint32_t)ffs(imm - 1u) - 1u), tmp), tmp);
      } else if (mul_cost > 2 && util_is_power_of_two_nonzero(imm + 1u)) {
         return vsub32(dst, vop2(aco_opcode::v_lshlrev_b32, def(v1), Operand((uint32_t)ffs(imm + 1u) - 1u), tmp), tmp);
      }

      unsigned instrs_required = util_bitcount(imm);
      if (!has_lshl_add) {
         instrs_required = util_bitcount(imm) - (imm & 0x1); /* shifts */
         instrs_required += util_bitcount(imm) - 1; /* additions */
      }
      if (instrs_required < mul_cost) {
         Result res(NULL);
         Temp cur;
         while (imm) {
            unsigned shift = u_bit_scan(&imm);
            Definition tmp_dst = imm ? def(v1) : dst;

            if (shift && cur.id())
               res = vadd32(Definition(tmp_dst), vop2(aco_opcode::v_lshlrev_b32, def(v1), Operand(shift), tmp), cur);
            else if (shift)
               res = vop2(aco_opcode::v_lshlrev_b32, Definition(tmp_dst), Operand(shift), tmp);
            else if (cur.id())
               res = vadd32(Definition(tmp_dst), tmp, cur);
            else
               tmp_dst = Definition(tmp);

            cur = tmp_dst.getTemp();
         }
         return res;
      }

      Temp imm_tmp = copy(def(s1), Operand(imm));
      return vop3(aco_opcode::v_mul_lo_u32, dst, imm_tmp, tmp);
   }

   Result v_mul24_imm(Definition dst, Temp tmp, uint32_t imm)
   {
      return v_mul_imm(dst, tmp, imm, true);
   }

   Result copy(Definition dst, Op op) {
      return pseudo(aco_opcode::p_parallelcopy, dst, op);
   }

   Result vadd32(Definition dst, Op a, Op b, bool carry_out=false, Op carry_in=Op(Operand(s2)), bool post_ra=false) {
      if (!b.op.isTemp() || b.op.regClass().type() != RegType::vgpr)
         std::swap(a, b);
      if (!post_ra && (!b.op.hasRegClass() || b.op.regClass().type() == RegType::sgpr))
         b = copy(def(v1), b);

      if (!carry_in.op.isUndefined())
         return vop2(aco_opcode::v_addc_co_u32, Definition(dst), hint_vcc(def(lm)), a, b, carry_in);
      else if (program->chip_class >= GFX10 && carry_out)
         return vop3(aco_opcode::v_add_co_u32_e64, Definition(dst), def(lm), a, b);
      else if (program->chip_class < GFX9 || carry_out)
         return vop2(aco_opcode::v_add_co_u32, Definition(dst), hint_vcc(def(lm)), a, b);
      else
         return vop2(aco_opcode::v_add_u32, Definition(dst), a, b);
   }

   Result vsub32(Definition dst, Op a, Op b, bool carry_out=false, Op borrow=Op(Operand(s2)))
   {
      if (!borrow.op.isUndefined() || program->chip_class < GFX9)
         carry_out = true;

      bool reverse = !b.op.isTemp() || b.op.regClass().type() != RegType::vgpr;
      if (reverse)
         std::swap(a, b);
      if (!b.op.hasRegClass() || b.op.regClass().type() == RegType::sgpr)
         b = copy(def(v1), b);

      aco_opcode op;
      Temp carry;
      if (carry_out) {
         carry = tmp(s2);
         if (borrow.op.isUndefined())
            op = reverse ? aco_opcode::v_subrev_co_u32 : aco_opcode::v_sub_co_u32;
         else
            op = reverse ? aco_opcode::v_subbrev_co_u32 : aco_opcode::v_subb_co_u32;
      } else {
         op = reverse ? aco_opcode::v_subrev_u32 : aco_opcode::v_sub_u32;
      }
      bool vop3 = false;
      if (program->chip_class >= GFX10 && op == aco_opcode::v_subrev_co_u32) {
        vop3 = true;
        op = aco_opcode::v_subrev_co_u32_e64;
      } else if (program->chip_class >= GFX10 && op == aco_opcode::v_sub_co_u32) {
        vop3 = true;
        op = aco_opcode::v_sub_co_u32_e64;
      }

      int num_ops = borrow.op.isUndefined() ? 2 : 3;
      int num_defs = carry_out ? 2 : 1;
      aco_ptr<Instruction> sub;
      if (vop3)
        sub.reset(create_instruction<VOP3_instruction>(op, Format::VOP3, num_ops, num_defs));
      else
        sub.reset(create_instruction<VOP2_instruction>(op, Format::VOP2, num_ops, num_defs));
      sub->operands[0] = a.op;
      sub->operands[1] = b.op;
      if (!borrow.op.isUndefined())
         sub->operands[2] = borrow.op;
      sub->definitions[0] = dst;
      if (carry_out) {
         sub->definitions[1] = Definition(carry);
         sub->definitions[1].setHint(aco::vcc);
      }
      return insert(std::move(sub));
   }

   Result readlane(Definition dst, Op vsrc, Op lane)
   {
      if (program->chip_class >= GFX8)
         return vop3(aco_opcode::v_readlane_b32_e64, dst, vsrc, lane);
      else
         return vop2(aco_opcode::v_readlane_b32, dst, vsrc, lane);
   }
   Result writelane(Definition dst, Op val, Op lane, Op vsrc) {
      if (program->chip_class >= GFX8)
         return vop3(aco_opcode::v_writelane_b32_e64, dst, val, lane, vsrc);
      else
         return vop2(aco_opcode::v_writelane_b32, dst, val, lane, vsrc);
   }
        
   Result pseudo(aco_opcode opcode)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 0, 0);
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 1, 0);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 2, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1, Op op2)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 3, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 4, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 5, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 0, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 4, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 5, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 0, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 1, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 3, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, Op op3)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 4, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 5, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 0, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 1, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 2, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 3, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2, Op op3)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 4, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 5, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 0, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setNUW(is_nuw);
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 1, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 2, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1, Op op2)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 3, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1, Op op2, Op op3)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 4, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Op op0, Op op1, Op op2, Op op3, Op op4)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 5, 4);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Definition def3, Definition def4, Definition def5, Definition def6, Definition def7, Op op0)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 1, 8);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->definitions[3] = def3;
            instr->definitions[3].setPrecise(is_precise);
            instr->definitions[3].setNUW(is_nuw);
            instr->definitions[4] = def4;
            instr->definitions[4].setPrecise(is_precise);
            instr->definitions[4].setNUW(is_nuw);
            instr->definitions[5] = def5;
            instr->definitions[5].setPrecise(is_precise);
            instr->definitions[5].setNUW(is_nuw);
            instr->definitions[6] = def6;
            instr->definitions[6].setPrecise(is_precise);
            instr->definitions[6].setNUW(is_nuw);
            instr->definitions[7] = def7;
            instr->definitions[7].setPrecise(is_precise);
            instr->definitions[7].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   Result pseudo(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6, Op op7)
   {
      Pseudo_instruction *instr = create_instruction<Pseudo_instruction>(opcode, (Format)((int)Format::PSEUDO), 8, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;
            instr->operands[7] = op7.op;
            
       
      return insert(instr);
   }

        
   Result sop1(aco_opcode opcode, Op op0)
   {
      SOP1_instruction *instr = create_instruction<SOP1_instruction>(opcode, (Format)((int)Format::SOP1), 1, 0);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Op op0)
   {
       return sop1(w64or32(opcode), op0);
   }

        
   Result sop1(aco_opcode opcode, Definition def0)
   {
      SOP1_instruction *instr = create_instruction<SOP1_instruction>(opcode, (Format)((int)Format::SOP1), 0, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Definition def0)
   {
       return sop1(w64or32(opcode), def0);
   }

        
   Result sop1(aco_opcode opcode, Definition def0, Op op0)
   {
      SOP1_instruction *instr = create_instruction<SOP1_instruction>(opcode, (Format)((int)Format::SOP1), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Definition def0, Op op0)
   {
       return sop1(w64or32(opcode), def0, op0);
   }

        
   Result sop1(aco_opcode opcode, Definition def0, Definition def1, Op op0)
   {
      SOP1_instruction *instr = create_instruction<SOP1_instruction>(opcode, (Format)((int)Format::SOP1), 1, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Definition def0, Definition def1, Op op0)
   {
       return sop1(w64or32(opcode), def0, def1, op0);
   }

        
   Result sop1(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1)
   {
      SOP1_instruction *instr = create_instruction<SOP1_instruction>(opcode, (Format)((int)Format::SOP1), 2, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   inline Result sop1(WaveSpecificOpcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1)
   {
       return sop1(w64or32(opcode), def0, def1, def2, op0, op1);
   }

        
   Result sop2(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      SOP2_instruction *instr = create_instruction<SOP2_instruction>(opcode, (Format)((int)Format::SOP2), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   inline Result sop2(WaveSpecificOpcode opcode, Definition def0, Op op0, Op op1)
   {
       return sop2(w64or32(opcode), def0, op0, op1);
   }

        
   Result sop2(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      SOP2_instruction *instr = create_instruction<SOP2_instruction>(opcode, (Format)((int)Format::SOP2), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   inline Result sop2(WaveSpecificOpcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
       return sop2(w64or32(opcode), def0, op0, op1, op2);
   }

        
   Result sop2(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      SOP2_instruction *instr = create_instruction<SOP2_instruction>(opcode, (Format)((int)Format::SOP2), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   inline Result sop2(WaveSpecificOpcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
       return sop2(w64or32(opcode), def0, def1, op0, op1);
   }

        
   Result sop2(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      SOP2_instruction *instr = create_instruction<SOP2_instruction>(opcode, (Format)((int)Format::SOP2), 3, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   inline Result sop2(WaveSpecificOpcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
       return sop2(w64or32(opcode), def0, def1, op0, op1, op2);
   }

        
   Result sopk(aco_opcode opcode, uint16_t imm)
   {
      SOPK_instruction *instr = create_instruction<SOPK_instruction>(opcode, (Format)((int)Format::SOPK), 0, 0);
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Op op0, uint16_t imm)
   {
      SOPK_instruction *instr = create_instruction<SOPK_instruction>(opcode, (Format)((int)Format::SOPK), 1, 0);
            instr->operands[0] = op0.op;
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Definition def0, uint16_t imm)
   {
      SOPK_instruction *instr = create_instruction<SOPK_instruction>(opcode, (Format)((int)Format::SOPK), 0, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Definition def0, Op op0, uint16_t imm)
   {
      SOPK_instruction *instr = create_instruction<SOPK_instruction>(opcode, (Format)((int)Format::SOPK), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Definition def0, Definition def1, uint16_t imm)
   {
      SOPK_instruction *instr = create_instruction<SOPK_instruction>(opcode, (Format)((int)Format::SOPK), 0, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopk(aco_opcode opcode, Definition def0, Definition def1, Op op0, uint16_t imm)
   {
      SOPK_instruction *instr = create_instruction<SOPK_instruction>(opcode, (Format)((int)Format::SOPK), 1, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopp(aco_opcode opcode, uint32_t block=-1, uint32_t imm=0)
   {
      SOPP_instruction *instr = create_instruction<SOPP_instruction>(opcode, (Format)((int)Format::SOPP), 0, 0);
      instr->block = block;
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopp(aco_opcode opcode, Op op0, uint32_t block=-1, uint32_t imm=0)
   {
      SOPP_instruction *instr = create_instruction<SOPP_instruction>(opcode, (Format)((int)Format::SOPP), 1, 0);
            instr->operands[0] = op0.op;
      instr->block = block;
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopp(aco_opcode opcode, Definition def0, uint32_t block=-1, uint32_t imm=0)
   {
      SOPP_instruction *instr = create_instruction<SOPP_instruction>(opcode, (Format)((int)Format::SOPP), 0, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
      instr->block = block;
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopp(aco_opcode opcode, Definition def0, Op op0, uint32_t block=-1, uint32_t imm=0)
   {
      SOPP_instruction *instr = create_instruction<SOPP_instruction>(opcode, (Format)((int)Format::SOPP), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
      instr->block = block;
      instr->imm = imm;
            
       
      return insert(instr);
   }

        
   Result sopc(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      SOPC_instruction *instr = create_instruction<SOPC_instruction>(opcode, (Format)((int)Format::SOPC), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   inline Result sopc(WaveSpecificOpcode opcode, Definition def0, Op op0, Op op1)
   {
       return sopc(w64or32(opcode), def0, op0, op1);
   }

        
   Result smem(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, memory_sync_info sync=memory_sync_info(), bool glc=false, bool dlc=false, bool nv=false)
   {
      SMEM_instruction *instr = create_instruction<SMEM_instruction>(opcode, (Format)((int)Format::SMEM), 4, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
      instr->sync = sync;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Op op0, Op op1, Op op2, memory_sync_info sync=memory_sync_info(), bool glc=false, bool dlc=false, bool nv=false)
   {
      SMEM_instruction *instr = create_instruction<SMEM_instruction>(opcode, (Format)((int)Format::SMEM), 3, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->sync = sync;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Definition def0, memory_sync_info sync=memory_sync_info(), bool glc=false, bool dlc=false, bool nv=false)
   {
      SMEM_instruction *instr = create_instruction<SMEM_instruction>(opcode, (Format)((int)Format::SMEM), 0, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
      instr->sync = sync;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, memory_sync_info sync=memory_sync_info(), bool glc=false, bool dlc=false, bool nv=false)
   {
      SMEM_instruction *instr = create_instruction<SMEM_instruction>(opcode, (Format)((int)Format::SMEM), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->sync = sync;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, Definition def0, Op op0, Op op1, memory_sync_info sync=memory_sync_info(), bool glc=false, bool dlc=false, bool nv=false)
   {
      SMEM_instruction *instr = create_instruction<SMEM_instruction>(opcode, (Format)((int)Format::SMEM), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
      instr->sync = sync;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result smem(aco_opcode opcode, memory_sync_info sync=memory_sync_info(), bool glc=false, bool dlc=false, bool nv=false)
   {
      SMEM_instruction *instr = create_instruction<SMEM_instruction>(opcode, (Format)((int)Format::SMEM), 0, 0);
      instr->sync = sync;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Definition def0, Op op0, int16_t offset0=0, int8_t offset1=0, bool gds=false)
   {
      DS_instruction *instr = create_instruction<DS_instruction>(opcode, (Format)((int)Format::DS), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
      instr->offset0 = offset0;
      instr->offset1 = offset1;
      instr->gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Definition def0, Op op0, Op op1, int16_t offset0=0, int8_t offset1=0, bool gds=false)
   {
      DS_instruction *instr = create_instruction<DS_instruction>(opcode, (Format)((int)Format::DS), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
      instr->offset0 = offset0;
      instr->offset1 = offset1;
      instr->gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Op op0, Op op1, Op op2, int16_t offset0=0, int8_t offset1=0, bool gds=false)
   {
      DS_instruction *instr = create_instruction<DS_instruction>(opcode, (Format)((int)Format::DS), 3, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->offset0 = offset0;
      instr->offset1 = offset1;
      instr->gds = gds;
            
       
      return insert(instr);
   }

        
   Result ds(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, int16_t offset0=0, int8_t offset1=0, bool gds=false)
   {
      DS_instruction *instr = create_instruction<DS_instruction>(opcode, (Format)((int)Format::DS), 4, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
      instr->offset0 = offset0;
      instr->offset1 = offset1;
      instr->gds = gds;
            
       
      return insert(instr);
   }

        
   Result mubuf(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, unsigned offset, bool offen, bool swizzled=false, bool idxen=false, bool addr64=false, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lds=false)
   {
      MUBUF_instruction *instr = create_instruction<MUBUF_instruction>(opcode, (Format)((int)Format::MUBUF), 4, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
      instr->offset = offset;
      instr->offen = offen;
      instr->swizzled = swizzled;
      instr->idxen = idxen;
      instr->addr64 = addr64;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lds = lds;
            
       
      return insert(instr);
   }

        
   Result mubuf(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, unsigned offset, bool offen, bool swizzled=false, bool idxen=false, bool addr64=false, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lds=false)
   {
      MUBUF_instruction *instr = create_instruction<MUBUF_instruction>(opcode, (Format)((int)Format::MUBUF), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->offset = offset;
      instr->offen = offen;
      instr->swizzled = swizzled;
      instr->idxen = idxen;
      instr->addr64 = addr64;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lds = lds;
            
       
      return insert(instr);
   }

        
   Result mtbuf(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, unsigned dfmt, unsigned nfmt, unsigned offset, bool offen, bool idxen=false, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false)
   {
      MTBUF_instruction *instr = create_instruction<MTBUF_instruction>(opcode, (Format)((int)Format::MTBUF), 4, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
      instr->dfmt = dfmt;
      instr->nfmt = nfmt;
      instr->offset = offset;
      instr->offen = offen;
      instr->idxen = idxen;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
            
       
      return insert(instr);
   }

        
   Result mtbuf(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, unsigned dfmt, unsigned nfmt, unsigned offset, bool offen, bool idxen=false, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false)
   {
      MTBUF_instruction *instr = create_instruction<MTBUF_instruction>(opcode, (Format)((int)Format::MTBUF), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->dfmt = dfmt;
      instr->nfmt = nfmt;
      instr->offset = offset;
      instr->offen = offen;
      instr->idxen = idxen;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 3, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 4, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 5, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 6, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 7, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 4, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 5, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 6, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result mimg(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, Op op3, Op op4, Op op5, Op op6, unsigned dmask=0xF, bool da=false, bool unrm=true, bool disable_wqm=false, bool glc=false, bool dlc=false, bool slc=false, bool tfe=false, bool lwe=false, bool r128_a16=false, bool d16=false)
   {
      MIMG_instruction *instr = create_instruction<MIMG_instruction>(opcode, (Format)((int)Format::MIMG), 7, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
            instr->operands[4] = op4.op;
            instr->operands[5] = op5.op;
            instr->operands[6] = op6.op;
      instr->dmask = dmask;
      instr->da = da;
      instr->unrm = unrm;
      instr->disable_wqm = disable_wqm;
      instr->glc = glc;
      instr->dlc = dlc;
      instr->slc = slc;
      instr->tfe = tfe;
      instr->lwe = lwe;
      instr->r128 = r128_a16;
      instr->d16 = d16;
            
       
      return insert(instr);
   }

        
   Result exp(aco_opcode opcode, Op op0, Op op1, Op op2, Op op3, unsigned enabled_mask, unsigned dest, bool compr=false, bool done=false, bool vm=false)
   {
      Export_instruction *instr = create_instruction<Export_instruction>(opcode, (Format)((int)Format::EXP), 4, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            instr->operands[3] = op3.op;
      instr->enabled_mask = enabled_mask;
      instr->dest = dest;
      instr->compressed = compr;
      instr->done = done;
      instr->valid_mask = vm;
            
       
      return insert(instr);
   }

        
   Result branch(aco_opcode opcode, Definition def0, uint32_t target0=0, uint32_t target1=0)
   {
      Pseudo_branch_instruction *instr = create_instruction<Pseudo_branch_instruction>(opcode, (Format)((int)Format::PSEUDO_BRANCH), 0, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
      instr->target[0] = target0;
      instr->target[1] = target1;
            
       
      return insert(instr);
   }

        
   Result branch(aco_opcode opcode, Definition def0, Op op0, uint32_t target0=0, uint32_t target1=0)
   {
      Pseudo_branch_instruction *instr = create_instruction<Pseudo_branch_instruction>(opcode, (Format)((int)Format::PSEUDO_BRANCH), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
      instr->target[0] = target0;
      instr->target[1] = target1;
            
       
      return insert(instr);
   }

        
   Result barrier(aco_opcode opcode, memory_sync_info sync, sync_scope exec_scope=scope_invocation)
   {
      Pseudo_barrier_instruction *instr = create_instruction<Pseudo_barrier_instruction>(opcode, (Format)((int)Format::PSEUDO_BARRIER), 0, 0);
      instr->sync = sync;
      instr->exec_scope = exec_scope;
            
       
      return insert(instr);
   }

        
   Result reduction(aco_opcode opcode, Definition def0, Definition def1, Definition def2, Op op0, Op op1, ReduceOp op, unsigned cluster_size=0)
   {
      Pseudo_reduction_instruction *instr = create_instruction<Pseudo_reduction_instruction>(opcode, (Format)((int)Format::PSEUDO_REDUCTION), 2, 3);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->definitions[2] = def2;
            instr->definitions[2].setPrecise(is_precise);
            instr->definitions[2].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
      instr->reduce_op = op;
      instr->cluster_size = cluster_size;
            
       
      return insert(instr);
   }

        
   Result vop1(aco_opcode opcode)
   {
      VOP1_instruction *instr = create_instruction<VOP1_instruction>(opcode, (Format)((int)Format::VOP1), 0, 0);
            
       
      return insert(instr);
   }

        
   Result vop1(aco_opcode opcode, Definition def0, Op op0)
   {
      VOP1_instruction *instr = create_instruction<VOP1_instruction>(opcode, (Format)((int)Format::VOP1), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   Result vop1(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      VOP1_instruction *instr = create_instruction<VOP1_instruction>(opcode, (Format)((int)Format::VOP1), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result vop1_sdwa(aco_opcode opcode, Definition def0, Op op0)
   {
      SDWA_instruction *instr = create_instruction<SDWA_instruction>(opcode, (Format)((int)Format::VOP1|(int)Format::SDWA), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
            instr->sel[0] = op0.op.bytes() == 2 ? sdwa_uword : (op0.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_sel = def0.bytes() == 2 ? sdwa_uword : (def0.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_preserve = true;
       
      return insert(instr);
   }

        
   Result vop2(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      VOP2_instruction *instr = create_instruction<VOP2_instruction>(opcode, (Format)((int)Format::VOP2), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result vop2(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      VOP2_instruction *instr = create_instruction<VOP2_instruction>(opcode, (Format)((int)Format::VOP2), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   Result vop2(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      VOP2_instruction *instr = create_instruction<VOP2_instruction>(opcode, (Format)((int)Format::VOP2), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result vop2(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      VOP2_instruction *instr = create_instruction<VOP2_instruction>(opcode, (Format)((int)Format::VOP2), 3, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   Result vop2_sdwa(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      SDWA_instruction *instr = create_instruction<SDWA_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::SDWA), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
            instr->sel[0] = op0.op.bytes() == 2 ? sdwa_uword : (op0.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->sel[1] = op1.op.bytes() == 2 ? sdwa_uword : (op1.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_sel = def0.bytes() == 2 ? sdwa_uword : (def0.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_preserve = true;
       
      return insert(instr);
   }

        
   Result vop2_sdwa(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      SDWA_instruction *instr = create_instruction<SDWA_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::SDWA), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
            instr->sel[0] = op0.op.bytes() == 2 ? sdwa_uword : (op0.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->sel[1] = op1.op.bytes() == 2 ? sdwa_uword : (op1.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_sel = def0.bytes() == 2 ? sdwa_uword : (def0.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_preserve = true;
       
      return insert(instr);
   }

        
   Result vop2_sdwa(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      SDWA_instruction *instr = create_instruction<SDWA_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::SDWA), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
            instr->sel[0] = op0.op.bytes() == 2 ? sdwa_uword : (op0.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->sel[1] = op1.op.bytes() == 2 ? sdwa_uword : (op1.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_sel = def0.bytes() == 2 ? sdwa_uword : (def0.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_preserve = true;
       
      return insert(instr);
   }

        
   Result vop2_sdwa(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      SDWA_instruction *instr = create_instruction<SDWA_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::SDWA), 3, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
            instr->sel[0] = op0.op.bytes() == 2 ? sdwa_uword : (op0.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->sel[1] = op1.op.bytes() == 2 ? sdwa_uword : (op1.op.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_sel = def0.bytes() == 2 ? sdwa_uword : (def0.bytes() == 1 ? sdwa_ubyte : sdwa_udword);
instr->dst_preserve = true;
       
      return insert(instr);
   }

        
   Result vopc(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      VOPC_instruction *instr = create_instruction<VOPC_instruction>(opcode, (Format)((int)Format::VOPC), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result vopc(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      VOPC_instruction *instr = create_instruction<VOPC_instruction>(opcode, (Format)((int)Format::VOPC), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result vop3(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP3), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
       
      return insert(instr);
   }

        
   Result vop3(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP3), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result vop3(aco_opcode opcode, Definition def0, Op op0)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP3), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
       
      return insert(instr);
   }

        
   Result vop3(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP3), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
       
      return insert(instr);
   }

        
   Result vop3p(aco_opcode opcode, Definition def0, Op op0, Op op1, uint8_t opsel_lo, uint8_t opsel_hi)
   {
      VOP3P_instruction *instr = create_instruction<VOP3P_instruction>(opcode, (Format)((int)Format::VOP3P), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
      instr->opsel_lo = opsel_lo;
      instr->opsel_hi = opsel_hi;
            
       
      return insert(instr);
   }

        
   Result vop3p(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint8_t opsel_lo, uint8_t opsel_hi)
   {
      VOP3P_instruction *instr = create_instruction<VOP3P_instruction>(opcode, (Format)((int)Format::VOP3P), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->opsel_lo = opsel_lo;
      instr->opsel_hi = opsel_hi;
            
       
      return insert(instr);
   }

        
   Result vintrp(aco_opcode opcode, Definition def0, Op op0, Op op1, unsigned attribute, unsigned component)
   {
      Interp_instruction *instr = create_instruction<Interp_instruction>(opcode, (Format)((int)Format::VINTRP), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
      instr->attribute = attribute;
      instr->component = component;
            
       
      return insert(instr);
   }

        
   Result vintrp(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, unsigned attribute, unsigned component)
   {
      Interp_instruction *instr = create_instruction<Interp_instruction>(opcode, (Format)((int)Format::VINTRP), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->attribute = attribute;
      instr->component = component;
            
       
      return insert(instr);
   }

        
   Result vop1_dpp(aco_opcode opcode, Definition def0, Op op0, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true)
   {
      DPP_instruction *instr = create_instruction<DPP_instruction>(opcode, (Format)((int)Format::VOP1|(int)Format::DPP), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
      instr->dpp_ctrl = dpp_ctrl;
      instr->row_mask = row_mask;
      instr->bank_mask = bank_mask;
      instr->bound_ctrl = bound_ctrl;
            
       
      return insert(instr);
   }

        
   Result vop2_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true)
   {
      DPP_instruction *instr = create_instruction<DPP_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::DPP), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
      instr->dpp_ctrl = dpp_ctrl;
      instr->row_mask = row_mask;
      instr->bank_mask = bank_mask;
      instr->bound_ctrl = bound_ctrl;
            
       
      return insert(instr);
   }

        
   Result vop2_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true)
   {
      DPP_instruction *instr = create_instruction<DPP_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::DPP), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
      instr->dpp_ctrl = dpp_ctrl;
      instr->row_mask = row_mask;
      instr->bank_mask = bank_mask;
      instr->bound_ctrl = bound_ctrl;
            
       
      return insert(instr);
   }

        
   Result vop2_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true)
   {
      DPP_instruction *instr = create_instruction<DPP_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::DPP), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
      instr->dpp_ctrl = dpp_ctrl;
      instr->row_mask = row_mask;
      instr->bank_mask = bank_mask;
      instr->bound_ctrl = bound_ctrl;
            
       
      return insert(instr);
   }

        
   Result vop2_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true)
   {
      DPP_instruction *instr = create_instruction<DPP_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::DPP), 3, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
      instr->dpp_ctrl = dpp_ctrl;
      instr->row_mask = row_mask;
      instr->bank_mask = bank_mask;
      instr->bound_ctrl = bound_ctrl;
            
       
      return insert(instr);
   }

        
   Result vopc_dpp(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true)
   {
      DPP_instruction *instr = create_instruction<DPP_instruction>(opcode, (Format)((int)Format::VOPC|(int)Format::DPP), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
      instr->dpp_ctrl = dpp_ctrl;
      instr->row_mask = row_mask;
      instr->bank_mask = bank_mask;
      instr->bound_ctrl = bound_ctrl;
            
       
      return insert(instr);
   }

        
   Result vopc_dpp(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, uint16_t dpp_ctrl, uint8_t row_mask=0xF, uint8_t bank_mask=0xF, bool bound_ctrl=true)
   {
      DPP_instruction *instr = create_instruction<DPP_instruction>(opcode, (Format)((int)Format::VOPC|(int)Format::DPP), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
      instr->dpp_ctrl = dpp_ctrl;
      instr->row_mask = row_mask;
      instr->bank_mask = bank_mask;
      instr->bound_ctrl = bound_ctrl;
            
       
      return insert(instr);
   }

        
   Result vop1_e64(aco_opcode opcode, Definition def0, Op op0)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP1|(int)Format::VOP3), 1, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            
            
       
      return insert(instr);
   }

        
   Result vop2_e64(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
            
       
      return insert(instr);
   }

        
   Result vop2_e64(aco_opcode opcode, Definition def0, Op op0, Op op1, Op op2)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3), 3, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
            
       
      return insert(instr);
   }

        
   Result vop2_e64(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
            
       
      return insert(instr);
   }

        
   Result vop2_e64(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1, Op op2)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOP2|(int)Format::VOP3), 3, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
            
            
       
      return insert(instr);
   }

        
   Result vopc_e64(aco_opcode opcode, Definition def0, Op op0, Op op1)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOPC|(int)Format::VOP3), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
            
       
      return insert(instr);
   }

        
   Result vopc_e64(aco_opcode opcode, Definition def0, Definition def1, Op op0, Op op1)
   {
      VOP3_instruction *instr = create_instruction<VOP3_instruction>(opcode, (Format)((int)Format::VOPC|(int)Format::VOP3), 2, 2);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->definitions[1] = def1;
            instr->definitions[1].setPrecise(is_precise);
            instr->definitions[1].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            
            
       
      return insert(instr);
   }

        
   Result flat(aco_opcode opcode, Op op0, Op op1, Op op2, uint16_t offset=0, memory_sync_info sync=memory_sync_info(), bool glc=false, bool slc=false, bool lds=false, bool nv=false)
   {
      FLAT_instruction *instr = create_instruction<FLAT_instruction>(opcode, (Format)((int)Format::FLAT), 3, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->offset = offset;
      instr->sync = sync;
      instr->glc = glc;
      instr->slc = slc;
      instr->lds = lds;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result flat(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t offset=0, memory_sync_info sync=memory_sync_info(), bool glc=false, bool slc=false, bool lds=false, bool nv=false)
   {
      FLAT_instruction *instr = create_instruction<FLAT_instruction>(opcode, (Format)((int)Format::FLAT), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
      instr->offset = offset;
      instr->sync = sync;
      instr->glc = glc;
      instr->slc = slc;
      instr->lds = lds;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result global(aco_opcode opcode, Op op0, Op op1, Op op2, uint16_t offset=0, memory_sync_info sync=memory_sync_info(), bool glc=false, bool slc=false, bool lds=false, bool nv=false)
   {
      FLAT_instruction *instr = create_instruction<FLAT_instruction>(opcode, (Format)((int)Format::GLOBAL), 3, 0);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
            instr->operands[2] = op2.op;
      instr->offset = offset;
      instr->sync = sync;
      instr->glc = glc;
      instr->slc = slc;
      instr->lds = lds;
      instr->nv = nv;
            
       
      return insert(instr);
   }

        
   Result global(aco_opcode opcode, Definition def0, Op op0, Op op1, uint16_t offset=0, memory_sync_info sync=memory_sync_info(), bool glc=false, bool slc=false, bool lds=false, bool nv=false)
   {
      FLAT_instruction *instr = create_instruction<FLAT_instruction>(opcode, (Format)((int)Format::GLOBAL), 2, 1);
            instr->definitions[0] = def0;
            instr->definitions[0].setPrecise(is_precise);
            instr->definitions[0].setNUW(is_nuw);
            instr->operands[0] = op0.op;
            instr->operands[1] = op1.op;
      instr->offset = offset;
      instr->sync = sync;
      instr->glc = glc;
      instr->slc = slc;
      instr->lds = lds;
      instr->nv = nv;
            
       
      return insert(instr);
   }

};

}
#endif /* _ACO_BUILDER_ */
