/*
Copyright (C) 2017 Belledonne Communications SARL

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef _ENUMS_HH
#define _ENUMS_HH

/**
 * @brief Namespace that holds all liblinphone's API classes and enumerations.
 */
namespace linphone {
	
	/**
	 * Enum describing RTP AVPF activation modes. 
	 * 
	 */
	enum class AVPFMode {
		/**
		 * Use default value defined at upper level. 
		 */
		Default = -1,
		/**
		 * AVPF is disabled. 
		 */
		Disabled = 0,
		/**
		 * AVPF is enabled. 
		 */
		Enabled = 1
	};

	/**
	 * Enum describing algorithm checking, used by the @ref AccountCreator. 
	 * 
	 */
	enum class AccountCreatorAlgoStatus {
		/**
		 * Algorithm ok. 
		 */
		Ok = 0,
		/**
		 * Algorithm not supported. 
		 */
		NotSupported = 1
	};

	/**
	 * Enum describing backend used in the @ref AccountCreator. 
	 * 
	 */
	enum class AccountCreatorBackend {
		/**
		 * XMLRPC Backend. 
		 */
		XMLRPC = 0,
		/**
		 * FlexiAPI Backend. 
		 */
		FlexiAPI = 1
	};

	/**
	 * Enum describing Ip family. 
	 * 
	 */
	enum class AddressFamily {
		/**
		 * IpV4. 
		 */
		Inet = 0,
		/**
		 * IpV6. 
		 */
		Inet6 = 1,
		/**
		 * Unknown. 
		 */
		Unspec = 2
	};

	/**
	 * Enum describing the authentication methods. 
	 * 
	 */
	enum class AuthMethod {
		/**
		 * Digest authentication requested. 
		 */
		HttpDigest = 0,
		/**
		 * Client certificate requested. 
		 */
		Tls = 1
	};

	/**
	 * @ref ChatRoomBackend is used to indicate the backend implementation of a chat
	 * room. 
	 * 
	 */
	enum class ChatRoomBackend {
		/**
		 * Basic (client-to-client) chat room. 
		 */
		Basic = 1<<0,
		/**
		 * Server-based chat room. 
		 */
		FlexisipChat = 1<<1
	};

	/**
	 * @ref ChatRoomCapabilities is used to indicate the capabilities of a chat room. 
	 * 
	 */
	enum class ChatRoomCapabilities {
		/**
		 * No capabilities. 
		 */
		None = 0,
		/**
		 * No server. 
		 */
		Basic = 1<<0,
		/**
		 * Supports RTT. 
		 */
		RealTimeText = 1<<1,
		/**
		 * Use server (supports group chat) 
		 */
		Conference = 1<<2,
		/**
		 * Special proxy chat room flag. 
		 */
		Proxy = 1<<3,
		/**
		 * Chat room migratable from Basic to Conference. 
		 */
		Migratable = 1<<4,
		/**
		 * A communication between two participants (can be Basic or Conference) 
		 */
		OneToOne = 1<<5,
		/**
		 * Chat room is encrypted. 
		 */
		Encrypted = 1<<6,
		/**
		 * Chat room can enable ephemeral messages. 
		 */
		Ephemeral = 1<<7
	};

	/**
	 * @ref ChatRoomEncryptionBackend is used to indicate the encryption engine used
	 * by a chat room. 
	 * 
	 */
	enum class ChatRoomEncryptionBackend {
		/**
		 * No encryption. 
		 */
		None = 0,
		/**
		 * Lime x3dh encryption. 
		 */
		Lime = 1<<0
	};

	/**
	 * @ref ChatRoomEphemeralMode is used to the ephemeral message mode used by a chat
	 * room. 
	 * 
	 */
	enum class ChatRoomEphemeralMode {
		/**
		 * Each device manages its own ephemeral settings. 
		 */
		DeviceManaged = 0,
		/**
		 * Ephemeral settings are chatroom wide and only admins can change them. 
		 */
		AdminManaged = 1
	};

	/**
	 * TODO move to encryption engine object when available @ref ChatRoomSecurityLevel
	 * is used to indicate the encryption security level of a chat room. 
	 * 
	 */
	enum class ChatRoomSecurityLevel {
		/**
		 * Security failure. 
		 */
		Unsafe = 0,
		/**
		 * No encryption. 
		 */
		ClearText = 1,
		/**
		 * Encrypted. 
		 */
		Encrypted = 2,
		/**
		 * Encrypted and verified. 
		 */
		Safe = 3
	};

	/**
	 * Describes conference invitations possible errors. 
	 * 
	 */
	enum class ConferenceInfoError {
		/**
		 * Could not create or retrieve chat room. 
		 */
		ChatRoomError = 0
	};

	/**
	 * Describes the state of the remote configuring process of the @ref Core object,
	 * 'Skipped' when the feature is disabled. 
	 * 
	 */
	enum class ConfiguringState {
		/**
		 */
		Successful = 0,
		/**
		 */
		Failed = 1,
		/**
		 */
		Skipped = 2
	};

	/**
	 * Consolidated presence information: 'online' means the user is open for
	 * communication, 'busy' means the user is open for communication but involved in
	 * an other activity, 'do not disturb' means the user is not open for
	 * communication, and 'offline' means that no presence information is available. 
	 * 
	 */
	enum class ConsolidatedPresence {
		/**
		 */
		Online = 0,
		/**
		 */
		Busy = 1,
		/**
		 */
		DoNotDisturb = 2,
		/**
		 */
		Offline = 3
	};

	/**
	 * Enum describing the result of the echo canceller calibration process. 
	 * 
	 */
	enum class EcCalibratorStatus {
		/**
		 * The echo canceller calibration process is on going. 
		 */
		InProgress = 0,
		/**
		 * The echo canceller calibration has been performed and produced an echo delay
		 * measure. 
		 */
		Done = 1,
		/**
		 * The echo canceller calibration process has failed. 
		 */
		Failed = 2,
		/**
		 * The echo canceller calibration has been performed and no echo has been
		 * detected. 
		 */
		DoneNoEcho = 3
	};

	/**
	 * Enum describing the capabilities of a @ref Friend, populated through presence
	 * subscribe/notify process. 
	 * 
	 */
	enum class FriendCapability {
		/**
		 * No capabilities populated. 
		 */
		None = 0,
		/**
		 * This friend can be invited in a Flexisip backend @ref ChatRoom. 
		 */
		GroupChat = 1<<0,
		/**
		 * This friend can be invited in a Flexisip backend end-to-end encrypted @ref
		 * ChatRoom. 
		 */
		LimeX3Dh = 1<<1,
		/**
		 * This friend is able to delete ephemeral messages once they have expired. 
		 */
		EphemeralMessages = 1<<2
	};

	/**
	 * The types of FriendList. 
	 * 
	 */
	enum class FriendListType {
		/**
		 */
		CardDAV = 0,
		/**
		 */
		VCard4 = 1
	};

	/**
	 * Describes the global state of the @ref Core object. 
	 * 
	 */
	enum class GlobalState {
		/**
		 * State in which we're in after Core::stop(). 
		 */
		Off = 0,
		/**
		 * Transient state for when we call Core::start() 
		 */
		Startup = 1,
		/**
		 * Indicates @ref Core has been started and is up and running. 
		 */
		On = 2,
		/**
		 * Transient state for when we call Core::stop() 
		 */
		Shutdown = 3,
		/**
		 * Transient state between Startup and On if there is a remote provisionning URI
		 * configured. 
		 */
		Configuring = 4,
		/**
		 * @ref Core state after being created by linphone_factory_create_core, generally
		 * followed by a call to Core::start() 
		 */
		Ready = 5
	};

	/**
	 * Enum describing ICE states. 
	 * 
	 */
	enum class IceState {
		/**
		 * ICE has not been activated for this call or stream. 
		 */
		NotActivated = 0,
		/**
		 * ICE processing has failed. 
		 */
		Failed = 1,
		/**
		 * ICE process is in progress. 
		 */
		InProgress = 2,
		/**
		 * ICE has established a direct connection to the remote host. 
		 */
		HostConnection = 3,
		/**
		 * ICE has established a connection to the remote host through one or several
		 * NATs. 
		 */
		ReflexiveConnection = 4,
		/**
		 * ICE has established a connection through a relay. 
		 */
		RelayConnection = 5
	};

	/**
	 * Enum describing how the authentification will be made. 
	 * 
	 */
	enum class LdapAuthMethod {
		/**
		 * Connection without passwords. 
		 */
		Anonymous = 0,
		/**
		 * Connection with username/password. 
		 */
		Simple = 1
	};

	/**
	 * Enum describing server certificates verification modes. 
	 * 
	 */
	enum class LdapCertVerificationMode {
		/**
		 * Use default value defined on core. 
		 */
		Default = -1,
		/**
		 * Verification is disabled. 
		 */
		Disabled = 0,
		/**
		 * Verification is enabled. 
		 */
		Enabled = 1
	};

	/**
	 * Enum describing errors in LDAP parameters. 
	 * 
	 */
	enum class LdapCheck {
		/**
		 * No error. 
		 */
		Ok = 0,
		/**
		 * Server field is empty. 
		 */
		ServerEmpty = 1,
		/**
		 * The server is not an url. 
		 */
		ServerNotUrl = 2,
		/**
		 * The server doesn't contain a scheme. 
		 */
		ServerNoScheme = 4,
		/**
		 * The server is not a LDAP scheme. 
		 */
		ServerNotLdap = 8,
		/**
		 * LDAP over SSL is non-standardized and deprecated: ldaps has been specified. 
		 */
		ServerLdaps = 16,
		/**
		 * Base Object has been specified. 
		 */
		BaseObjectEmpty = 32,
		/**
		 * Some required fields are missing. 
		 */
		MissingFields = 64
	};

	/**
	 * Enum Debug verbosity for OpenLdap. 
	 * 
	 */
	enum class LdapDebugLevel {
		/**
		 * Set OpenLdap verbosity to none. 
		 */
		Off = 0,
		/**
		 * Set OpenLdap verbosity to debug level. 
		 */
		Verbose = 1
	};

	/**
	 * 
	 */
	enum class LimeState {
		/**
		 * Lime is not used at all. 
		 */
		Disabled = 0,
		/**
		 * Lime is always used. 
		 */
		Mandatory = 1,
		/**
		 * Lime is used only if we already shared a secret with remote. 
		 */
		Preferred = 2
	};

	/**
	 * Whether or not to keep a file with the logs. 
	 * 
	 */
	enum class LogCollectionState {
		/**
		 */
		Disabled = 0,
		/**
		 */
		Enabled = 1,
		/**
		 */
		EnabledWithoutPreviousLogHandler = 2
	};

	/**
	 * Verbosity levels of log messages. 
	 * 
	 */
	enum class LogLevel {
		/**
		 * Level for debug messages. 
		 */
		Debug = 1<<0,
		/**
		 * Level for traces. 
		 */
		Trace = 1<<1,
		/**
		 * Level for information messages. 
		 */
		Message = 1<<2,
		/**
		 * Level for warning messages. 
		 */
		Warning = 1<<3,
		/**
		 * Level for error messages. 
		 */
		Error = 1<<4,
		/**
		 * Level for fatal error messages. 
		 */
		Fatal = 1<<5
	};

	/**
	 * Enum describing how to merge @ref SearchResult from @ref MagicSearch. 
	 * 
	 */
	enum class MagicSearchAggregation {
		/**
		 * No aggregation is done, you can have multiple SearchResult with the same
		 * Friend. 
		 */
		None = 0,
		/**
		 * Aggregation is done by friend, you will have at most a SearchResult per Friend. 
		 */
		Friend = 1
	};

	/**
	 * Enum describing the search categories for Magic Search. 
	 * 
	 */
	enum class MagicSearchSource {
		/**
		 * no Source specified. 
		 */
		None = 0,
		/**
		 * Search in friends only. 
		 */
		Friends = 1<<0,
		/**
		 * Search in Call Logs. 
		 */
		CallLogs = 1<<1,
		/**
		 * Search in LDAP servers. 
		 */
		LdapServers = 1<<2,
		/**
		 * Search in Chat rooms participants. 
		 */
		ChatRooms = 1<<3,
		/**
		 * Search from request : it is usually an address built from the request. 
		 */
		Request = 1<<4,
		/**
		 * Search in all sources. 
		 */
		All = -1
	};

	/**
	 * Indicates for a given media the stream direction. 
	 * 
	 */
	enum class MediaDirection {
		/**
		 */
		Invalid = -1,
		/**
		 */
		Inactive = 0,
		/**
		 * No active media not supported yet. 
		 */
		SendOnly = 1,
		/**
		 * Send only mode. 
		 */
		RecvOnly = 2,
		/**
		 * recv only mode 
		 */
		SendRecv = 3
	};

	/**
	 * Enum describing type of media encryption types. 
	 * 
	 */
	enum class MediaEncryption {
		/**
		 * No media encryption is used. 
		 */
		None = 0,
		/**
		 * Use SRTP media encryption. 
		 */
		SRTP = 1,
		/**
		 * Use ZRTP media encryption. 
		 */
		ZRTP = 2,
		/**
		 * Use DTLS media encryption. 
		 */
		DTLS = 3
	};

	/**
	 * Media resource usage. 
	 * 
	 */
	enum class MediaResourceMode {
		/**
		 * Media resources are not shared. 
		 */
		ExclusiveMediaResources = 0,
		/**
		 * Media resources are shared. 
		 */
		SharedMediaResources = 1
	};

	/**
	 * Basic status as defined in section 4.1.4 of RFC 3863. 
	 * 
	 */
	enum class PresenceBasicStatus {
		/**
		 * This value means that the associated contact element, if any, is ready to
		 * accept communication. 
		 */
		Open = 0,
		/**
		 * This value means that the associated contact element, if any, is unable to
		 * accept communication. 
		 */
		Closed = 1
	};

	/**
	 * Defines privacy policy to apply as described by rfc3323. 
	 * 
	 */
	enum class Privacy {
		/**
		 * Privacy services must not perform any privacy function. 
		 */
		None = 0,
		/**
		 * Request that privacy services provide a user-level privacy function. 
		 */
		User = 1,
		/**
		 * Request that privacy services modify headers that cannot be set arbitrarily by
		 * the user (Contact/Via). 
		 */
		Header = 2,
		/**
		 * Request that privacy services provide privacy for session media. 
		 */
		Session = 4,
		/**
		 * rfc3325 The presence of this privacy type in a Privacy header field indicates
		 * that the user would like the Network Asserted Identity to be kept private with
		 * respect to SIP entities outside the Trust Domain with which the user
		 * authenticated. 
		 */
		Id = 8,
		/**
		 * Privacy service must perform the specified services or fail the request. 
		 */
		Critical = 16,
		/**
		 * Special keyword to use privacy as defined either globally or by proxy using
		 * ProxyConfig::setPrivacy() 
		 */
		Default = 32768
	};

	/**
	 * Enum for publish states. 
	 * 
	 */
	enum class PublishState {
		/**
		 * Initial state, do not use. 
		 */
		None = 0,
		/**
		 * An outgoing publish was created and submitted. 
		 */
		Progress = 1,
		/**
		 * Publish is accepted. 
		 */
		Ok = 2,
		/**
		 * Publish encoutered an error, Event::getReason() gives reason code. 
		 */
		Error = 3,
		/**
		 * Publish is about to expire, only sent if [sip]->refresh_generic_publish
		 * property is set to 0. 
		 */
		Expiring = 4,
		/**
		 * Event has been un published. 
		 */
		Cleared = 5
	};

	/**
	 * Enum describing various failure reasons or contextual information for some
	 * events. 
	 * 
	 */
	enum class Reason {
		/**
		 * No reason has been set by the core. 
		 */
		None = 0,
		/**
		 * No response received from remote. 
		 */
		NoResponse = 1,
		/**
		 * Authentication failed due to bad credentials or resource forbidden. 
		 */
		Forbidden = 2,
		/**
		 * The call has been declined. 
		 */
		Declined = 3,
		/**
		 * Destination of the call was not found. 
		 */
		NotFound = 4,
		/**
		 * The call was not answered in time (request timeout) 
		 */
		NotAnswered = 5,
		/**
		 * Phone line was busy. 
		 */
		Busy = 6,
		/**
		 * Unsupported content. 
		 */
		UnsupportedContent = 7,
		/**
		 * Bad event. 
		 */
		BadEvent = 8,
		/**
		 * Transport error: connection failures, disconnections etc... 
		 */
		IOError = 9,
		/**
		 * Do not disturb reason. 
		 */
		DoNotDisturb = 10,
		/**
		 * Operation is unauthorized because missing credential. 
		 */
		Unauthorized = 11,
		/**
		 * Operation is rejected due to incompatible or unsupported media parameters. 
		 */
		NotAcceptable = 12,
		/**
		 * Operation could not be executed by server or remote client because it didn't
		 * have any context for it. 
		 */
		NoMatch = 13,
		/**
		 * Resource moved permanently. 
		 */
		MovedPermanently = 14,
		/**
		 * Resource no longer exists. 
		 */
		Gone = 15,
		/**
		 * Temporarily unavailable. 
		 */
		TemporarilyUnavailable = 16,
		/**
		 * Address incomplete. 
		 */
		AddressIncomplete = 17,
		/**
		 * Not implemented. 
		 */
		NotImplemented = 18,
		/**
		 * Bad gateway. 
		 */
		BadGateway = 19,
		/**
		 * The received request contains a Session-Expires header field with a duration
		 * below the minimum timer. 
		 */
		SessionIntervalTooSmall = 20,
		/**
		 * Server timeout. 
		 */
		ServerTimeout = 21,
		/**
		 * Unknown reason. 
		 */
		Unknown = 22,
		/**
		 * The call has been transferred. 
		 */
		Transferred = 23
	};

	/**
	 * Enum representing the file format of a recording. 
	 * 
	 */
	enum class RecorderFileFormat {
		/**
		 */
		Unknown,
		/**
		 */
		Wav,
		/**
		 */
		Mkv
	};

	/**
	 * Enum representing the state of a recording. 
	 * 
	 */
	enum class RecorderState {
		/**
		 * No file is opened for recording. 
		 */
		Closed,
		/**
		 * The recorder is paused. 
		 */
		Paused,
		/**
		 * The recorder is running. 
		 */
		Running
	};

	/**
	 * Describes proxy registration states. 
	 * 
	 */
	enum class RegistrationState {
		/**
		 * Initial state for registrations. 
		 */
		None = 0,
		/**
		 * Registration is in progress. 
		 */
		Progress = 1,
		/**
		 * Registration is successful. 
		 */
		Ok = 2,
		/**
		 * Unregistration succeeded. 
		 */
		Cleared = 3,
		/**
		 * Registration failed. 
		 */
		Failed = 4
	};

	/**
	 * Session Timers refresher. 
	 * 
	 */
	enum class SessionExpiresRefresher {
		/**
		 */
		Unspecified = 0,
		/**
		 */
		UAS = 1,
		/**
		 */
		UAC = 2
	};

	/**
	 * Enum describing the stream types. 
	 * 
	 */
	enum class StreamType {
		/**
		 */
		Audio = 0,
		/**
		 */
		Video = 1,
		/**
		 */
		Text = 2,
		/**
		 */
		Unknown = 3
	};

	/**
	 * Enum controlling behavior for incoming subscription request. 
	 * 
	 */
	enum class SubscribePolicy {
		/**
		 * Does not automatically accept an incoming subscription request. 
		 */
		SPWait = 0,
		/**
		 * Rejects incoming subscription request. 
		 */
		SPDeny = 1,
		/**
		 * Automatically accepts a subscription request. 
		 */
		SPAccept = 2
	};

	/**
	 * Enum for subscription direction (incoming or outgoing). 
	 * 
	 */
	enum class SubscriptionDir {
		/**
		 * Incoming subscription. 
		 */
		Incoming = 0,
		/**
		 * Outgoing subscription. 
		 */
		Outgoing = 1,
		/**
		 * Invalid subscription direction. 
		 */
		InvalidDir = 2
	};

	/**
	 * Enum for subscription states. 
	 * 
	 */
	enum class SubscriptionState {
		/**
		 * Initial state, should not be used. 
		 */
		None = 0,
		/**
		 * An outgoing subcription was sent. 
		 */
		OutgoingProgress = 1,
		/**
		 * An incoming subcription is received. 
		 */
		IncomingReceived = 2,
		/**
		 * Subscription is pending, waiting for user approval. 
		 */
		Pending = 3,
		/**
		 * Subscription is accepted. 
		 */
		Active = 4,
		/**
		 * Subscription is terminated normally. 
		 */
		Terminated = 5,
		/**
		 * Subscription was terminated by an error, indicated by Event::getReason() 
		 */
		Error = 6,
		/**
		 * Subscription is about to expire, only sent if [sip]->refresh_generic_subscribe
		 * property is set to 0. 
		 */
		Expiring = 7
	};

	/**
	 * 
	 */
	enum class SupportLevel {
		/**
		 * No support for the feature. 
		 */
		NoSupport = 0,
		/**
		 * Optional support for the feature. 
		 */
		Optional = 1,
		/**
		 * Mandatory support for the feature. 
		 */
		Mandatory = 2
	};

	/**
	 * Enum listing frequent telephony tones. 
	 * 
	 */
	enum class ToneID {
		/**
		 * Not a tone. 
		 */
		Undefined = 0,
		/**
		 * Busy tone. 
		 */
		Busy = 1,
		/**
		 */
		CallWaiting = 2,
		/**
		 * Call waiting tone. 
		 */
		CallOnHold = 3,
		/**
		 * Call on hold tone. 
		 */
		CallLost = 4,
		/**
		 * Tone played when call is abruptly disconnected (media lost) 
		 */
		CallEnd = 5
	};

	/**
	 * Enum describing transport type for LinphoneAddress. 
	 * 
	 */
	enum class TransportType {
		/**
		 */
		Udp = 0,
		/**
		 */
		Tcp = 1,
		/**
		 */
		Tls = 2,
		/**
		 */
		Dtls = 3
	};

	/**
	 * Enum describing uPnP states. 
	 * 
	 */
	enum class UpnpState {
		/**
		 * uPnP is not activate 
		 */
		Idle = 0,
		/**
		 * uPnP process is in progress 
		 */
		Pending = 1,
		/**
		 * Internal use: Only used by port binding. 
		 */
		Adding = 2,
		/**
		 * Internal use: Only used by port binding. 
		 */
		Removing = 3,
		/**
		 * uPnP is not available 
		 */
		NotAvailable = 4,
		/**
		 * uPnP is enabled 
		 */
		Ok = 5,
		/**
		 * uPnP processing has failed 
		 */
		Ko = 6,
		/**
		 * IGD router is blacklisted. 
		 */
		Blacklisted = 7
	};

	/**
	 * Enum describing the result of a version update check. 
	 * 
	 */
	enum class VersionUpdateCheckResult {
		/**
		 */
		UpToDate = 0,
		/**
		 */
		NewVersionAvailable = 1,
		/**
		 */
		Error = 2
	};

	/**
	 * Enum describing the types of argument for LinphoneXmlRpcRequest. 
	 * 
	 */
	enum class XmlRpcArgType {
		/**
		 */
		None = 0,
		/**
		 */
		Int = 1,
		/**
		 */
		String = 2,
		/**
		 */
		StringStruct = 3
	};

	/**
	 * Enum describing the status of a LinphoneXmlRpcRequest. 
	 * 
	 */
	enum class XmlRpcStatus {
		/**
		 */
		Pending = 0,
		/**
		 */
		Ok = 1,
		/**
		 */
		Failed = 2
	};

	/**
	 * Enum describing the ZRTP SAS validation status of a peer URI. 
	 * 
	 */
	enum class ZrtpPeerStatus {
		/**
		 * Peer URI unkown or never validated/invalidated the SAS. 
		 */
		Unknown = 0,
		/**
		 * Peer URI SAS rejected in database. 
		 */
		Invalid = 1,
		/**
		 * Peer URI SAS validated in database. 
		 */
		Valid = 2
	};

	
	typedef int Status;
};

#endif //_ENUMS_HH
