"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _constants = require("../utils/constants");

var _jsYaml = require("js-yaml");

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class RulesService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "createRule", async (_context, request, response) => {
      try {
        const {
          id,
          title,
          category,
          description,
          detection,
          status,
          author,
          references,
          tags,
          level,
          false_positives
        } = request.body;
        const today = (0, _moment.default)(_moment.default.now()).format('YYYY/MM/DD');
        const jsonPayload = {
          id,
          title,
          description: description || title,
          status,
          author,
          date: today,
          modified: today,
          logsource: {
            category
          },
          level,
          detection: (0, _jsYaml.load)(detection)
        };

        if (tags.length > 0) {
          jsonPayload['tags'] = tags.map(tag => tag.value);
        }

        if (references.length > 0) {
          jsonPayload['references'] = references.map(ref => ref.value);
        }

        if (false_positives.length > 0) {
          jsonPayload['falsepositives'] = false_positives.map(falsePos => falsePos.value);
        }

        console.log(jsonPayload);
        const ruleYamlPayload = (0, _jsYaml.safeDump)(jsonPayload);
        console.log(ruleYamlPayload);
        const params = {
          body: ruleYamlPayload,
          category: encodeURIComponent(category)
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const createRuleResponse = await callWithRequest(_constants.CLIENT_RULE_METHODS.CREATE_RULE, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: createRuleResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - RulesService - createRule:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "getRules", async (_context, request, response) => {
      try {
        const {
          prePackaged
        } = request.query;
        const params = {
          prePackaged,
          body: request.body
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getRuleResponse = await callWithRequest(_constants.CLIENT_RULE_METHODS.GET_RULES, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: getRuleResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - RulesService - getRules:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "deleteRule", async (_context, request, response) => {
      try {
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const deleteRuleResponse = await callWithRequest(_constants.CLIENT_RULE_METHODS.DELETE_RULE, request.params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: deleteRuleResponse
          }
        });
      } catch (error) {
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "updateRule", async (_context, request, response) => {
      try {
        const {
          id,
          title,
          category,
          description,
          detection,
          status,
          author,
          references,
          tags,
          level,
          false_positives
        } = request.body;
        const today = (0, _moment.default)(_moment.default.now()).format('YYYY/MM/DD');
        const {
          ruleId
        } = request.params;
        const jsonPayload = {
          id,
          title,
          description: description || title,
          status,
          author,
          date: today,
          modified: today,
          logsource: {
            category
          },
          level,
          detection: (0, _jsYaml.load)(detection)
        };

        if (tags.length > 0) {
          jsonPayload['tags'] = tags.map(tag => tag.value);
        }

        if (references.length > 0) {
          jsonPayload['references'] = references.map(ref => ref.value);
        }

        if (false_positives.length > 0) {
          jsonPayload['falsepositives'] = false_positives.map(falsePos => falsePos.value);
        }

        console.log(jsonPayload);
        const ruleYamlPayload = (0, _jsYaml.safeDump)(jsonPayload);
        console.log(ruleYamlPayload);
        const params = {
          body: ruleYamlPayload,
          category,
          ruleId
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const createRuleResponse = await callWithRequest(_constants.CLIENT_RULE_METHODS.UPDATE_RULE, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: createRuleResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - RulesService - updateRule:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }
  /**
   * Calls backend POST Rules API.
   */


}

exports.default = RulesService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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