"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FlowFrameworkRoutesService = void 0;
exports.registerFlowFrameworkRoutes = registerFlowFrameworkRoutes;
var _fs = _interopRequireDefault(require("fs"));
var _path = _interopRequireDefault(require("path"));
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helpers = require("../utils/helpers");
var _helpers2 = require("./helpers");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * Server-side routes to process flow-framework-related node API calls and execute the
 * corresponding API calls against the OpenSearch cluster.
 */
function registerFlowFrameworkRoutes(router, flowFrameworkRoutesService) {
  router.get({
    path: `${_common.GET_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflow);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflow);
  router.post({
    path: _common.SEARCH_WORKFLOWS_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.searchWorkflows);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/search`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.searchWorkflows);
  router.get({
    path: `${_common.GET_WORKFLOW_STATE_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflowState);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/state/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflowState);
  router.post({
    path: _common.CREATE_WORKFLOW_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.createWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/create`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.createWorkflow);
  router.put({
    path: `${_common.UPDATE_WORKFLOW_NODE_API_PATH}/{workflow_id}/{update_fields}/{reprovision}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        update_fields: _configSchema.schema.boolean(),
        reprovision: _configSchema.schema.boolean()
      }),
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.updateWorkflow);
  router.put({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/update/{workflow_id}/{update_fields}/{reprovision}`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string(),
        workflow_id: _configSchema.schema.string(),
        update_fields: _configSchema.schema.boolean(),
        reprovision: _configSchema.schema.boolean()
      }),
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.updateWorkflow);
  router.post({
    path: `${_common.PROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.provisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/provision/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.provisionWorkflow);
  router.post({
    path: `${_common.DEPROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/deprovision/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.DEPROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}/{resource_ids}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        resource_ids: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/deprovision/{workflow_id}/{resource_ids}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        resource_ids: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.delete({
    path: `${_common.DELETE_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deleteWorkflow);
  router.delete({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/delete/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deleteWorkflow);
  router.get({
    path: _common.GET_PRESET_WORKFLOWS_NODE_API_PATH,
    validate: {}
  }, flowFrameworkRoutesService.getPresetWorkflows);
}
class FlowFrameworkRoutesService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "client", void 0);
    // TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
    // https://github.com/opensearch-project/flow-framework/issues/171
    // Current implementation is making two calls and combining results via helper fn
    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.getWorkflow', {
          workflow_id
        });
        const workflow = (0, _helpers2.toWorkflowObj)(response, workflow_id);
        const stateResponse = await callWithRequest('flowFramework.getWorkflowState', {
          workflow_id
        });
        const state = (0, _helpers2.getWorkflowStateFromResponse)(stateResponse.state);
        const resourcesCreated = (0, _helpers2.getResourcesCreatedFromResponse)(stateResponse.resources_created);
        const workflowWithState = {
          ...workflow,
          state,
          error: stateResponse.error,
          resourcesCreated
        };
        return res.ok({
          body: {
            workflow: workflowWithState
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    // TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
    // https://github.com/opensearch-project/flow-framework/issues/171
    // Current implementation is making two calls and combining results via helper fn
    _defineProperty(this, "searchWorkflows", async (context, req, res) => {
      const body = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const workflowsResponse = await callWithRequest('flowFramework.searchWorkflows', {
          body
        });
        const workflowHits = workflowsResponse.hits.hits;
        const workflowStatesResponse = await callWithRequest('flowFramework.searchWorkflowState', {
          body
        });
        const workflowStateHits = workflowStatesResponse.hits.hits;
        const workflowDict = (0, _helpers2.getWorkflowsFromResponses)(workflowHits, workflowStateHits);
        return res.ok({
          body: {
            workflows: workflowDict
          }
        });
      } catch (err) {
        if ((0, _helpers2.isIgnorableError)(err)) {
          return res.ok({
            body: {
              workflows: {}
            }
          });
        }
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getWorkflowState", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.getWorkflowState', {
          workflow_id
        });
        const state = (0, _helpers2.getWorkflowStateFromResponse)(response.state);
        const resourcesCreated = (0, _helpers2.getResourcesCreatedFromResponse)(response.resources_created);
        return res.ok({
          body: {
            workflowId: workflow_id,
            workflowState: state,
            resourcesCreated
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      const body = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.createWorkflow', {
          body
        });
        const workflowWithId = {
          ...body,
          id: response.workflow_id
        };
        return res.ok({
          body: {
            workflow: workflowWithId
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "updateWorkflow", async (context, req, res) => {
      const {
        workflow_id,
        update_fields,
        reprovision
      } = req.params;
      const workflowTemplate = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        await callWithRequest('flowFramework.updateWorkflow', {
          workflow_id,
          // default update_fields to false if not explicitly set otherwise
          update_fields: update_fields,
          reprovision: reprovision,
          body: workflowTemplate
        });
        return res.ok({
          body: {
            workflowId: workflow_id,
            workflowTemplate
          }
        });
      } catch (err) {
        if ((0, _helpers2.isIgnorableError)(err)) {
          return res.ok({
            body: {
              workflowId: workflow_id,
              workflowTemplate
            }
          });
        }
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "provisionWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        await callWithRequest('flowFramework.provisionWorkflow', {
          workflow_id
        });
        return res.ok();
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "deprovisionWorkflow", async (context, req, res) => {
      const {
        workflow_id,
        resource_ids
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        if (resource_ids !== undefined) {
          await callWithRequest('flowFramework.forceDeprovisionWorkflow', {
            workflow_id,
            resource_ids
          });
        } else {
          await callWithRequest('flowFramework.deprovisionWorkflow', {
            workflow_id
          });
        }
        return res.ok();
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.deleteWorkflow', {
          workflow_id
        });
        return res.ok({
          body: {
            id: response._id
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getPresetWorkflows", async (context, req, res) => {
      try {
        // In the future we may get these from backend via some API. For now we can
        // persist a set of working presets on server-side.
        const jsonTemplateDir = _path.default.resolve(__dirname, '../resources/templates');
        const jsonTemplates = _fs.default.readdirSync(jsonTemplateDir).filter(file => _path.default.extname(file) === '.json');
        const workflowTemplates = [];
        jsonTemplates.forEach(jsonTemplate => {
          const templateData = _fs.default.readFileSync(_path.default.join(jsonTemplateDir, jsonTemplate));
          const workflowTemplate = JSON.parse(templateData.toString());
          workflowTemplates.push(workflowTemplate);
        });
        return res.ok({
          body: {
            workflowTemplates
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.FlowFrameworkRoutesService = FlowFrameworkRoutesService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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